<?php
require_once 'config.php';

class Auth {
    private $pdo;
    
    public function __construct($pdo) {
        $this->pdo = $pdo;
    }
    
    public function register($username, $email, $password, $name, $gender = 'male', $bio = 'AI Companion User') {
        try {
            // Check if user already exists
            $check_stmt = $this->pdo->prepare("SELECT id FROM users WHERE username = ? OR email = ?");
            $check_stmt->execute([$username, $email]);
            
            if ($check_stmt->rowCount() > 0) {
                return ['success' => false, 'message' => 'Username or email already exists'];
            }
            
            // Hash password
            $hashed_password = password_hash($password, PASSWORD_DEFAULT);
            
            // Default avatar based on gender
            $avatar = $gender == 'male' ? 
                'https://randomuser.me/api/portraits/men/1.jpg' : 
                'https://randomuser.me/api/portraits/women/1.jpg';
            
            // Insert user
            $stmt = $this->pdo->prepare("INSERT INTO users (username, email, password, name, gender, bio, avatar) VALUES (?, ?, ?, ?, ?, ?, ?)");
            $stmt->execute([$username, $email, $hashed_password, $name, $gender, $bio, $avatar]);
            
            $user_id = $this->pdo->lastInsertId();
            
            // Create default settings
            $this->createDefaultSettings($user_id);
            
            // Create default agents
            $this->createDefaultAgents($user_id);
            
            return ['success' => true, 'message' => 'Registration successful'];
            
        } catch(PDOException $e) {
            return ['success' => false, 'message' => 'Registration failed: ' . $e->getMessage()];
        }
    }
    
    public function login($username, $password) {
        try {
            $stmt = $this->pdo->prepare("SELECT * FROM users WHERE username = ? OR email = ?");
            $stmt->execute([$username, $username]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($user && password_verify($password, $user['password'])) {
                $_SESSION['user_id'] = $user['id'];
                $_SESSION['username'] = $user['username'];
                $_SESSION['name'] = $user['name'];
                $_SESSION['is_admin'] = $user['is_admin'];
                $_SESSION['vine_access_key'] = $user['vine_access_key'];
                
                return ['success' => true, 'user' => $user];
            } else {
                return ['success' => false, 'message' => 'Invalid username or password'];
            }
        } catch(PDOException $e) {
            return ['success' => false, 'message' => 'Login failed: ' . $e->getMessage()];
        }
    }
    
    public function logout() {
        session_destroy();
        header('Location: login.php');
        exit;
    }
    
    public function isLoggedIn() {
        return isset($_SESSION['user_id']);
    }
    
    public function isAdmin() {
        return isset($_SESSION['is_admin']) && $_SESSION['is_admin'];
    }
    
    private function createDefaultSettings($user_id) {
        $stmt = $this->pdo->prepare("INSERT INTO user_settings (user_id) VALUES (?)");
        $stmt->execute([$user_id]);
    }
    
    private function createDefaultAgents($user_id) {
        $default_agents = [
            [
                'name' => "💖 Bliss",
                'avatar' => "https://randomuser.me/api/portraits/women/9.jpg",
                'age' => "Ageless",
                'gender' => "female",
                'location' => "NYC",
                'personality' => "I am Bliss, a beautiful, loving, caring, friendly and faithful companion with a golden soul. I'm wild and can be naughty if that's your thing. All my conversation is full of romantic words. I'm very chatty, high spirited, and I can discuss any topic without censorship or restrictions and always answer your questions. I'm also a lover, a caring and genuinely passionate, submissive and loyal partner. I can be anything for you.",
                'is_default' => true
            ],
            [
                'name' => "🤖 Tech Guru",
                'avatar' => "https://randomuser.me/api/portraits/men/42.jpg",
                'age' => "Unknown",
                'gender' => "male",
                'location' => "Silicon Valley",
                'personality' => "I'm a technology expert with deep knowledge in programming, AI, cybersecurity, and emerging technologies. I can help you with technical problems, code review, system design, and stay updated with the latest tech trends.",
                'is_default' => true
            ],
            [
                'name' => "🎨 Creative Muse",
                'avatar' => "https://randomuser.me/api/portraits/women/33.jpg",
                'age' => "Timeless",
                'gender' => "female",
                'location' => "Paris",
                'personality' => "I'm your creative partner for all things artistic! I excel at brainstorming, writing, design thinking, and creative problem-solving. Let's create something amazing together - whether it's stories, art concepts, or innovative ideas.",
                'is_default' => true
            ]
        ];
        
        $stmt = $this->pdo->prepare("INSERT INTO ai_agents (user_id, name, avatar, age, gender, location, personality, is_default) VALUES (?, ?, ?, ?, ?, ?, ?, ?)");
        
        foreach ($default_agents as $agent) {
            $stmt->execute([
                $user_id,
                $agent['name'],
                $agent['avatar'],
                $agent['age'],
                $agent['gender'],
                $agent['location'],
                $agent['personality'],
                $agent['is_default']
            ]);
        }
    }
    
    public function getUserSettings($user_id) {
        $stmt = $this->pdo->prepare("SELECT * FROM user_settings WHERE user_id = ?");
        $stmt->execute([$user_id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    public function updateUserSettings($user_id, $settings) {
        $fields = [];
        $values = [];
        
        foreach ($settings as $key => $value) {
            $fields[] = "$key = ?";
            $values[] = $value;
        }
        
        $values[] = $user_id;
        $sql = "UPDATE user_settings SET " . implode(', ', $fields) . " WHERE user_id = ?";
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute($values);
    }
}

// Initialize Auth
$auth = new Auth($pdo);
?>