<?php
require_once 'auth.php';

if (!$auth->isLoggedIn()) {
    header('Location: login.php');
    exit;
}

// Get user data
$user_stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
$user_stmt->execute([$_SESSION['user_id']]);
$user = $user_stmt->fetch(PDO::FETCH_ASSOC);

// Get user settings
$settings_stmt = $pdo->prepare("SELECT * FROM user_settings WHERE user_id = ?");
$settings_stmt->execute([$_SESSION['user_id']]);
$user_settings = $settings_stmt->fetch(PDO::FETCH_ASSOC);

// Get user's AI agents
$agents_stmt = $pdo->prepare("SELECT * FROM ai_agents WHERE user_id = ? ORDER BY is_default DESC, name ASC");
$agents_stmt->execute([$_SESSION['user_id']]);
$agents = $agents_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get default agent for initial load
$default_agent = $agents[0] ?? null;

// Check if user has API key
$has_api_key = !empty($user['vine_access_key']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>💖 Vine-AI Companion 5.0</title>
    <link rel="stylesheet" href="styles.css">
    <link href="https://fonts.googleapis.com/icon?family=Material+Icons" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Roboto:wght@300;400;500;700&display=swap" rel="stylesheet">
    <!-- Material Design Icons -->
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/@mdi/font@6.9.96/css/materialdesignicons.min.css">
    <style>
        /* Welcome Popup Styles */
        .welcome-popup {
            position: fixed;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            width: 90%;
            max-width: 600px;
            background: rgba(144, 224, 239, 0.95);
            backdrop-filter: blur(10px);
            border-radius: 15px;
            padding: 20px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.2);
            z-index: 1002;
            border: 1px solid rgba(255, 255, 255, 0.3);
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
            max-height: 80vh;
            overflow: hidden;
            display: flex;
            flex-direction: column;
        }

        .welcome-popup.active {
            opacity: 1;
            visibility: visible;
        }

        .welcome-popup h2 {
            color: var(--primary-color);
            margin: 0 0 15px 0;
            font-size: clamp(20px, 5vw, 24px);
            display: flex;
            align-items: center;
            gap: 10px;
            flex-wrap: wrap;
        }

        .welcome-popup h2 .material-icons {
            font-size: clamp(24px, 6vw, 28px);
        }

        .welcome-popup .version {
            background-color: var(--primary-color);
            color: white;
            padding: 3px 8px;
            border-radius: 12px;
            font-size: 12px;
            margin-left: auto;
        }

        .welcome-popup .close-btn {
            position: absolute;
            top: 10px;
            right: 10px;
            background: none;
            border: none;
            color: black;
            cursor: pointer;
            font-size: 20px;
            padding: 5px;
        }

        .welcome-popup .content {
            flex: 1;
            overflow-y: auto;
            padding-right: 5px;
            -webkit-overflow-scrolling: touch;
        }

        .welcome-popup h3 {
            color: purple;
            margin: 15px 0 8px;
            font-size: clamp(16px, 4vw, 18px);
        }

        .welcome-popup ol, .welcome-popup ul {
            padding-left: 20px;
            margin: 0 0 15px 0;
        }

        .welcome-popup li {
            margin-bottom: 8px;
            line-height: 1.5;
            font-size: clamp(14px, 3.5vw, 16px);
        }

        .welcome-popup .highlight {
            background-color: rgba(0, 0, 255, 0.1);
            padding: 2px 5px;
            border-radius: 4px;
            font-weight: 500;
            color: var(--primary-color);
        }

        /* Resources Section */
        .resources-section {
            margin-top: 15px;
            background-color: var(--setting-section-bg);
            padding: 12px;
            border-radius: 8px;
        }

        .resources-section h4 {
            margin-bottom: 12px;
            color: var(--primary-color);
            font-size: 14px;
            text-transform: uppercase;
            display: flex;
            align-items: center;
        }

        .resources-section h4::before {
            content: '';
            display: inline-block;
            width: 8px;
            height: 8px;
            background-color: var(--primary-color);
            border-radius: 50%;
            margin-right: 8px;
        }

        .resource-btn {
            width: 100%;
            padding: 10px;
            margin-bottom: 10px;
            border: none;
            border-radius: 6px;
            background-color: var(--primary-color);
            color: white;
            font-weight: 500;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
            transition: all 0.3s ease;
            font-size: 14px;
        }

        .resource-btn:hover {
            background-color: var(--secondary-color);
            transform: translateY(-2px);
        }

        .resource-btn .material-icons {
            font-size: 18px;
        }

        .overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.5);
            z-index: 1001;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
        }

        .overlay.active {
            opacity: 1;
            visibility: visible;
        }

        /* Message Actions */
        .message-actions {
            display: flex;
            gap: 5px;
            margin-bottom: 5px;
            opacity: 0;
            transition: opacity 0.2s ease;
        }

        .message:hover .message-actions {
            opacity: 1;
        }

        .message-action-btn {
            background: none;
            border: none;
            color: var(--secondary-color);
            cursor: pointer;
            padding: 4px;
            border-radius: 4px;
            width: 28px;
            height: 28px;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.2s ease;
            font-size: 16px;
        }

        .message-action-btn:hover {
            background-color: var(--setting-section-bg);
            color: var(--primary-color);
        }

        /* API Key Status */
        .api-key-active {
            background-color: var(--success-color) !important;
            color: white !important;
        }

        .api-key-inactive {
            background-color: var(--error-color) !important;
            color: white !important;
        }

        /* Responsive adjustments */
        @media (max-width: 480px) {
            .welcome-popup {
                padding: 15px;
                width: 95%;
            }
            
            .welcome-popup h2 {
                font-size: 20px;
            }
            
            .welcome-popup .content {
                padding-right: 0;
            }
            
            .welcome-popup li {
                margin-bottom: 6px;
            }
        }
        
        /* Logout button */
        .logout-btn {
            background-color: transparent;
            border: 1px solid var(--error-color);
            color: var(--error-color);
            padding: 5px 10px;
            border-radius: 15px;
            font-size: 12px;
            cursor: pointer;
            transition: all 0.3s ease;
            margin-left: 10px;
        }

        .logout-btn:hover {
            background-color: var(--error-color);
            color: white;
        }

        /* Close Sidebar Button */
        .close-sidebar-btn {
            position: absolute;
            top: 15px;
            right: 15px;
            background: var(--primary-color);
            color: white;
            border: none;
            border-radius: 50%;
            width: 32px;
            height: 32px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 101;
            transition: all 0.3s ease;
        }

        .close-sidebar-btn:hover {
            background: var(--secondary-color);
            transform: scale(1.1);
        }

        @media (min-width: 769px) {
            .close-sidebar-btn {
                display: none;
            }
        }
    </style>
</head>
<body>
    <!-- Welcome Popup -->
    <div class="welcome-popup" id="welcomePopup">
        <button class="close-btn" id="closeWelcomePopup">
            <span class="material-icons">close</span>
        </button>
        <h2>
            <span class="material-icons">rocket_launch</span>
            Welcome to Vine-AI Companion
            <span class="version">v5.0</span>
        </h2>
        <div class="content">
            <h3>How to Use Vine-AI Companion</h3>
            <ol>
                <li><span class="highlight">Set your Vine Access Key</span> in the VINE AI Settings section to unlock all features.</li>
                <li><span class="highlight">Choose or create an AI Agent</span> from the AI Agents section to start chatting.</li>
                <li><span class="highlight">Customize your agent's personality</span> using the Core Personality and Interaction Style settings.</li>
                <li><span class="highlight">Type your message</span> in the chat box and press Send or Enter to get responses.</li>
                <li><span class="highlight">Use the emoji picker</span> to add expressions to your messages.</li>
                <li><span class="highlight">Upload files</span> to get context-aware responses from your agent.</li>
                <li><span class="highlight">Adjust advanced settings</span> to fine-tune your AI companion's behavior.</li>
                <li><span class="highlight">Save conversations</span> using the export button in the chat header.</li>
            </ol>
            
            <h3>Pro Tips</h3>
            <ul>
                <li>Right-click on an agent to edit its details.</li>
                <li>Try different themes to match your mood.</li>
                <li>Increase creativity level for more imaginative responses.</li>
                <li>Use the memory capacity setting to control how much context your agent remembers.</li>
            </ul>
            
            <div class="resources-section">
                <h4>Quick Actions</h4>
                <button class="resource-btn" id="howToUseBtnPopup">
                    <span class="material-icons">help_outline</span>
                    How to Use Guide
                </button>
                <button class="resource-btn" id="buyKeyBtnPopup">
                    <span class="material-icons">vpn_key</span>
                    Get Vine Access Key
                </button>
                <p style="text-align: center; margin: 10px 0; font-size: 14px; color: #666;">or</p>
                <button class="resource-btn" id="adminRequestBtnPopup" style="background-color: #4CAF50;">
                    <span class="material-icons">admin_panel_settings</span>
                    Request Key from Admin
                </button>
            </div>
        </div>
    </div>
    <div class="overlay" id="welcomeOverlay"></div>

    <!-- Main App Structure -->
    <div class="app-container">
        <!-- Sidebar -->
        <div class="sidebar" id="sidebar">
            <!-- Close Sidebar Button (Mobile) -->
            <button class="close-sidebar-btn" id="closeSidebarBtn">
                <span class="material-icons">close</span>
            </button>

            <!-- User Profile -->
            <div class="user-profile">
                <img src="<?php echo htmlspecialchars($user['avatar']); ?>" alt="User Avatar" class="user-avatar" id="userAvatar">
                <div class="user-info">
                    <h3 id="userName"><?php echo htmlspecialchars($user['name']); ?></h3>
                    <p id="userBio"><?php echo htmlspecialchars($user['bio']); ?></p>
                </div>
                <button class="edit-profile-btn" id="editProfileBtn">
                    <span class="material-icons">edit</span>
                </button>
                <button class="logout-btn" id="logoutBtn">
                    <span class="material-icons">logout</span>
                </button>
            </div>

            <!-- AI Agents Section -->
            <div class="sidebar-section">
                <div class="section-header">
                    <h3><span class="material-icons">smart_toy</span>AI Agents</h3>
                    <button class="add-btn" id="addAgentBtn">
                        <span class="material-icons">add</span>
                    </button>
                </div>
                <div class="agents-list" id="agentsList">
                    <?php foreach ($agents as $agent): ?>
                        <div class="agent-item <?php echo $agent['id'] == ($default_agent['id'] ?? '') ? 'active' : ''; ?>" data-agent-id="<?php echo $agent['id']; ?>">
                            <img src="<?php echo htmlspecialchars($agent['avatar']); ?>" alt="<?php echo htmlspecialchars($agent['name']); ?>" class="agent-avatar">
                            <span class="agent-name"><?php echo htmlspecialchars($agent['name']); ?></span>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>

            <!-- VINE AI Settings -->
            <div class="sidebar-section">
                <div class="section-header">
                    <h3><span class="material-icons">vpn_key</span>VINE AI Settings</h3>
                </div>
                <div class="settings-group">
                    <div class="setting-item">
                        <label for="apiKeyInput">Vine Access Key:</label>
                        <div class="input-with-status">
                            <input type="password" id="apiKeyInput" placeholder="Enter Vine Access Key" value="<?php echo $has_api_key ? '••••••••' : ''; ?>">
                            <span class="api-key-status <?php echo $has_api_key ? 'api-key-active' : 'api-key-inactive'; ?>" id="apiKeyStatus">
                                <?php if ($has_api_key): ?>
                                    ✅ Vine Key Active
                                <?php else: ?>
                                    🔒 Vine Key Required
                                <?php endif; ?>
                            </span>
                        </div>
                    </div>
                    <div class="setting-item">
                        <label for="themeSelect">Theme:</label>
                        <select id="themeSelect">
                            <option value="cyberpunk">Cyberpunk</option>
                            <option value="dark">Dark</option>
                            <option value="light">Light</option>
                            <option value="nature">Nature</option>
                            <option value="ocean">Ocean</option>
                            <option value="sunset">Sunset</option>
                            <option value="matrix">Matrix</option>
                            <option value="neon">Neon</option>
                        </select>
                    </div>
                </div>
            </div>

            <!-- Core Personality Settings -->
            <div class="sidebar-section">
                <div class="section-header">
                    <h3><span class="material-icons">psychology</span>Core Personality</h3>
                </div>
                <div class="settings-group">
                    <div class="setting-item">
                        <label for="sentienceLevel">Sentience Level:</label>
                        <input type="range" id="sentienceLevel" min="1" max="5" step="1" value="<?php echo $user_settings['sentience_level'] ?? 3; ?>">
                        <span class="slider-value" id="sentienceLevelValue"><?php echo $user_settings['sentience_level'] ?? 3; ?></span>
                    </div>
                    <div class="setting-item">
                        <label for="emotionalDepth">Emotional Depth:</label>
                        <input type="range" id="emotionalDepth" min="1" max="5" step="1" value="<?php echo $user_settings['emotional_depth'] ?? 3; ?>">
                        <span class="slider-value" id="emotionalDepthValue"><?php echo $user_settings['emotional_depth'] ?? 3; ?></span>
                    </div>
                    <div class="setting-item">
                        <label for="creativityLevel">Creativity Level:</label>
                        <input type="range" id="creativityLevel" min="1" max="5" step="1" value="<?php echo $user_settings['creativity_level'] ?? 3; ?>">
                        <span class="slider-value" id="creativityLevelValue"><?php echo $user_settings['creativity_level'] ?? 3; ?></span>
                    </div>
                    <div class="setting-item">
                        <label for="memoryCapacity">Memory Capacity:</label>
                        <input type="range" id="memoryCapacity" min="1" max="5" step="1" value="<?php echo $user_settings['memory_capacity'] ?? 3; ?>">
                        <span class="slider-value" id="memoryCapacityValue"><?php echo $user_settings['memory_capacity'] ?? 3; ?></span>
                    </div>
                </div>
            </div>

            <!-- Interaction Style -->
            <div class="sidebar-section">
                <div class="section-header">
                    <h3><span class="material-icons">chat</span>Interaction Style</h3>
                </div>
                <div class="settings-group">
                    <div class="setting-item">
                        <label for="responseSpeed">Response Speed:</label>
                        <select id="responseSpeed">
                            <option value="slow" <?php echo ($user_settings['response_speed'] ?? 'normal') == 'slow' ? 'selected' : ''; ?>>Slow & Thoughtful</option>
                            <option value="normal" <?php echo ($user_settings['response_speed'] ?? 'normal') == 'normal' ? 'selected' : ''; ?>>Normal</option>
                            <option value="fast" <?php echo ($user_settings['response_speed'] ?? 'normal') == 'fast' ? 'selected' : ''; ?>>Fast & Energetic</option>
                        </select>
                    </div>
                    <div class="setting-item">
                        <label for="verbosity">Verbosity:</label>
                        <select id="verbosity">
                            <option value="concise" <?php echo ($user_settings['verbosity'] ?? 'balanced') == 'concise' ? 'selected' : ''; ?>>Concise</option>
                            <option value="balanced" <?php echo ($user_settings['verbosity'] ?? 'balanced') == 'balanced' ? 'selected' : ''; ?>>Balanced</option>
                            <option value="detailed" <?php echo ($user_settings['verbosity'] ?? 'balanced') == 'detailed' ? 'selected' : ''; ?>>Detailed</option>
                        </select>
                    </div>
                    <div class="setting-item">
                        <label for="formalityLevel">Formality Level:</label>
                        <input type="range" id="formalityLevel" min="1" max="5" step="1" value="<?php echo $user_settings['formality_level'] ?? 3; ?>">
                        <span class="slider-value" id="formalityLevelValue"><?php echo $user_settings['formality_level'] ?? 3; ?></span>
                    </div>
                </div>
            </div>

            <!-- Advanced Settings -->
            <div class="sidebar-section">
                <div class="section-header">
                    <h3><span class="material-icons">tune</span>Advanced Settings</h3>
                </div>
                <div class="settings-group">
                    <div class="setting-item checkbox">
                        <input type="checkbox" id="internetAccess" <?php echo ($user_settings['internet_access'] ?? true) ? 'checked' : ''; ?>>
                        <label for="internetAccess">Internet Access</label>
                    </div>
                    <div class="setting-item checkbox">
                        <input type="checkbox" id="realTimeData" <?php echo ($user_settings['real_time_data'] ?? true) ? 'checked' : ''; ?>>
                        <label for="realTimeData">Real-time Data</label>
                    </div>
                    <div class="setting-item checkbox">
                        <input type="checkbox" id="multilingual" <?php echo ($user_settings['multilingual'] ?? true) ? 'checked' : ''; ?>>
                        <label for="multilingual">Multilingual Support</label>
                    </div>
                    <div class="setting-item checkbox">
                        <input type="checkbox" id="technicalKnowledge" <?php echo ($user_settings['technical_knowledge'] ?? true) ? 'checked' : ''; ?>>
                        <label for="technicalKnowledge">Technical Knowledge</label>
                    </div>
                    <div class="setting-item checkbox">
                        <input type="checkbox" id="creativeWriting" <?php echo ($user_settings['creative_writing'] ?? true) ? 'checked' : ''; ?>>
                        <label for="creativeWriting">Creative Writing</label>
                    </div>
                    <div class="setting-item checkbox">
                        <input type="checkbox" id="analyticalSkills" <?php echo ($user_settings['analytical_skills'] ?? true) ? 'checked' : ''; ?>>
                        <label for="analyticalSkills">Analytical Skills</label>
                    </div>
                </div>
            </div>

            <!-- Emotional Traits -->
            <div class="sidebar-section">
                <div class="section-header">
                    <h3><span class="material-icons">mood</span>Emotional Traits</h3>
                </div>
                <div class="settings-group">
                    <div class="setting-item">
                        <label for="humorLevel">Humor Level:</label>
                        <input type="range" id="humorLevel" min="1" max="5" step="1" value="<?php echo $user_settings['humor_level'] ?? 2; ?>">
                        <span class="slider-value" id="humorLevelValue"><?php echo $user_settings['humor_level'] ?? 2; ?></span>
                    </div>
                    <div class="setting-item">
                        <label for="empathyLevel">Empathy Level:</label>
                        <input type="range" id="empathyLevel" min="1" max="5" step="1" value="<?php echo $user_settings['empathy_level'] ?? 3; ?>">
                        <span class="slider-value" id="empathyLevelValue"><?php echo $user_settings['empathy_level'] ?? 3; ?></span>
                    </div>
                    <div class="setting-item">
                        <label for="sarcasmLevel">Sarcasm Level:</label>
                        <input type="range" id="sarcasmLevel" min="1" max="5" step="1" value="<?php echo $user_settings['sarcasm_level'] ?? 2; ?>">
                        <span class="slider-value" id="sarcasmLevelValue"><?php echo $user_settings['sarcasm_level'] ?? 2; ?></span>
                    </div>
                    <div class="setting-item">
                        <label for="optimismLevel">Optimism Level:</label>
                        <input type="range" id="optimismLevel" min="1" max="5" step="1" value="<?php echo $user_settings['optimism_level'] ?? 3; ?>">
                        <span class="slider-value" id="optimismLevelValue"><?php echo $user_settings['optimism_level'] ?? 3; ?></span>
                    </div>
                </div>
            </div>

            <!-- Additional Traits -->
            <div class="sidebar-section">
                <div class="section-header">
                    <h3><span class="material-icons">auto_awesome</span>Additional Traits</h3>
                </div>
                <div class="settings-group">
                    <div class="setting-item">
                        <label for="storytelling">Storytelling:</label>
                        <input type="range" id="storytelling" min="1" max="5" step="1" value="<?php echo $user_settings['storytelling'] ?? 3; ?>">
                        <span class="slider-value" id="storytellingValue"><?php echo $user_settings['storytelling'] ?? 3; ?></span>
                    </div>
                    <div class="setting-item">
                        <label for="wisdomLevel">Wisdom Level:</label>
                        <input type="range" id="wisdomLevel" min="1" max="5" step="1" value="<?php echo $user_settings['wisdom_level'] ?? 3; ?>">
                        <span class="slider-value" id="wisdomLevelValue"><?php echo $user_settings['wisdom_level'] ?? 3; ?></span>
                    </div>
                    <div class="setting-item">
                        <label for="intellectualLevel">Intellectual Level:</label>
                        <input type="range" id="intellectualLevel" min="1" max="5" step="1" value="<?php echo $user_settings['intellectual_level'] ?? 3; ?>">
                        <span class="slider-value" id="intellectualLevelValue"><?php echo $user_settings['intellectual_level'] ?? 3; ?></span>
                    </div>
                    <div class="setting-item">
                        <label for="confidenceLevel">Confidence Level:</label>
                        <input type="range" id="confidenceLevel" min="1" max="5" step="1" value="<?php echo $user_settings['confidence_level'] ?? 3; ?>">
                        <span class="slider-value" id="confidenceLevelValue"><?php echo $user_settings['confidence_level'] ?? 3; ?></span>
                    </div>
                </div>
            </div>
        </div>

        <!-- Main Chat Area -->
        <div class="main-content">
            <!-- Chat Header -->
            <div class="chat-header">
                <button class="sidebar-toggle" id="sidebarToggle">
                    <span class="material-icons">menu</span>
                </button>
                <div class="agent-info">
                    <img src="<?php echo htmlspecialchars($default_agent['avatar'] ?? 'https://randomuser.me/api/portraits/women/49.jpg'); ?>" alt="Agent Avatar" class="agent-avatar" id="currentAgentAvatar">
                    <div class="agent-details">
                        <h2 id="currentAgentName"><?php echo htmlspecialchars($default_agent['name'] ?? '💖 Vine-AI Companion'); ?></h2>
                        <span class="agent-status">Online</span>
                    </div>
                </div>
                <div class="chat-actions">
                    <button class="chat-action-btn" id="deleteChatBtn" title="Delete Conversation">
                        <span class="material-icons">delete</span>
                    </button>
                    <button class="chat-action-btn" id="exportChatBtn" title="Export Conversation">
                        <span class="material-icons">download</span>
                    </button>
                </div>
            </div>

            <!-- Chat Messages -->
            <div class="chat-messages" id="chatMessages">
                <!-- Messages will be loaded here -->
                <div class="message agent-message">
                    <div class="message-content">
                        <div class="message-text">
                            Hello! I'm your Vine-AI Companion. I'm here to help you with conversations, answer questions, analyze images, and provide assistance on various topics. How can I help you today?
                        </div>
                        <div class="message-time"><?php echo date('H:i'); ?></div>
                    </div>
                </div>
            </div>

            <!-- Chat Input -->
            <div class="chat-input-container">
                <div class="chat-input-actions">
                    <button class="input-action-btn" id="fileUploadBtn" title="Upload File">
                        <span class="material-icons">attach_file</span>
                    </button>
                    <button class="input-action-btn" id="emojiPickerBtn" title="Emoji Picker">
                        <span class="material-icons">mood</span>
                    </button>
                </div>
                <div style="display: flex; gap: 10px; align-items: flex-end;">
                    <textarea id="messageInput" placeholder="Type your message here... (Shift+Enter for new line)" rows="1"></textarea>
                    <button class="send-btn" id="sendMessageBtn">
                        <span class="material-icons">send</span>
                    </button>
                </div>
                <input type="file" id="fileInput" style="display: none;" accept="image/*,.pdf,.doc,.docx,.txt">
            </div>
        </div>
    </div>

    <!-- Hidden inputs for PHP data -->
    <input type="hidden" id="currentUserId" value="<?php echo $_SESSION['user_id']; ?>">
    <input type="hidden" id="currentAgentId" value="<?php echo $default_agent['id'] ?? ''; ?>">
    <input type="hidden" id="currentConversationId" value="">

    <!-- Modals -->
    <?php include 'modals.php'; ?>

    <script src="script.js"></script>
</body>
</html>