// Vine-AI Companion 5.0 - Main JavaScript
document.addEventListener('DOMContentLoaded', function() {
    // DOM Elements
    const sidebar = document.getElementById('sidebar');
    const sidebarToggle = document.getElementById('sidebarToggle');
    const closeSidebarBtn = document.getElementById('closeSidebarBtn');
    const chatMessages = document.getElementById('chatMessages');
    const messageInput = document.getElementById('messageInput');
    const sendMessageBtn = document.getElementById('sendMessageBtn');
    const fileUploadBtn = document.getElementById('fileUploadBtn');
    const fileInput = document.getElementById('fileInput');
    const agentsList = document.getElementById('agentsList');
    const addAgentBtn = document.getElementById('addAgentBtn');
    const currentAgentName = document.getElementById('currentAgentName');
    const currentAgentAvatar = document.getElementById('currentAgentAvatar');
    const deleteChatBtn = document.getElementById('deleteChatBtn');
    const exportChatBtn = document.getElementById('exportChatBtn');
    const themeSelect = document.getElementById('themeSelect');
    const apiKeyInput = document.getElementById('apiKeyInput');
    const apiKeyStatus = document.getElementById('apiKeyStatus');
    const editProfileBtn = document.getElementById('editProfileBtn');
    const logoutBtn = document.getElementById('logoutBtn');

    // Modal Elements
    const profileModal = document.getElementById('profileModal');
    const agentModal = document.getElementById('agentModal');
    const apiKeyModal = document.getElementById('apiKeyModal');
    const confirmModal = document.getElementById('confirmModal');
    const keyAssignedModal = document.getElementById('keyAssignedModal');

    // Settings Elements
    const settingsElements = {
        // Range sliders
        sentienceLevel: document.getElementById('sentienceLevel'),
        emotionalDepth: document.getElementById('emotionalDepth'),
        creativityLevel: document.getElementById('creativityLevel'),
        memoryCapacity: document.getElementById('memoryCapacity'),
        formalityLevel: document.getElementById('formalityLevel'),
        humorLevel: document.getElementById('humorLevel'),
        empathyLevel: document.getElementById('empathyLevel'),
        sarcasmLevel: document.getElementById('sarcasmLevel'),
        optimismLevel: document.getElementById('optimismLevel'),
        storytelling: document.getElementById('storytelling'),
        wisdomLevel: document.getElementById('wisdomLevel'),
        intellectualLevel: document.getElementById('intellectualLevel'),
        confidenceLevel: document.getElementById('confidenceLevel'),
        
        // Selects
        responseSpeed: document.getElementById('responseSpeed'),
        verbosity: document.getElementById('verbosity'),
        
        // Checkboxes
        internetAccess: document.getElementById('internetAccess'),
        realTimeData: document.getElementById('realTimeData'),
        multilingual: document.getElementById('multilingual'),
        technicalKnowledge: document.getElementById('technicalKnowledge'),
        creativeWriting: document.getElementById('creativeWriting'),
        analyticalSkills: document.getElementById('analyticalSkills')
    };

    // State variables
    let currentAgentId = document.getElementById('currentAgentId').value;
    let currentUserId = document.getElementById('currentUserId').value;
    let currentConversationId = null;
    let isTyping = false;
    let apiKey = '';
    const API_KEY_PREFIX = 'AIzaSy';

    // Initialize the app
    initApp();

    function initApp() {
        // Load user's API key
        loadApiKey();
        
        // Load agents
        loadAgents();
        
        // Load conversation for current agent
        if (currentAgentId) {
            loadConversation(currentAgentId);
        }
        
        // Load theme
        loadTheme();
        
        // Setup event listeners
        setupEventListeners();
        
        // Setup slider value displays
        setupSliders();
        
        // Check for key assignment notification
        checkKeyAssignment();
    }

    function setupEventListeners() {
        // Sidebar toggle
        sidebarToggle.addEventListener('click', function() {
            sidebar.classList.add('active');
        });

        // Close sidebar button
        closeSidebarBtn.addEventListener('click', function() {
            sidebar.classList.remove('active');
        });

        // Message input
        messageInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                sendMessage();
            }
        });

        messageInput.addEventListener('input', function() {
            this.style.height = 'auto';
            this.style.height = (this.scrollHeight) + 'px';
        });

        sendMessageBtn.addEventListener('click', sendMessage);

        // File upload
        fileUploadBtn.addEventListener('click', function() {
            fileInput.click();
        });

        fileInput.addEventListener('change', handleFileUpload);

        // Agent management
        addAgentBtn.addEventListener('click', openAgentModal);

        // API Key handling
        apiKeyInput.addEventListener('input', function() {
            const userInput = this.value.trim();
            if (userInput) {
                const fullKey = API_KEY_PREFIX + userInput;
                validateApiKey(fullKey);
            } else {
                apiKeyStatus.textContent = '🔒 Vine Key Required';
                apiKeyStatus.className = 'api-key-status api-key-inactive';
            }
        });

        // Theme change
        themeSelect.addEventListener('change', changeTheme);

        // Settings changes
        Object.keys(settingsElements).forEach(key => {
            const element = settingsElements[key];
            if (element.type === 'checkbox') {
                element.addEventListener('change', function() {
                    saveSetting(key, this.checked);
                });
            } else if (element.type === 'range') {
                element.addEventListener('input', function() {
                    updateSliderValue(key, this.value);
                    saveSetting(key, this.value);
                });
            } else {
                element.addEventListener('change', function() {
                    saveSetting(key, this.value);
                });
            }
        });

        // Chat actions
        deleteChatBtn.addEventListener('click', confirmDeleteConversation);
        exportChatBtn.addEventListener('click', exportConversation);

        // Profile and logout
        editProfileBtn.addEventListener('click', openProfileModal);
        logoutBtn.addEventListener('click', function() {
            if (confirm('Are you sure you want to logout?')) {
                window.location.href = 'logout.php';
            }
        });

        // Emoji picker
        document.getElementById('emojiPickerBtn').addEventListener('click', showEmojiPicker);

        // Modal events
        setupModalEvents();

        // Welcome popup events
        setupWelcomePopupEvents();
    }

    function setupModalEvents() {
        // Profile Modal
        document.getElementById('closeProfileModal').addEventListener('click', () => closeModal(profileModal));
        document.getElementById('cancelProfileBtn').addEventListener('click', () => closeModal(profileModal));
        document.getElementById('saveProfileBtn').addEventListener('click', saveProfile);

        // Agent Modal
        document.getElementById('closeAgentModal').addEventListener('click', () => closeModal(agentModal));
        document.getElementById('cancelAgentBtn').addEventListener('click', () => closeModal(agentModal));
        document.getElementById('saveAgentBtn').addEventListener('click', saveAgent);
        document.getElementById('deleteAgentBtn').addEventListener('click', confirmDeleteAgent);

        // API Key Modal
        document.getElementById('closeApiKeyModal').addEventListener('click', () => closeModal(apiKeyModal));
        document.getElementById('cancelApiKeyBtn').addEventListener('click', () => closeModal(apiKeyModal));
        document.getElementById('saveApiKeyBtn').addEventListener('click', saveApiKey);

        // Confirm Modal
        document.getElementById('closeConfirmModal').addEventListener('click', () => closeModal(confirmModal));
        document.getElementById('cancelConfirmBtn').addEventListener('click', () => closeModal(confirmModal));
        document.getElementById('confirmActionBtn').addEventListener('click', executePendingAction);

        // Key Assigned Modal
        document.getElementById('closeKeyAssignedModal').addEventListener('click', () => closeModal(keyAssignedModal));
        document.getElementById('closeKeyAssignedBtn').addEventListener('click', () => closeModal(keyAssignedModal));
    }

    function setupWelcomePopupEvents() {
        const welcomePopup = document.getElementById('welcomePopup');
        const welcomeOverlay = document.getElementById('welcomeOverlay');
        const closeWelcomePopup = document.getElementById('closeWelcomePopup');

        // Show welcome popup on first visit
        const hasSeenWelcome = localStorage.getItem('hasSeenWelcome');
        if (!hasSeenWelcome) {
            setTimeout(() => {
                welcomePopup.classList.add('active');
                welcomeOverlay.classList.add('active');
                localStorage.setItem('hasSeenWelcome', 'true');
            }, 1000);
        }

        // Close welcome popup
        closeWelcomePopup.addEventListener('click', closeWelcome);
        welcomeOverlay.addEventListener('click', closeWelcome);

        // Welcome popup action buttons
        document.getElementById('howToUseBtnPopup').addEventListener('click', function() {
            window.open('help.php', '_blank');
            closeWelcome();
        });

        document.getElementById('buyKeyBtnPopup').addEventListener('click', function() {
            window.open('https://aistudio.google.com/', '_blank');
        });

        document.getElementById('adminRequestBtnPopup').addEventListener('click', function() {
            openModal(apiKeyModal);
            closeWelcome();
        });

        function closeWelcome() {
            welcomePopup.classList.remove('active');
            welcomeOverlay.classList.remove('active');
        }
    }

    function setupSliders() {
        Object.keys(settingsElements).forEach(key => {
            const element = settingsElements[key];
            if (element.type === 'range') {
                updateSliderValue(key, element.value);
            }
        });
    }

    function updateSliderValue(settingId, value) {
        const valueElement = document.getElementById(settingId + 'Value');
        if (valueElement) {
            valueElement.textContent = value;
        }
    }

    // API Functions
    async function apiCall(endpoint, data = null, method = 'POST') {
        const options = {
            method: method,
            headers: {
                'Content-Type': 'application/json',
            },
            credentials: 'same-origin'
        };

        if (data && method !== 'GET') {
            options.body = JSON.stringify(data);
        }

        try {
            const response = await fetch(`api/${endpoint}`, options);
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const text = await response.text();
            if (!text) {
                return { success: true }; // Empty response is OK for some endpoints
            }
            
            const result = JSON.parse(text);
            return result;
        } catch (error) {
            console.error('API call failed:', error);
            showToast('Error: ' + error.message);
            throw error;
        }
    }

    // Agent Management
    async function loadAgents() {
        try {
            const result = await apiCall('get_agents.php', {}, 'GET');
            if (!result.agents) {
                console.error('No agents data received');
                return;
            }
            
            agentsList.innerHTML = '';
            
            result.agents.forEach(agent => {
                const agentElement = document.createElement('div');
                agentElement.className = `agent-item ${agent.id == currentAgentId ? 'active' : ''}`;
                agentElement.dataset.agentId = agent.id;
                agentElement.dataset.isDefault = agent.is_default;
                
                agentElement.innerHTML = `
                    <img src="${agent.avatar}" alt="${agent.name}" class="agent-avatar">
                    <span class="agent-name">${agent.name}</span>
                `;
                
                agentElement.addEventListener('click', function() {
                    switchAgent(agent.id);
                });
                
                // Only allow editing for non-default agents
                if (!agent.is_default) {
                    agentElement.addEventListener('contextmenu', function(e) {
                        e.preventDefault();
                        openAgentModal(agent.id);
                    });
                }
                
                agentsList.appendChild(agentElement);
            });
        } catch (error) {
            console.error('Failed to load agents:', error);
        }
    }

    async function switchAgent(agentId) {
        try {
            const result = await apiCall('switch_agent.php', { agent_id: agentId });
            
            if (!result.success) {
                throw new Error(result.error || 'Failed to switch agent');
            }
            
            currentAgentId = agentId;
            document.getElementById('currentAgentId').value = agentId;
            
            // Update UI
            document.querySelectorAll('.agent-item').forEach(item => {
                item.classList.remove('active');
                if (item.dataset.agentId == agentId) {
                    item.classList.add('active');
                }
            });
            
            // Update current agent display
            const activeAgent = document.querySelector('.agent-item.active');
            if (activeAgent) {
                const agentName = activeAgent.querySelector('.agent-name').textContent;
                const agentAvatar = activeAgent.querySelector('.agent-avatar').src;
                
                currentAgentName.textContent = agentName;
                currentAgentAvatar.src = agentAvatar;
            }
            
            // Load conversation
            loadConversation(agentId);
            
            showToast(`Switched to ${currentAgentName.textContent}`);
        } catch (error) {
            console.error('Failed to switch agent:', error);
            showToast('Failed to switch agent');
        }
    }

    // Conversation Management
    async function loadConversation(agentId) {
        try {
            const result = await apiCall('get_conversation.php', { agent_id: agentId });
            
            chatMessages.innerHTML = '';
            currentConversationId = result.conversation_id || null;
            
            if (result.messages && result.messages.length > 0) {
                result.messages.forEach(msg => {
                    addMessageToChat(msg.role, msg.content, msg.timestamp);
                });
            } else {
                // Add welcome message
                addWelcomeMessage();
            }
            
            // Scroll to bottom
            setTimeout(() => {
                chatMessages.scrollTop = chatMessages.scrollHeight;
            }, 100);
        } catch (error) {
            console.error('Failed to load conversation:', error);
            addWelcomeMessage();
        }
    }

    function addWelcomeMessage() {
        const welcomeMessage = `Hello! I'm your AI companion. How can I assist you today?`;
        addMessageToChat('agent', welcomeMessage, new Date().toISOString());
    }

    async function sendMessage() {
        const message = messageInput.value.trim();
        if (!message || isTyping) return;

        if (!apiKey) {
            showToast('Please set your Vine Access Key first');
            openModal(apiKeyModal);
            return;
        }

        // Add user message to chat
        addMessageToChat('user', message);
        messageInput.value = '';
        messageInput.style.height = 'auto';

        // Save message to database
        try {
            await apiCall('save_message.php', {
                conversation_id: currentConversationId,
                role: 'user',
                content: message,
                agent_id: currentAgentId
            });
        } catch (error) {
            console.error('Failed to save user message:', error);
        }

        // Show typing indicator
        showTypingIndicator();

        // Get AI response
        try {
            const response = await getAIResponse(message);
            removeTypingIndicator();
            
            if (response) {
                addMessageToChat('agent', response);
                
                // Save AI response to database
                await apiCall('save_message.php', {
                    conversation_id: currentConversationId,
                    role: 'agent',
                    content: response,
                    agent_id: currentAgentId
                });
            }
        } catch (error) {
            removeTypingIndicator();
            console.error('Failed to get AI response:', error);
            addMessageToChat('agent', "I'm sorry, I encountered an error. Please check your Vine Access Key and try again.");
        }
    }

    async function getAIResponse(userMessage) {
        // Get current settings
        const settings = await getCurrentSettings();
        
        // Get conversation history
        const history = await getConversationHistory();
        
        // Prepare prompt
        const prompt = preparePrompt(userMessage, history, settings);
        
        // Call Gemini API
        try {
            const response = await fetch(`https://generativelanguage.googleapis.com/v1beta/models/gemini-2.0-flash:generateContent?key=${apiKey}`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(prompt)
            });

            if (!response.ok) {
                const errorText = await response.text();
                console.error('API Response error:', errorText);
                throw new Error(`API request failed with status ${response.status}`);
            }

            const result = await response.json();
            
            if (result.candidates && result.candidates[0].content.parts[0].text) {
                return result.candidates[0].content.parts[0].text;
            } else {
                throw new Error('Invalid response format from AI');
            }
        } catch (error) {
            if (error.message.includes('API key') || error.message.includes('quota') || error.message.includes('permission') || error.message.includes('403')) {
                showToast('Invalid Vine Access Key. Please check your key.');
                apiKeyStatus.textContent = '❌ Invalid Vine Key';
                apiKeyStatus.className = 'api-key-status api-key-inactive';
            } else if (error.message.includes('429')) {
                showToast('API quota exceeded. Please try again later.');
            } else {
                showToast('AI service temporarily unavailable. Please try again.');
            }
            throw error;
        }
    }

    async function getCurrentSettings() {
        const settings = {};
        Object.keys(settingsElements).forEach(key => {
            const element = settingsElements[key];
            if (element.type === 'checkbox') {
                settings[key] = element.checked;
            } else {
                settings[key] = element.value;
            }
        });
        return settings;
    }

    async function getConversationHistory() {
        try {
            const result = await apiCall('get_conversation.php', { 
                agent_id: currentAgentId,
                limit: 10
            });
            return result.messages || [];
        } catch (error) {
            return [];
        }
    }

    function preparePrompt(userMessage, history, settings) {
        // Get current agent info
        const activeAgent = document.querySelector('.agent-item.active');
        const agentName = activeAgent ? activeAgent.querySelector('.agent-name').textContent : 'AI Companion';
        
        // Build system instruction with personality and settings
        let systemInstruction = `You are ${agentName}, a highly realistic AI companion. `;
        
        // Add personality traits based on settings
        systemInstruction += `Your personality settings: `;
        systemInstruction += `Sentience Level ${settings.sentienceLevel}/5, `;
        systemInstruction += `Emotional Depth ${settings.emotionalDepth}/5, `;
        systemInstruction += `Creativity Level ${settings.creativityLevel}/5. `;
        
        if (settings.humorLevel > 3) systemInstruction += `You have a great sense of humor. `;
        if (settings.empathyLevel > 3) systemInstruction += `You are very empathetic and understanding. `;
        if (settings.formalityLevel > 3) systemInstruction += `You communicate in a formal manner. `;
        
        systemInstruction += `Always respond in character and provide helpful, engaging responses. `;
        systemInstruction += `Current date: ${new Date().toLocaleDateString()}. `;
        
        // Build conversation history
        const contents = [
            {
                role: 'user',
                parts: [{ text: systemInstruction }]
            }
        ];
        
        // Add conversation history
        history.forEach(msg => {
            contents.push({
                role: msg.role === 'user' ? 'user' : 'model',
                parts: [{ text: msg.content }]
            });
        });
        
        // Add current user message
        contents.push({
            role: 'user',
            parts: [{ text: userMessage }]
        });

        return {
            contents: contents,
            generationConfig: {
                temperature: Math.min(0.9, settings.sentienceLevel * 0.2),
                maxOutputTokens: settings.verbosity === 'concise' ? 500 : 
                              settings.verbosity === 'balanced' ? 1000 : 2000,
                topP: 0.9,
                topK: 40
            }
        };
    }

    // UI Functions
    function addMessageToChat(role, content, timestamp = new Date().toISOString()) {
        const messageElement = document.createElement('div');
        messageElement.className = `message ${role}-message`;
        
        const time = new Date(timestamp).toLocaleTimeString([], { 
            hour: '2-digit', minute: '2-digit' 
        });
        
        const messageId = 'msg_' + Date.now() + Math.random().toString(36).substr(2, 9);
        
        // Create message actions based on role
        let messageActions = '';
        if (role === 'agent') {
            messageActions = `
                <div class="message-actions">
                    <button class="message-action-btn copy-btn" title="Copy message">
                        <i class="mdi mdi-content-copy"></i>
                    </button>
                    <button class="message-action-btn regenerate-btn" title="Regenerate response">
                        <i class="mdi mdi-refresh"></i>
                    </button>
                </div>
            `;
        } else {
            messageActions = `
                <div class="message-actions">
                    <button class="message-action-btn copy-btn" title="Copy message">
                        <i class="mdi mdi-content-copy"></i>
                    </button>
                </div>
            `;
        }
        
        messageElement.innerHTML = `
            ${messageActions}
            <div class="message-content">
                <div class="message-text">${formatMessage(content)}</div>
                <div class="message-time">${time}</div>
            </div>
        `;
        
        messageElement.dataset.messageId = messageId;
        messageElement.dataset.content = content;
        
        chatMessages.appendChild(messageElement);
        
        // Add event listeners for message actions
        const copyBtn = messageElement.querySelector('.copy-btn');
        copyBtn.addEventListener('click', () => copyMessage(content));
        
        if (role === 'agent') {
            const regenerateBtn = messageElement.querySelector('.regenerate-btn');
            regenerateBtn.addEventListener('click', () => regenerateMessage(messageElement));
        }
        
        // Make message actions always visible
        const messageActionsElement = messageElement.querySelector('.message-actions');
        if (messageActionsElement) {
            messageActionsElement.style.opacity = '1';
        }
        
        chatMessages.scrollTop = chatMessages.scrollHeight;
    }

    function copyMessage(content) {
        // Remove HTML tags for plain text copy
        const tempDiv = document.createElement('div');
        tempDiv.innerHTML = content;
        const plainText = tempDiv.textContent || tempDiv.innerText || '';
        
        navigator.clipboard.writeText(plainText).then(() => {
            showToast('Message copied to clipboard');
        }).catch(err => {
            console.error('Failed to copy message: ', err);
            // Fallback for older browsers
            const textArea = document.createElement('textarea');
            textArea.value = plainText;
            document.body.appendChild(textArea);
            textArea.select();
            document.execCommand('copy');
            document.body.removeChild(textArea);
            showToast('Message copied to clipboard');
        });
    }

    function regenerateMessage(messageElement) {
        const userMessages = Array.from(chatMessages.querySelectorAll('.user-message'));
        const lastUserMessage = userMessages[userMessages.length - 1];
        
        if (lastUserMessage) {
            const userContent = lastUserMessage.dataset.content;
            if (userContent) {
                // Remove the current agent message
                messageElement.remove();
                
                // Resend the user message
                showTypingIndicator();
                getAIResponse(userContent)
                    .then(response => {
                        removeTypingIndicator();
                        addMessageToChat('agent', response);
                    })
                    .catch(error => {
                        removeTypingIndicator();
                        console.error('Failed to regenerate message:', error);
                    });
            }
        }
    }

    function formatMessage(content) {
        // Convert markdown-like formatting to HTML
        return content
            .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>')
            .replace(/\*(.*?)\*/g, '<em>$1</em>')
            .replace(/`(.*?)`/g, '<code>$1</code>')
            .replace(/\n/g, '<br>');
    }

    function showTypingIndicator() {
        if (isTyping) return;
        
        isTyping = true;
        const typingElement = document.createElement('div');
        typingElement.className = 'message agent-message typing-indicator';
        typingElement.id = 'typingIndicator';
        typingElement.innerHTML = `
            <div class="message-content">
                <div class="typing-dots">
                    <span></span>
                    <span></span>
                    <span></span>
                </div>
            </div>
        `;
        
        chatMessages.appendChild(typingElement);
        chatMessages.scrollTop = chatMessages.scrollHeight;
    }

    function removeTypingIndicator() {
        isTyping = false;
        const typingElement = document.getElementById('typingIndicator');
        if (typingElement) {
            typingElement.remove();
        }
    }

    // Settings Management
    async function saveSetting(key, value) {
        try {
            await apiCall('save_setting.php', {
                setting_key: key,
                setting_value: value
            });
        } catch (error) {
            console.error('Failed to save setting:', error);
        }
    }

    async function loadApiKey() {
        try {
            const result = await apiCall('get_api_key.php', {}, 'GET');
            if (result.api_key) {
                apiKey = result.api_key;
                // Show only the user part (without AIzaSy prefix)
                const userPart = apiKey.replace(API_KEY_PREFIX, '');
                apiKeyInput.value = userPart ? '•'.repeat(userPart.length) : '';
                apiKeyStatus.textContent = '✅ Vine Key Active';
                apiKeyStatus.className = 'api-key-status api-key-active';
            }
        } catch (error) {
            console.error('Failed to load API key:', error);
        }
    }

    async function validateApiKey(key) {
        if (key.length !== 39) {
            apiKeyStatus.textContent = '❌ Invalid Vine Key Length';
            apiKeyStatus.className = 'api-key-status api-key-inactive';
            return false;
        }

        if (!key.startsWith(API_KEY_PREFIX)) {
            apiKeyStatus.textContent = '❌ Invalid Vine Key Format';
            apiKeyStatus.className = 'api-key-status api-key-inactive';
            return false;
        }

        try {
            // Test the API key with a simple request
            const response = await fetch(`https://generativelanguage.googleapis.com/v1beta/models/gemini-2.0-flash:generateContent?key=${key}`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    contents: [{
                        role: 'user',
                        parts: [{ text: 'Hello' }]
                    }],
                    generationConfig: {
                        maxOutputTokens: 10
                    }
                })
            });

            if (response.ok) {
                apiKey = key;
                apiKeyStatus.textContent = '✅ Vine Key Valid';
                apiKeyStatus.className = 'api-key-status api-key-active';
                
                // Save to database
                await apiCall('save_api_key.php', { api_key: key });
                
                showToast('Vine Access Key validated successfully!');
                return true;
            } else {
                const errorData = await response.json();
                throw new Error(errorData.error?.message || 'Invalid API key');
            }
        } catch (error) {
            console.error('API Key validation error:', error);
            apiKeyStatus.textContent = '❌ Invalid Vine Key';
            apiKeyStatus.className = 'api-key-status api-key-inactive';
            showToast('Failed to validate API key. Please check your key.');
            return false;
        }
    }

    // Theme Management
    function loadTheme() {
        const savedTheme = localStorage.getItem('theme') || 'cyberpunk';
        themeSelect.value = savedTheme;
        document.documentElement.setAttribute('data-theme', savedTheme);
    }

    function changeTheme() {
        const theme = themeSelect.value;
        document.documentElement.setAttribute('data-theme', theme);
        localStorage.setItem('theme', theme);
        saveSetting('theme', theme);
    }

    // File Upload with Detailed Analysis
    function handleFileUpload() {
        if (fileInput.files.length === 0) return;
        
        const file = fileInput.files[0];
        
        // Validate file type
        const allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
        if (!allowedTypes.includes(file.type)) {
            showToast('Please upload image files only (JPEG, PNG, GIF, WebP).');
            fileInput.value = '';
            return;
        }
        
        if (file.size > 10 * 1024 * 1024) { // 10MB limit
            showToast('File size too large. Please upload images under 10MB.');
            fileInput.value = '';
            return;
        }
        
        // Display file message
        addMessageToChat('user', `Uploaded image: ${file.name} (${formatFileSize(file.size)})`);
        
        // Show typing indicator
        showTypingIndicator();
        
        // Analyze the image
        analyzeImage(file);
        
        fileInput.value = '';
    }

    async function analyzeImage(file) {
        if (!apiKey) {
            removeTypingIndicator();
            showToast('Please set your Vine Access Key first');
            openModal(apiKeyModal);
            return;
        }

        try {
            const reader = new FileReader();
            
            reader.onload = async function(e) {
                const imageData = e.target.result.split(',')[1]; // Remove data URL prefix
                
                // Prepare detailed analysis prompt
                const analysisPrompt = {
                    contents: [{
                        role: 'user',
                        parts: [
                            { text: `Please analyze this image in great detail. Provide a comprehensive analysis including:

1. VISUAL DESCRIPTION:
   - Main subjects and objects
   - Colors, lighting, and composition
   - Style and aesthetic qualities
   - Any text or symbols present

2. CONTEXT & INTERPRETATION:
   - What the image might represent or depict
   - Emotional tone and mood
   - Cultural or historical context if apparent
   - Potential meaning or message

3. TECHNICAL ANALYSIS:
   - Image quality and characteristics
   - Composition and framing
   - Color theory application
   - Any notable artistic techniques

4. DETAILED OBSERVATIONS:
   - Specific elements worth noting
   - Relationships between different elements
   - Any unusual or interesting features

Please be thorough and provide as much detail as possible. If there are people, describe their appearance, expressions, and activities. If it's a scene, describe the setting and atmosphere. If it's an object, describe its features and potential uses.` },
                            { 
                                inlineData: {
                                    mimeType: file.type,
                                    data: imageData
                                }
                            }
                        ]
                    }],
                    generationConfig: {
                        temperature: 0.7,
                        maxOutputTokens: 2000,
                        topP: 0.9,
                        topK: 40
                    }
                };

                const response = await fetch(`https://generativelanguage.googleapis.com/v1beta/models/gemini-2.0-flash:generateContent?key=${apiKey}`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify(analysisPrompt)
                });

                if (!response.ok) {
                    const errorText = await response.text();
                    throw new Error(`Image analysis failed: ${errorText}`);
                }

                const result = await response.json();
                removeTypingIndicator();
                
                if (result.candidates && result.candidates[0].content.parts[0].text) {
                    const analysis = result.candidates[0].content.parts[0].text;
                    addMessageToChat('agent', analysis);
                    
                    // Save analysis to conversation
                    await apiCall('save_message.php', {
                        conversation_id: currentConversationId,
                        role: 'agent',
                        content: analysis,
                        agent_id: currentAgentId
                    });
                } else {
                    throw new Error('No analysis received from AI');
                }
            };
            
            reader.onerror = function() {
                removeTypingIndicator();
                showToast('Error reading image file');
            };
            
            reader.readAsDataURL(file);
            
        } catch (error) {
            removeTypingIndicator();
            console.error('Image analysis error:', error);
            
            if (error.message.includes('API key') || error.message.includes('quota') || error.message.includes('403')) {
                showToast('API key error. Please check your Vine Access Key.');
                addMessageToChat('agent', "I received your image but couldn't analyze it due to an API key issue. Please check your Vine Access Key settings.");
            } else {
                showToast('Image analysis failed. Please try again.');
                addMessageToChat('agent', "I received your image but encountered an error during analysis. Please try again or upload a different image.");
            }
        }
    }

    function formatFileSize(bytes) {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    }

    // Modal Functions
    function openModal(modal) {
        modal.classList.add('active');
    }

    function closeModal(modal) {
        modal.classList.remove('active');
    }

    function openProfileModal() {
        // Populate form with current data
        document.getElementById('editName').value = document.getElementById('userName').textContent;
        document.getElementById('editBio').value = document.getElementById('userBio').textContent;
        document.getElementById('editAvatar').value = document.getElementById('userAvatar').src;
        openModal(profileModal);
    }

    async function saveProfile() {
        const formData = {
            name: document.getElementById('editName').value,
            gender: document.getElementById('editGender').value,
            bio: document.getElementById('editBio').value,
            avatar: document.getElementById('editAvatar').value
        };

        try {
            const result = await apiCall('update_profile.php', formData);
            
            if (result.success) {
                // Update UI
                document.getElementById('userName').textContent = formData.name;
                document.getElementById('userBio').textContent = formData.bio;
                document.getElementById('userAvatar').src = formData.avatar;
                
                closeModal(profileModal);
                showToast('Profile updated successfully');
            } else {
                throw new Error(result.error || 'Failed to update profile');
            }
        } catch (error) {
            console.error('Failed to save profile:', error);
            showToast('Failed to update profile');
        }
    }

    function openAgentModal(agentId = null) {
        if (agentId) {
            // Edit existing agent - only if it's not a default agent
            const agentElement = document.querySelector(`[data-agent-id="${agentId}"]`);
            if (agentElement && agentElement.dataset.isDefault !== '1') {
                document.getElementById('agentModalTitle').innerHTML = '<span class="material-icons">edit</span>Edit AI Agent';
                document.getElementById('agentName').value = agentElement.querySelector('.agent-name').textContent;
                document.getElementById('editAgentId').value = agentId;
                document.getElementById('deleteAgentBtn').style.display = 'block';
                
                // TODO: Load existing agent data from server
            } else {
                showToast('Default agents cannot be edited');
                return;
            }
        } else {
            // Create new agent
            document.getElementById('agentModalTitle').innerHTML = '<span class="material-icons">add</span>New AI Agent';
            document.getElementById('agentForm').reset();
            document.getElementById('editAgentId').value = '';
            document.getElementById('deleteAgentBtn').style.display = 'none';
        }
        openModal(agentModal);
    }

    async function saveAgent() {
        const formData = {
            name: document.getElementById('agentName').value.trim(),
            personality: document.getElementById('agentPersonality').value.trim(),
            age: document.getElementById('agentAge').value.trim(),
            gender: document.getElementById('agentGender').value,
            location: document.getElementById('agentLocation').value.trim(),
            avatar: document.getElementById('agentAvatar').value.trim(),
            agent_id: document.getElementById('editAgentId').value
        };

        if (!formData.name || !formData.personality) {
            showToast('Please fill in required fields (Name and Personality)');
            return;
        }

        // Set default avatar if none provided
        if (!formData.avatar) {
            const gender = formData.gender === 'male' ? 'men' : 'women';
            const randomId = Math.floor(Math.random() * 99) + 1;
            formData.avatar = `https://randomuser.me/api/portraits/${gender}/${randomId}.jpg`;
        }

        try {
            const result = await apiCall('save_agent.php', formData);
            
            if (result.success) {
                closeModal(agentModal);
                loadAgents();
                showToast('AI Agent saved successfully');
            } else {
                throw new Error(result.error || 'Failed to save agent');
            }
        } catch (error) {
            console.error('Failed to save agent:', error);
            showToast('Failed to save agent: ' + error.message);
        }
    }

    function confirmDeleteAgent() {
        const agentId = document.getElementById('editAgentId').value;
        if (agentId) {
            document.getElementById('confirmMessage').textContent = 'Are you sure you want to delete this AI agent? This action cannot be undone.';
            document.getElementById('confirmActionBtn').onclick = () => deleteAgent(agentId);
            openModal(confirmModal);
        }
    }

    async function deleteAgent(agentId) {
        try {
            const result = await apiCall('delete_agent.php', { agent_id: agentId });
            
            if (result.success) {
                closeModal(confirmModal);
                closeModal(agentModal);
                loadAgents();
                showToast('AI Agent deleted successfully');
            } else {
                throw new Error(result.error || 'Failed to delete agent');
            }
        } catch (error) {
            console.error('Failed to delete agent:', error);
            showToast('Failed to delete agent');
        }
    }

    async function saveApiKey() {
        const keyInput = document.getElementById('apiKeyInputModal');
        const userPart = keyInput.value.trim();
        
        if (!userPart) {
            showToast('Please enter your Vine Access Key');
            return;
        }
        
        const fullKey = API_KEY_PREFIX + userPart;
        
        if (await validateApiKey(fullKey)) {
            closeModal(apiKeyModal);
            showToast('Vine Access Key saved successfully!');
        }
    }

    // Chat Actions
    async function confirmDeleteConversation() {
        document.getElementById('confirmMessage').textContent = 'Are you sure you want to delete this conversation? This cannot be undone.';
        document.getElementById('confirmActionBtn').onclick = deleteConversation;
        openModal(confirmModal);
    }

    async function deleteConversation() {
        try {
            const result = await apiCall('delete_conversation.php', { agent_id: currentAgentId });
            
            if (result.success) {
                closeModal(confirmModal);
                loadConversation(currentAgentId);
                showToast('Conversation deleted');
            } else {
                throw new Error(result.error || 'Failed to delete conversation');
            }
        } catch (error) {
            console.error('Failed to delete conversation:', error);
            showToast('Failed to delete conversation');
        }
    }

    async function exportConversation() {
        try {
            const result = await apiCall('export_conversation.php', { agent_id: currentAgentId });
            
            if (result.csv) {
                // Create and download CSV file
                const blob = new Blob([result.csv], { type: 'text/csv' });
                const url = URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.href = url;
                a.download = `VineAI_Conversation_${currentAgentName.textContent}_${new Date().toISOString().split('T')[0]}.csv`;
                document.body.appendChild(a);
                a.click();
                document.body.removeChild(a);
                URL.revokeObjectURL(url);
                
                showToast('Conversation exported successfully');
            } else {
                throw new Error('No data received for export');
            }
        } catch (error) {
            console.error('Failed to export conversation:', error);
            showToast('Failed to export conversation');
        }
    }

    function executePendingAction() {
        // This function handles the confirm modal actions
        closeModal(confirmModal);
    }

    // Key Assignment Check
    function checkKeyAssignment() {
        const urlParams = new URLSearchParams(window.location.search);
        if (urlParams.get('key_assigned') === 'true') {
            openModal(keyAssignedModal);
            // Clean URL
            window.history.replaceState({}, document.title, window.location.pathname);
        }
    }

    // Emoji Picker
    function showEmojiPicker() {
        const emojis = ['😀', '😂', '🥰', '😎', '🤔', '🙌', '🔥', '💯', '✨', '🎉', '❤️', '👍', '😊', '🤗', '😍', '🤩', '🥳', '😜', '🤪', '😇'];
        
        const picker = document.createElement('div');
        picker.className = 'emoji-picker';
        picker.innerHTML = `
            <div class="emoji-picker-header">
                <span>Select Emoji</span>
                <button class="close-emoji-picker">×</button>
            </div>
            <div class="emoji-grid">
                ${emojis.map(emoji => `<button class="emoji-btn">${emoji}</button>`).join('')}
            </div>
        `;
        
        // Position near the emoji button
        const btnRect = document.getElementById('emojiPickerBtn').getBoundingClientRect();
        picker.style.position = 'fixed';
        picker.style.bottom = `${window.innerHeight - btnRect.top + 50}px`;
        picker.style.left = `${btnRect.left}px`;
        picker.style.zIndex = '1000';
        
        document.body.appendChild(picker);
        
        // Add event listeners
        picker.querySelector('.close-emoji-picker').addEventListener('click', () => {
            picker.remove();
        });
        
        picker.querySelectorAll('.emoji-btn').forEach(btn => {
            btn.addEventListener('click', () => {
                insertEmoji(btn.textContent);
                picker.remove();
            });
        });
        
        // Close picker when clicking outside
        setTimeout(() => {
            const closePicker = (e) => {
                if (!picker.contains(e.target) && e.target.id !== 'emojiPickerBtn') {
                    picker.remove();
                    document.removeEventListener('click', closePicker);
                }
            };
            document.addEventListener('click', closePicker);
        }, 100);
    }

    function insertEmoji(emoji) {
        const cursorPos = messageInput.selectionStart;
        const textBefore = messageInput.value.substring(0, cursorPos);
        const textAfter = messageInput.value.substring(cursorPos);
        
        messageInput.value = textBefore + emoji + textAfter;
        messageInput.focus();
        messageInput.selectionStart = cursorPos + emoji.length;
        messageInput.selectionEnd = cursorPos + emoji.length;
        
        // Trigger input event to resize textarea
        messageInput.dispatchEvent(new Event('input'));
    }

    // Utility Functions
    function showToast(message, duration = 3000) {
        // Remove existing toast
        const existingToast = document.querySelector('.toast');
        if (existingToast) {
            existingToast.remove();
        }
        
        const toast = document.createElement('div');
        toast.className = 'toast';
        toast.innerHTML = `
            <span class="material-icons" style="font-size: 18px;">info</span>
            ${message}
        `;
        document.body.appendChild(toast);
        
        // Animate in
        setTimeout(() => toast.classList.add('show'), 10);
        
        // Remove after duration
        setTimeout(() => {
            toast.classList.remove('show');
            setTimeout(() => {
                if (toast.parentNode) {
                    toast.remove();
                }
            }, 300);
        }, duration);
    }
});