<?php
// api.php - Handle API key operations
require_once 'auth.php';
require_once 'config.php';

header('Content-Type: application/json');

$auth = new Auth();

if (!$auth->isLoggedIn()) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Not authenticated']);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);
    $action = $input['action'] ?? '';
    
    switch ($action) {
        case 'save_api_key':
            $apiKeySuffix = trim($input['api_key_suffix'] ?? '');
            
            if (empty($apiKeySuffix)) {
                echo json_encode(['success' => false, 'message' => 'API key suffix cannot be empty']);
                exit;
            }
            
            // Basic validation
            if (strlen($apiKeySuffix) < 10) {
                echo json_encode(['success' => false, 'message' => 'API key suffix appears to be too short']);
                exit;
            }
            
            // Test the API key before saving
            $fullApiKey = GEMINI_API_PREFIX . $apiKeySuffix;
            $testResult = testGeminiAPI($fullApiKey);
            
            if (!$testResult['success']) {
                echo json_encode(['success' => false, 'message' => 'API key validation failed: ' . $testResult['message']]);
                exit;
            }
            
            // Save to database
            $pdo = getDBConnection();
            $stmt = $pdo->prepare("UPDATE users SET api_key = ? WHERE id = ?");
            
            if ($stmt->execute([$fullApiKey, $_SESSION['user_id']])) {
                $_SESSION['api_key'] = $fullApiKey;
                echo json_encode(['success' => true, 'message' => 'API key saved and validated successfully!']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Failed to save API key to database']);
            }
            exit;
            
        default:
            echo json_encode(['success' => false, 'message' => 'Invalid action']);
            exit;
    }
}

function testGeminiAPI($apiKey) {
    try {
        $url = GEMINI_API_BASE . '?key=' . urlencode($apiKey);
        
        $data = [
            'contents' => [
                [
                    'parts' => [
                        ['text' => 'Hello! Please respond with "OK" if you can read this.']
                    ]
                ]
            ],
            'generationConfig' => [
                'maxOutputTokens' => 10
            ]
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json'
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            return ['success' => false, 'message' => 'Connection error: ' . $error];
        }
        
        if ($httpCode === 400) {
            return ['success' => false, 'message' => 'Invalid API key format'];
        } elseif ($httpCode === 403) {
            return ['success' => false, 'message' => 'API key is invalid or has insufficient permissions'];
        } elseif ($httpCode === 429) {
            return ['success' => false, 'message' => 'API rate limit exceeded'];
        } elseif ($httpCode !== 200) {
            return ['success' => false, 'message' => 'API error (HTTP ' . $httpCode . ')'];
        }
        
        $responseData = json_decode($response, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            return ['success' => false, 'message' => 'Invalid response from API'];
        }
        
        if (isset($responseData['error'])) {
            return ['success' => false, 'message' => 'API error: ' . ($responseData['error']['message'] ?? 'Unknown error')];
        }
        
        return ['success' => true, 'message' => 'API key is valid'];
        
    } catch (Exception $e) {
        return ['success' => false, 'message' => 'Test failed: ' . $e->getMessage()];
    }
}

echo json_encode(['success' => false, 'message' => 'Invalid request method']);
?>