<?php
require_once 'auth.php';
$auth = new Auth();

if (!$auth->isLoggedIn()) {
    header('Location: index.php');
    exit;
}

// Initialize database
initializeDatabase();

// Get user data
$pdo = getDBConnection();
$userId = $_SESSION['user_id'];
$currentUser = $auth->getCurrentUser();

// Handle various actions
if ($_POST && isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    switch ($_POST['action']) {
        case 'delete_article':
            $stmt = $pdo->prepare("DELETE FROM articles WHERE id = ? AND user_id = ?");
            $result = $stmt->execute([$_POST['article_id'], $userId]);
            echo json_encode(['success' => $result]);
            exit;
            
        case 'create_ai_agent':
            $name = $_POST['name'] ?? '';
            $description = $_POST['description'] ?? '';
            $expertise = $_POST['expertise'] ?? '';
            $tone = $_POST['tone'] ?? '';
            
            if (empty($name)) {
                echo json_encode(['success' => false, 'message' => 'Agent name is required']);
                exit;
            }
            
            $personaConfig = json_encode([
                'role' => $name,
                'expertise' => $expertise,
                'tone' => $tone,
                'description' => $description,
                'created_at' => date('Y-m-d H:i:s')
            ]);
            
            $stmt = $pdo->prepare("INSERT INTO ai_agents (user_id, name, description, expertise_area, tone_style, persona_config) VALUES (?, ?, ?, ?, ?, ?)");
            $result = $stmt->execute([$userId, $name, $description, $expertise, $tone, $personaConfig]);
            echo json_encode(['success' => $result, 'message' => $result ? 'AI Agent created successfully' : 'Failed to create agent']);
            exit;
            
        case 'delete_ai_agent':
            $stmt = $pdo->prepare("DELETE FROM ai_agents WHERE id = ? AND user_id = ?");
            $result = $stmt->execute([$_POST['agent_id'], $userId]);
            echo json_encode(['success' => $result]);
            exit;
            
        case 'update_settings':
            $preferences = [
                'theme' => $_POST['theme'] ?? 'light',
                'results_per_page' => intval($_POST['results_per_page'] ?? 10),
                'default_ai_agent' => $_POST['default_ai_agent'] ?? '',
                'auto_save' => isset($_POST['auto_save']),
                'streaming_enabled' => isset($_POST['streaming_enabled'])
            ];
            
            $result = $auth->updateUserPreferences($userId, $preferences);
            echo json_encode(['success' => $result, 'message' => $result ? 'Settings updated successfully' : 'Failed to update settings']);
            exit;
    }
}

// Get user stats
$stmt = $pdo->prepare("SELECT COUNT(*) FROM articles WHERE user_id = ?");
$stmt->execute([$userId]);
$articleCount = $stmt->fetchColumn();

$stmt = $pdo->prepare("SELECT COUNT(*) FROM ai_agents WHERE user_id = ?");
$stmt->execute([$userId]);
$agentCount = $stmt->fetchColumn();

// Get recent articles
$stmt = $pdo->prepare("SELECT * FROM articles WHERE user_id = ? ORDER BY created_at DESC LIMIT 6");
$stmt->execute([$userId]);
$recentArticles = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get AI agents
$stmt = $pdo->prepare("SELECT * FROM ai_agents WHERE user_id = ? ORDER BY created_at DESC");
$stmt->execute([$userId]);
$aiAgents = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get user preferences
$userPreferences = $currentUser['preferences'] ? json_decode($currentUser['preferences'], true) : [];
?>
<!DOCTYPE html>
<html lang="en" data-theme="<?php echo $userPreferences['theme'] ?? 'light'; ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard - Uncensored AI Writer</title>
    <link rel="stylesheet" href="styles.css">
    <style>
        .tab-content { display: none; }
        .tab-content.active { display: block; }
        .articles-grid, .agents-grid { 
            display: grid; 
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr)); 
            gap: 1.5rem; 
            margin-top: 1rem;
        }
        .modal { display: none; }
        .modal.active { display: flex; }
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            margin: 1.5rem 0;
        }
        .stat-item {
            background: var(--bg-secondary);
            padding: 1rem;
            border-radius: 0.5rem;
            text-align: center;
        }
        .stat-number {
            font-size: 2rem;
            font-weight: bold;
            color: var(--primary-color);
        }
        .quick-actions {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 1rem;
            margin: 1.5rem 0;
        }
        .action-card {
            background: var(--bg-primary);
            border: 1px solid var(--border-color);
            border-radius: 0.5rem;
            padding: 1.5rem;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s;
        }
        .action-card:hover {
            transform: translateY(-2px);
            box-shadow: var(--shadow-lg);
            border-color: var(--primary-color);
        }
        .action-icon {
            font-size: 2rem;
            margin-bottom: 0.5rem;
        }
    </style>
</head>
<body>
    <header class="header">
        <nav class="nav container">
            <a href="dashboard.php" class="logo">Uncensored AI Writer</a>
            <div class="nav-links">
                <a href="dashboard.php" class="nav-link active">Dashboard</a>
                <a href="writer.php" class="nav-link">AI Writer</a>
                <a href="#" class="nav-link" onclick="showTab('articles')">My Articles</a>
                <a href="#" class="nav-link" onclick="showTab('agents')">AI Agents</a>
                <button id="themeToggle" class="theme-toggle">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                        <path d="M21 12.79A9 9 0 1 1 11.21 3 7 7 0 0 0 21 12.79z"></path>
                    </svg>
                </button>
                <button id="logoutBtn" class="btn btn-secondary">Logout</button>
            </div>
        </nav>
    </header>

    <main class="container" style="padding: 2rem 0;">
        <!-- Dashboard Tabs -->
        <div class="tabs" style="margin-bottom: 2rem;">
            <button class="btn btn-secondary active" onclick="showTab('overview')">📊 Overview</button>
            <button class="btn btn-secondary" onclick="showTab('articles')">📚 My Articles</button>
            <button class="btn btn-secondary" onclick="showTab('agents')">👥 AI Agents</button>
            <button class="btn btn-secondary" onclick="showTab('settings')">⚙️ Settings</button>
        </div>

        <!-- Overview Tab -->
        <div id="overview" class="tab-content active">
            <div class="dashboard-header">
                <h1>Welcome back, <?php echo htmlspecialchars($_SESSION['username']); ?>! 👋</h1>
                <p style="color: var(--text-secondary);">Ready to create some amazing content?</p>
            </div>

            <!-- API Key Section -->
            <div class="api-key-section">
                <h3>🔑 API Key Configuration</h3>
                <p style="color: var(--text-secondary); margin-bottom: 1rem; font-size: 0.875rem;">
                    Enter your Google Gemini API key suffix to enable content generation
                </p>
                <div class="api-key-input">
                    <span class="prefix">AIzaSy</span>
                    <input type="password" class="form-input" id="api_key_suffix" 
                           placeholder="Enter your API key suffix" 
                           value="<?php echo isset($_SESSION['api_key']) ? str_replace('AIzaSy', '', $_SESSION['api_key']) : ''; ?>">
                    <button id="saveApiKey" class="btn btn-primary">Save Key</button>
                </div>
            </div>

            <!-- Stats Grid -->
            <div class="stats-grid">
                <div class="stat-item">
                    <div class="stat-number"><?php echo $articleCount; ?></div>
                    <div>Articles Generated</div>
                </div>
                <div class="stat-item">
                    <div class="stat-number"><?php echo $agentCount; ?></div>
                    <div>AI Agents</div>
                </div>
                <div class="stat-item">
                    <div class="stat-number"><?php echo $currentUser['credits_remaining'] ?? 100; ?></div>
                    <div>Credits Remaining</div>
                </div>
                <div class="stat-item">
                    <div class="stat-number">∞</div>
                    <div>Unlimited Creativity</div>
                </div>
            </div>

            <!-- Quick Actions -->
            <div class="quick-actions">
                <div class="action-card" onclick="window.location.href='writer.php'">
                    <div class="action-icon">🚀</div>
                    <div>Start Writing</div>
                </div>
                <div class="action-card" onclick="showTab('agents')">
                    <div class="action-icon">👥</div>
                    <div>Manage AI Agents</div>
                </div>
                <div class="action-card" onclick="showTab('articles')">
                    <div class="action-icon">📚</div>
                    <div>Article Library</div>
                </div>
                <div class="action-card" onclick="showTab('settings')">
                    <div class="action-icon">⚙️</div>
                    <div>Settings</div>
                </div>
            </div>

            <!-- Recent Articles -->
            <div style="margin-top: 2rem;">
                <div style="display: flex; justify-content: between; align-items: center; margin-bottom: 1rem;">
                    <h2>Recent Articles</h2>
                    <button class="btn btn-primary" onclick="showTab('articles')">View All</button>
                </div>
                <?php if ($recentArticles): ?>
                    <div class="articles-grid">
                        <?php foreach ($recentArticles as $article): ?>
                            <div class="article-card" data-id="<?php echo $article['id']; ?>">
                                <div class="article-title"><?php echo htmlspecialchars($article['title']); ?></div>
                                <div class="article-meta">
                                    <?php echo date('M j, Y', strtotime($article['created_at'])); ?> • 
                                    <?php echo $article['content_type'] ?? 'Article'; ?> •
                                    <?php echo $article['word_count'] ?? 'N/A'; ?> words
                                </div>
                                <div class="article-actions">
                                    <button class="btn btn-secondary" onclick="viewArticle(<?php echo $article['id']; ?>)">View</button>
                                    <button class="btn btn-secondary" onclick="editArticle(<?php echo $article['id']; ?>)">Edit</button>
                                    <button class="btn btn-danger" onclick="deleteArticle(<?php echo $article['id']; ?>)">Delete</button>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php else: ?>
                    <div style="text-align: center; padding: 3rem; color: var(--text-secondary); background: var(--bg-secondary); border-radius: 0.5rem;">
                        <p>No articles yet. Start creating your first masterpiece!</p>
                        <a href="writer.php" class="btn btn-primary mt-2">Start Writing</a>
                    </div>
                <?php endif; ?>
            </div>

            <!-- Support Section -->
            <div style="text-align: center; margin-top: 3rem; padding: 2rem; background: var(--bg-secondary); border-radius: 1rem;">
                <h3>💝 Support the Development</h3>
                <p style="color: var(--text-secondary); margin-bottom: 1rem;">
                    Help make Uncensored AI Writer even better!
                </p>
                <div style="display: flex; gap: 1rem; justify-content: center; align-items: center;">
                    <select id="donationAmount" class="form-input" style="width: auto;">
                        <option value="5">$5</option>
                        <option value="10" selected>$10</option>
                        <option value="25">$25</option>
                        <option value="50">$50</option>
                        <option value="100">$100</option>
                    </select>
                    <button id="donateBtn" class="btn btn-success">Donate Now</button>
                </div>
            </div>
        </div>

        <!-- Articles Tab -->
        <div id="articles" class="tab-content">
            <div style="display: flex; justify-content: between; align-items: center; margin-bottom: 2rem;">
                <h2>📚 Article Library</h2>
                <div style="display: flex; gap: 1rem;">
                    <input type="text" class="form-input" id="articleSearch" placeholder="Search articles..." style="width: 300px;">
                    <button class="btn btn-primary" onclick="searchArticles()">Search</button>
                </div>
            </div>

            <div class="articles-grid" id="articlesContainer">
                <?php 
                $stmt = $pdo->prepare("SELECT * FROM articles WHERE user_id = ? ORDER BY created_at DESC");
                $stmt->execute([$userId]);
                $allArticles = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                if ($allArticles): ?>
                    <?php foreach ($allArticles as $article): ?>
                        <div class="article-card" data-id="<?php echo $article['id']; ?>">
                            <div class="article-title"><?php echo htmlspecialchars($article['title']); ?></div>
                            <div class="article-meta">
                                <?php echo date('M j, Y g:i A', strtotime($article['created_at'])); ?><br>
                                Type: <?php echo $article['content_type'] ?? 'Article'; ?><br>
                                Words: <?php echo $article['word_count'] ?? 'N/A'; ?><br>
                                Tone: <?php echo $article['tone'] ?? 'Not specified'; ?>
                            </div>
                            <div class="article-preview" style="
                                max-height: 100px;
                                overflow: hidden;
                                margin: 0.5rem 0;
                                color: var(--text-secondary);
                                font-size: 0.875rem;
                                line-height: 1.4;
                            ">
                                <?php echo substr(strip_tags($article['content']), 0, 150) . '...'; ?>
                            </div>
                            <div class="article-actions">
                                <button class="btn btn-secondary" onclick="viewArticle(<?php echo $article['id']; ?>)">View</button>
                                <button class="btn btn-secondary" onclick="editArticle(<?php echo $article['id']; ?>)">Edit</button>
                                <button class="btn btn-secondary" onclick="exportArticle(<?php echo $article['id']; ?>, 'doc')">Export DOC</button>
                                <button class="btn btn-danger" onclick="deleteArticle(<?php echo $article['id']; ?>)">Delete</button>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php else: ?>
                    <div style="grid-column: 1 / -1; text-align: center; padding: 3rem; color: var(--text-secondary);">
                        <div style="font-size: 4rem; margin-bottom: 1rem;">📝</div>
                        <h3>No articles yet</h3>
                        <p>Start creating your first article to see it here.</p>
                        <a href="writer.php" class="btn btn-primary mt-2">Create Your First Article</a>
                    </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- AI Agents Tab -->
        <div id="agents" class="tab-content">
            <div style="display: flex; justify-content: between; align-items: center; margin-bottom: 2rem;">
                <h2>👥 AI Agents</h2>
                <button class="btn btn-primary" onclick="showCreateAgentModal()">+ Create New Agent</button>
            </div>

            <div class="agents-grid" id="agentsContainer">
                <?php if ($aiAgents): ?>
                    <?php foreach ($aiAgents as $agent): ?>
                        <?php $config = json_decode($agent['persona_config'], true); ?>
                        <div class="article-card" data-id="<?php echo $agent['id']; ?>">
                            <div class="article-title"><?php echo htmlspecialchars($agent['name']); ?></div>
                            <div class="article-meta">
                                Created: <?php echo date('M j, Y', strtotime($agent['created_at'])); ?><br>
                                Expertise: <?php echo htmlspecialchars($agent['expertise_area'] ?? 'General'); ?><br>
                                Tone: <?php echo htmlspecialchars($agent['tone_style'] ?? 'Professional'); ?><br>
                                Used: <?php echo $agent['usage_count']; ?> times
                            </div>
                            <div style="color: var(--text-secondary); font-size: 0.875rem; margin: 0.5rem 0;">
                                <?php echo htmlspecialchars($agent['description'] ?? 'No description provided'); ?>
                            </div>
                            <div class="article-actions">
                                <button class="btn btn-primary" onclick="useAgent(<?php echo $agent['id']; ?>)">Use</button>
                                <button class="btn btn-secondary" onclick="editAgent(<?php echo $agent['id']; ?>)">Edit</button>
                                <button class="btn btn-danger" onclick="deleteAgent(<?php echo $agent['id']; ?>)">Delete</button>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php else: ?>
                    <div style="grid-column: 1 / -1; text-align: center; padding: 3rem; color: var(--text-secondary);">
                        <div style="font-size: 4rem; margin-bottom: 1rem;">🤖</div>
                        <h3>No AI Agents yet</h3>
                        <p>Create your first AI agent to customize your writing experience.</p>
                        <button class="btn btn-primary mt-2" onclick="showCreateAgentModal()">Create Your First Agent</button>
                    </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Settings Tab -->
        <div id="settings" class="tab-content">
            <h2>⚙️ Settings</h2>
            
            <form id="settingsForm" style="max-width: 600px;">
                <div class="form-group">
                    <label class="form-label">Theme</label>
                    <select class="form-input" name="theme">
                        <option value="light" <?php echo ($userPreferences['theme'] ?? 'light') === 'light' ? 'selected' : ''; ?>>Light</option>
                        <option value="dark" <?php echo ($userPreferences['theme'] ?? 'light') === 'dark' ? 'selected' : ''; ?>>Dark</option>
                    </select>
                </div>

                <div class="form-group">
                    <label class="form-label">Results Per Page</label>
                    <select class="form-input" name="results_per_page">
                        <option value="10" <?php echo ($userPreferences['results_per_page'] ?? 10) === 10 ? 'selected' : ''; ?>>10</option>
                        <option value="25" <?php echo ($userPreferences['results_per_page'] ?? 10) === 25 ? 'selected' : ''; ?>>25</option>
                        <option value="50" <?php echo ($userPreferences['results_per_page'] ?? 10) === 50 ? 'selected' : ''; ?>>50</option>
                    </select>
                </div>

                <div class="form-group">
                    <label class="form-label">Default AI Agent</label>
                    <select class="form-input" name="default_ai_agent">
                        <option value="">-- Select Default --</option>
                        <?php foreach ($aiAgents as $agent): ?>
                            <option value="<?php echo $agent['id']; ?>" 
                                <?php echo ($userPreferences['default_ai_agent'] ?? '') == $agent['id'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($agent['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <div class="form-group">
                    <label class="checkbox-label">
                        <input type="checkbox" name="auto_save" <?php echo ($userPreferences['auto_save'] ?? false) ? 'checked' : ''; ?>>
                        Auto-save articles during generation
                    </label>
                </div>

                <div class="form-group">
                    <label class="checkbox-label">
                        <input type="checkbox" name="streaming_enabled" <?php echo ($userPreferences['streaming_enabled'] ?? true) ? 'checked' : ''; ?>>
                        Enable real-time streaming
                    </label>
                </div>

                <button type="button" class="btn btn-primary" onclick="saveSettings()">Save Settings</button>
            </form>

            <div style="margin-top: 3rem; padding-top: 2rem; border-top: 1px solid var(--border-color);">
                <h3>Account Information</h3>
                <div style="background: var(--bg-secondary); padding: 1rem; border-radius: 0.5rem;">
                    <p><strong>Username:</strong> <?php echo htmlspecialchars($currentUser['username']); ?></p>
                    <p><strong>Email:</strong> <?php echo htmlspecialchars($currentUser['email']); ?></p>
                    <p><strong>Member since:</strong> <?php echo date('F j, Y', strtotime($currentUser['created_at'])); ?></p>
                    <p><strong>Subscription:</strong> <?php echo ucfirst($currentUser['subscription_type'] ?? 'free'); ?></p>
                </div>
            </div>
        </div>
    </main>

    <!-- Create Agent Modal -->
    <div id="createAgentModal" class="modal">
        <div class="modal-content">
            <div class="panel-header">
                <h3>Create New AI Agent</h3>
                <button onclick="closeModal('createAgentModal')" style="background: none; border: none; font-size: 1.5rem; cursor: pointer;">×</button>
            </div>
            <form id="createAgentForm">
                <div class="form-group">
                    <label class="form-label">Agent Name *</label>
                    <input type="text" class="form-input" name="name" required placeholder="e.g., Technical Writer Expert">
                </div>
                <div class="form-group">
                    <label class="form-label">Description</label>
                    <textarea class="form-input" name="description" rows="3" placeholder="Describe this agent's specialty and style"></textarea>
                </div>
                <div class="form-group">
                    <label class="form-label">Expertise Area</label>
                    <input type="text" class="form-input" name="expertise" placeholder="e.g., SEO, Technical Writing, Marketing">
                </div>
                <div class="form-group">
                    <label class="form-label">Default Tone</label>
                    <select class="form-input" name="tone">
                        <option value="Professional">Professional</option>
                        <option value="Casual">Casual</option>
                        <option value="Friendly">Friendly</option>
                        <option value="Authoritative">Authoritative</option>
                        <option value="Conversational">Conversational</option>
                    </select>
                </div>
                <div class="controls">
                    <button type="button" class="btn btn-secondary" onclick="closeModal('createAgentModal')">Cancel</button>
                    <button type="submit" class="btn btn-primary">Create Agent</button>
                </div>
            </form>
        </div>
    </div>

    <!-- View Article Modal -->
    <div id="viewArticleModal" class="modal">
        <div class="modal-content" style="max-width: 800px; max-height: 90vh;">
            <div class="panel-header">
                <h3 id="viewArticleTitle">Article Title</h3>
                <button onclick="closeModal('viewArticleModal')" style="background: none; border: none; font-size: 1.5rem; cursor: pointer;">×</button>
            </div>
            <div id="viewArticleContent" style="max-height: 70vh; overflow-y: auto; padding: 1rem 0;">
                <!-- Article content will be loaded here -->
            </div>
            <div class="controls" style="margin-top: 1rem;">
                <button class="btn btn-secondary" onclick="closeModal('viewArticleModal')">Close</button>
                <button class="btn btn-primary" onclick="exportCurrentArticle()">Export</button>
            </div>
        </div>
    </div>

    <script src="script.js"></script>
    <script>
        // Tab management
        function showTab(tabName) {
            // Hide all tabs
            document.querySelectorAll('.tab-content').forEach(tab => {
                tab.classList.remove('active');
            });
            
            // Remove active class from all buttons
            document.querySelectorAll('.tabs .btn').forEach(btn => {
                btn.classList.remove('active');
            });
            
            // Show selected tab
            document.getElementById(tabName).classList.add('active');
            
            // Set active button (find button that matches the tab)
            document.querySelectorAll('.tabs .btn, .nav-link').forEach(btn => {
                if (btn.textContent.includes(tabName.replace('overview', 'Dashboard').replace('articles', 'My Articles').replace('agents', 'AI Agents').replace('settings', 'Settings'))) {
                    btn.classList.add('active');
                }
            });
        }

        // Modal management
        function showModal(modalId) {
            document.getElementById(modalId).classList.add('active');
        }

        function closeModal(modalId) {
            document.getElementById(modalId).classList.remove('active');
        }

        function showCreateAgentModal() {
            document.getElementById('createAgentForm').reset();
            showModal('createAgentModal');
        }

        // Article management
        async function viewArticle(articleId) {
            try {
                const response = await fetch('get_article.php?id=' + articleId);
                const article = await response.json();
                
                document.getElementById('viewArticleTitle').textContent = article.title;
                document.getElementById('viewArticleContent').innerHTML = article.content;
                showModal('viewArticleModal');
            } catch (error) {
                alert('Error loading article: ' + error.message);
            }
        }

        function editArticle(articleId) {
            // Redirect to writer with article data
            window.location.href = 'writer.php?edit=' + articleId;
        }

        async function deleteArticle(articleId) {
            if (!confirm('Are you sure you want to delete this article? This action cannot be undone.')) {
                return;
            }

            try {
                const response = await fetch('dashboard.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'delete_article',
                        article_id: articleId
                    })
                });

                const result = await response.json();
                
                if (result.success) {
                    document.querySelector(`.article-card[data-id="${articleId}"]`).remove();
                    showNotification('Article deleted successfully', 'success');
                    
                    // Refresh article count
                    location.reload();
                } else {
                    showNotification('Failed to delete article', 'error');
                }
            } catch (error) {
                showNotification('Error deleting article', 'error');
            }
        }

        function exportArticle(articleId, format) {
            // Implementation for exporting specific article
            showNotification(`Exporting article as ${format.toUpperCase()}...`, 'info');
            // Actual export logic would go here
        }

        // AI Agent management
        document.getElementById('createAgentForm').addEventListener('submit', async function(e) {
            e.preventDefault();
            const formData = new FormData(this);
            const data = Object.fromEntries(formData);

            try {
                const response = await fetch('dashboard.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'create_ai_agent',
                        ...data
                    })
                });

                const result = await response.json();
                
                if (result.success) {
                    closeModal('createAgentModal');
                    showNotification('AI Agent created successfully!', 'success');
                    setTimeout(() => location.reload(), 1000);
                } else {
                    showNotification(result.message, 'error');
                }
            } catch (error) {
                showNotification('Error creating agent', 'error');
            }
        });

        function useAgent(agentId) {
            // Set this agent as default and redirect to writer
            localStorage.setItem('selected_agent', agentId);
            window.location.href = 'writer.php?agent=' + agentId;
        }

        function editAgent(agentId) {
            // Implementation for editing agent
            showNotification('Edit agent functionality coming soon', 'info');
        }

        async function deleteAgent(agentId) {
            if (!confirm('Are you sure you want to delete this AI agent?')) {
                return;
            }

            try {
                const response = await fetch('dashboard.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'delete_ai_agent',
                        agent_id: agentId
                    })
                });

                const result = await response.json();
                
                if (result.success) {
                    document.querySelector(`.article-card[data-id="${agentId}"]`).remove();
                    showNotification('AI Agent deleted successfully', 'success');
                    location.reload();
                } else {
                    showNotification('Failed to delete agent', 'error');
                }
            } catch (error) {
                showNotification('Error deleting agent', 'error');
            }
        }

        // Settings management
        async function saveSettings() {
            const formData = new FormData(document.getElementById('settingsForm'));
            const data = Object.fromEntries(formData);

            try {
                const response = await fetch('dashboard.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'update_settings',
                        ...data
                    })
                });

                const result = await response.json();
                
                if (result.success) {
                    showNotification('Settings saved successfully!', 'success');
                    // Apply theme immediately
                    const theme = data.theme;
                    document.documentElement.setAttribute('data-theme', theme);
                    localStorage.setItem('theme', theme);
                } else {
                    showNotification(result.message, 'error');
                }
            } catch (error) {
                showNotification('Error saving settings', 'error');
            }
        }

        // Search functionality
        function searchArticles() {
            const searchTerm = document.getElementById('articleSearch').value.toLowerCase();
            const articles = document.querySelectorAll('.article-card');
            
            articles.forEach(article => {
                const title = article.querySelector('.article-title').textContent.toLowerCase();
                const content = article.querySelector('.article-preview')?.textContent.toLowerCase() || '';
                
                if (title.includes(searchTerm) || content.includes(searchTerm)) {
                    article.style.display = 'block';
                } else {
                    article.style.display = 'none';
                }
            });
        }

        // Utility function for notifications
        function showNotification(message, type = 'info') {
            // Use the notification system from script.js
            if (window.uiManager) {
                window.uiManager.showNotification(message, type);
            } else {
                // Fallback simple notification
                alert(message);
            }
        }

        // Initialize tab based on URL hash
        document.addEventListener('DOMContentLoaded', function() {
            const hash = window.location.hash.substring(1);
            if (hash && document.getElementById(hash)) {
                showTab(hash);
            }
        });
        // Add this function to help with session updates
function updateSessionData() {
    // This function can be used to update session-related data if needed
    console.log('Session data updated');
}

// Enhanced API key validation with better error handling
function validateApiKeySuffix(suffix) {
    const input = document.getElementById('api_key_suffix');
    if (!input) return;
    
    if (!suffix) {
        input.style.borderColor = '';
        input.style.backgroundColor = '';
        return { isValid: false, message: '' };
    }
    
    // Remove any spaces
    suffix = suffix.replace(/\s/g, '');
    
    // Basic validation
    const isValidFormat = /^[a-zA-Z0-9_-]+$/.test(suffix);
    const isValidLength = suffix.length >= 10;
    
    if (!isValidFormat) {
        input.style.borderColor = '#ef4444';
        input.style.backgroundColor = 'rgba(239, 68, 68, 0.1)';
        return { isValid: false, message: 'Invalid characters in API key' };
    }
    
    if (!isValidLength) {
        input.style.borderColor = '#f59e0b';
        input.style.backgroundColor = 'rgba(245, 158, 11, 0.1)';
        return { isValid: false, message: 'API key seems too short' };
    }
    
    input.style.borderColor = '#10b981';
    input.style.backgroundColor = 'rgba(16, 185, 129, 0.1)';
    return { isValid: true, message: 'API key format looks good' };
}

// Update the API key input event listener
document.addEventListener('DOMContentLoaded', function() {
    const apiKeyInput = document.getElementById('api_key_suffix');
    if (apiKeyInput) {
        apiKeyInput.addEventListener('input', function(e) {
            const validation = validateApiKeySuffix(e.target.value);
            // You can show the validation message if you want
            if (!validation.isValid && e.target.value) {
                console.log(validation.message);
            }
        });
        
        // Load saved API key if available
        const savedSuffix = localStorage.getItem('gemini_api_suffix');
        if (savedSuffix) {
            apiKeyInput.value = savedSuffix;
            validateApiKeySuffix(savedSuffix);
        }
    }
});
    </script>
    
</body>
</html>