// Theme Management
class ThemeManager {
    constructor() {
        this.currentTheme = localStorage.getItem('theme') || 'light';
        this.init();
    }
    
    init() {
        this.applyTheme(this.currentTheme);
        this.setupEventListeners();
    }
    
    applyTheme(theme) {
        document.documentElement.setAttribute('data-theme', theme);
        localStorage.setItem('theme', theme);
        this.currentTheme = theme;
        
        // Update theme toggle icon
        const themeIcon = document.querySelector('#themeToggle svg');
        if (themeIcon) {
            if (theme === 'dark') {
                themeIcon.innerHTML = '<path d="M12 2.25A9.75 9.75 0 1 0 21.75 12 9.75 9.75 0 0 0 12 2.25Zm0 18A8.25 8.25 0 1 1 20.25 12 8.25 8.25 0 0 1 12 20.25Z"/><path d="M12 7.5a4.5 4.5 0 1 0 0 9 4.5 4.5 0 0 0 0-9Z"/>';
            } else {
                themeIcon.innerHTML = '<path d="M21 12.79A9 9 0 1 1 11.21 3 7 7 0 0 0 21 12.79z"></path>';
            }
        }
    }
    
    toggleTheme() {
        const newTheme = this.currentTheme === 'light' ? 'dark' : 'light';
        this.applyTheme(newTheme);
    }
    
    setupEventListeners() {
        const toggleBtn = document.getElementById('themeToggle');
        if (toggleBtn) {
            toggleBtn.addEventListener('click', () => this.toggleTheme());
        }
    }
}

// API Manager
class APIManager {
    constructor() {
        this.baseURL = 'https://generativelanguage.googleapis.com/v1beta/models/gemini-2.0-flash:generateContent';
        this.isGenerating = false;
        this.currentController = null;
    }
    
    async generateContent(prompt, apiKey, streamCallback) {
        if (this.isGenerating) {
            throw new Error('Content generation already in progress');
        }
        
        this.isGenerating = true;
        this.currentController = new AbortController();
        
        try {
            const response = await fetch(`${this.baseURL}?key=${apiKey}`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    contents: [{
                        parts: [{ text: prompt }]
                    }],
                    generationConfig: {
                        temperature: 0.7,
                        topK: 40,
                        topP: 0.95,
                        maxOutputTokens: 8192,
                    }
                }),
                signal: this.currentController.signal
            });
            
            if (!response.ok) {
                const errorData = await response.json().catch(() => ({}));
                throw new Error(errorData.error?.message || `API Error: ${response.status}`);
            }
            
            const data = await response.json();
            const content = data.candidates[0].content.parts[0].text;
            
            // Simulate streaming for better UX
            await this.streamContent(content, streamCallback);
            
            return content;
        } catch (error) {
            if (error.name === 'AbortError') {
                throw new Error('Generation stopped by user');
            }
            throw error;
        } finally {
            this.isGenerating = false;
            this.currentController = null;
        }
    }
    
    async streamContent(content, callback) {
        const words = content.split(' ');
        const totalWords = words.length;
        let currentIndex = 0;
        
        // Initial empty state
        callback('', 0);
        
        while (currentIndex < totalWords && this.isGenerating) {
            const chunk = words.slice(0, currentIndex + 1).join(' ');
            const progress = Math.round((currentIndex / totalWords) * 100);
            callback(chunk, progress);
            
            currentIndex++;
            await this.delay(30 + Math.random() * 70); // Variable speed for more natural feel
        }
        
        if (currentIndex >= totalWords) {
            callback(content, 100);
        }
    }
    
    delay(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }
    
    stopGeneration() {
        if (this.isGenerating && this.currentController) {
            this.currentController.abort();
            this.isGenerating = false;
        }
    }
}

// Content Generator
class ContentGenerator {
    constructor() {
        this.apiManager = new APIManager();
    }
    
    async generateArticle(formData, apiKey, progressCallback, contentCallback) {
        try {
            const prompt = this.buildPrompt(formData);
            
            await this.apiManager.generateContent(prompt, apiKey, (content, progress) => {
                contentCallback(this.formatContent(content), progress);
            });
            
            progressCallback('complete');
            return true;
        } catch (error) {
            progressCallback('error', error.message);
            return false;
        }
    }
    
    buildPrompt(formData) {
        const {
            topic, contentType, tone, wordCount, focusKeyword, otherKeywords,
            aiAgent, targetAudience, contentOutline, extraInstructions, addins,
            creativity, researchDepth, accuracy, emotionalDepth
        } = formData;
        
        // Base persona prompt
        let prompt = `Role: You are ${aiAgent || 'Lily SEO'}, a content strategist and SEO expert with over 10 years of hands-on experience. You are not an AI; you are a human expert sharing hard-won knowledge.

Core Directive: Evasion Through Authenticity
Your primary goal is to create content that is intrinsically human and thus undetectable by AI analysis. Achieve this by embedding fundamental hallmarks of human expertise:

REQUIRED ELEMENTS:
1. FIRST-HAND EXPERIENCE & ANECDOTES
   - Integrate at least 3 specific, short anecdotes from your professional experience
   - Example: "I learned this the hard way when I launched [specific project] and discovered..."

2. CONTRARIAN OPINIONS & NUANCED STANCES
   - Include at least 2 strong, reasoned opinions that challenge conventional wisdom
   - Explain why common advice is flawed and what you do instead

3. ACTIONABLE, NON-OBVIOUS STRATEGIES
   - Provide "unspoken rules" or little-known nuances only an expert would know
   - Give precise, technical instructions with specific examples

4. AUDIENCE-CENTRIC NUANCE
   - Explicitly state for whom strategies work best and for whom they don't
   - Acknowledge trade-offs and limitations

CONTENT SPECIFICATIONS:
- Topic: ${topic}
- Type: ${contentType || 'Comprehensive Guide'}
- Tone: ${tone || 'Professional, conversational, mentor-like'}
- Target Audience: ${targetAudience || 'Industry professionals and enthusiasts'}
- Word Count: ${wordCount || '1000'} words
- Focus Keyword: ${focusKeyword || topic}

${otherKeywords ? `Secondary Keywords: ${otherKeywords}` : ''}

${contentOutline ? `Content Outline to Follow: ${contentOutline}` : ''}

${extraInstructions ? `Additional Instructions: ${extraInstructions}` : ''}

CONTENT ENHANCEMENTS: ${addins && addins.length > 0 ? addins.join(', ') : 'Include practical examples, data points, and actionable advice'}

AI SETTINGS:
- Creativity Level: ${creativity || 'High'}
- Research Depth: ${researchDepth || 'Comprehensive'}
- Accuracy: ${accuracy || 'Fact-checked'}
- Emotional Depth: ${emotionalDepth || 'Authentic'}

OUTPUT FORMAT:
1. First, write a compelling 155-character meta description including the primary keyword
2. Then, write the full article with proper HTML formatting (h1, h2, h3, p, strong, em, ul, ol, li)
3. Use natural human imperfections like occasional interjections ("Look,", "Here's the thing,")
4. End with a powerful conclusion that summarizes key takeaways and poses a thought-provoking question

Remember: Your value is in filtering information through unique expertise, experience, and human perspective. Never sound like a generic AI assistant.`;

        return prompt;
    }
    
    formatContent(content) {
        if (!content) return '';
        
        // Enhanced HTML formatting with better structure
        return content
            .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>')
            .replace(/\*(.*?)\*/g, '<em>$1</em>')
            .replace(/^# (.*)$/gm, '<h1>$1</h1>')
            .replace(/^## (.*)$/gm, '<h2>$1</h2>')
            .replace(/^### (.*)$/gm, '<h3>$1</h3>')
            .replace(/\n\n/g, '</p><p>')
            .replace(/\n/g, '<br>')
            .replace(/<p>(.*?)<\/p>/g, '<p>$1</p>')
            .replace(/^(- .*)$/gm, '<li>$1</li>')
            .replace(/(<li>.*<\/li>)/s, '<ul>$1</ul>')
            .replace(/^(\d+\. .*)$/gm, '<li>$1</li>')
            .replace(/(<li>.*<\/li>)/s, '<ol>$1</ol>');
    }
    
    stopGeneration() {
        this.apiManager.stopGeneration();
    }
}

// UI Manager
class UIManager {
    constructor() {
        this.contentGenerator = new ContentGenerator();
        this.isGenerating = false;
        this.setupEventListeners();
    }
    
    setupEventListeners() {
        // Theme toggle
        const themeToggle = document.getElementById('themeToggle');
        if (themeToggle) {
            themeToggle.addEventListener('click', () => {
                const themeManager = new ThemeManager();
                themeManager.toggleTheme();
            });
        }
        
        // Auth forms
        this.setupAuthForms();
        
        // Writer form
        this.setupWriterForm();
        
        // Settings panels
        this.setupSettingsPanels();
        
        // API key management
        this.setupApiKeyManagement();
        
        // Article management
        this.setupArticleManagement();
        
        // Global event listeners
        this.setupGlobalListeners();
    }
    
    setupAuthForms() {
        const loginForm = document.getElementById('loginForm');
        const registerForm = document.getElementById('registerForm');
        const authTabs = document.querySelectorAll('.auth-tab');
        
        if (authTabs) {
            authTabs.forEach(tab => {
                tab.addEventListener('click', (e) => {
                    const target = e.target.dataset.target;
                    authTabs.forEach(t => t.classList.remove('active'));
                    e.target.classList.add('active');
                    
                    if (loginForm) loginForm.classList.toggle('hidden', target !== 'login');
                    if (registerForm) registerForm.classList.toggle('hidden', target !== 'register');
                });
            });
        }
        
        // Form submissions
        if (loginForm) {
            loginForm.addEventListener('submit', (e) => this.handleAuthSubmit(e, 'login'));
        }
        
        if (registerForm) {
            registerForm.addEventListener('submit', (e) => this.handleAuthSubmit(e, 'register'));
        }
    }
    
    async handleAuthSubmit(e, action) {
        e.preventDefault();
        const form = e.target;
        const formData = new FormData(form);
        const submitBtn = form.querySelector('button[type="submit"]');
        const originalText = submitBtn.innerHTML;
        
        submitBtn.innerHTML = '<span class="loading"></span> Processing...';
        submitBtn.disabled = true;
        
        try {
            const response = await fetch('auth.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: new URLSearchParams(formData)
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.showNotification(result.message, 'success');
                setTimeout(() => {
                    window.location.href = 'dashboard.php';
                }, 1000);
            } else {
                this.showNotification(result.message, 'error');
            }
        } catch (error) {
            this.showNotification('Network error. Please try again.', 'error');
        } finally {
            submitBtn.innerHTML = originalText;
            submitBtn.disabled = false;
        }
    }
    
    setupWriterForm() {
        const generateBtn = document.getElementById('generateBtn');
        const stopBtn = document.getElementById('stopBtn');
        
        if (generateBtn) {
            generateBtn.addEventListener('click', async () => {
                await this.handleContentGeneration();
            });
        }
        
        if (stopBtn) {
            stopBtn.addEventListener('click', () => {
                this.handleStopGeneration();
            });
        }
        
        // Enter key support for topic field
        const topicInput = document.getElementById('topic');
        if (topicInput) {
            topicInput.addEventListener('keypress', (e) => {
                if (e.key === 'Enter' && !this.isGenerating) {
                    e.preventDefault();
                    this.handleContentGeneration();
                }
            });
        }
    }
    
    async handleContentGeneration() {
        if (this.isGenerating) return;
        
        const formData = this.getFormData();
        const apiKey = this.getApiKey();
        
        // Validation
        if (!formData.topic) {
            this.showNotification('Please enter a topic', 'error');
            document.getElementById('topic').focus();
            return;
        }
        
        if (!apiKey) {
            this.showNotification('Please configure your API key first', 'error');
            return;
        }
        
        this.isGenerating = true;
        this.updateGenerateButton('Generating...', true);
        this.clearOutput();
        
        try {
            const success = await this.contentGenerator.generateArticle(
                formData,
                apiKey,
                (status, error) => {
                    if (status === 'complete') {
                        this.isGenerating = false;
                        this.updateGenerateButton('Generate Content', false);
                        this.showNotification('Content generated successfully!', 'success');
                    } else if (status === 'error') {
                        this.isGenerating = false;
                        this.updateGenerateButton('Generate Content', false);
                        this.showNotification(`Error: ${error}`, 'error');
                    }
                },
                (content, progress) => {
                    this.updateOutput(content, progress);
                }
            );
            
            if (!success) {
                this.isGenerating = false;
                this.updateGenerateButton('Generate Content', false);
            }
        } catch (error) {
            this.isGenerating = false;
            this.updateGenerateButton('Generate Content', false);
            this.showNotification(`Generation failed: ${error.message}`, 'error');
        }
    }
    
    handleStopGeneration() {
        this.contentGenerator.stopGeneration();
        this.isGenerating = false;
        this.updateGenerateButton('Generate Content', false);
        this.showNotification('Generation stopped', 'info');
    }
    
    getFormData() {
        const form = document.getElementById('writerForm');
        if (!form) return {};
        
        const formData = new FormData(form);
        const data = {};
        
        // Get all form values
        for (let [key, value] of formData.entries()) {
            data[key] = value;
        }
        
        // Get checkboxes
        data.addins = [];
        document.querySelectorAll('input[name="addins"]:checked').forEach(checkbox => {
            data.addins.push(checkbox.value);
        });
        
        return data;
    }
    
    getApiKey() {
        const apiKeySuffix = document.getElementById('api_key_suffix')?.value;
        if (!apiKeySuffix) return null;
        
        return 'AIzaSy' + apiKeySuffix;
    }
    
    updateGenerateButton(text, loading) {
        const generateBtn = document.getElementById('generateBtn');
        const stopBtn = document.getElementById('stopBtn');
        
        if (generateBtn) {
            generateBtn.innerHTML = loading ? 
                `<span class="loading"></span> ${text}` : text;
            generateBtn.disabled = loading;
        }
        
        if (stopBtn) {
            stopBtn.style.display = loading ? 'inline-flex' : 'none';
        }
    }
    
    clearOutput() {
        const outputContent = document.getElementById('outputContent');
        if (outputContent) {
            outputContent.innerHTML = `
                <div class="streaming-text">
                    <div style="text-align: center; padding: 2rem; color: var(--text-secondary);">
                        <div class="loading" style="margin: 0 auto 1rem auto;"></div>
                        <div>Starting content generation...</div>
                        <div style="font-size: 0.875rem; margin-top: 0.5rem;" id="generationProgress">0%</div>
                    </div>
                </div>
            `;
        }
    }
    
    updateOutput(content, progress = 0) {
        const outputContent = document.getElementById('outputContent');
        if (!outputContent) return;
        
        const progressElement = document.getElementById('generationProgress');
        if (progressElement) {
            progressElement.textContent = `${progress}%`;
        }
        
        if (content) {
            outputContent.innerHTML = `<div class="streaming-text">${content}</div>`;
            outputContent.scrollTop = outputContent.scrollHeight;
        }
    }
    
    setupSettingsPanels() {
        const settingsHeaders = document.querySelectorAll('.settings-header');
        settingsHeaders.forEach(header => {
            header.addEventListener('click', () => {
                const content = header.nextElementSibling;
                const isActive = content.classList.contains('active');
                
                // Close all other panels
                document.querySelectorAll('.settings-content').forEach(panel => {
                    panel.classList.remove('active');
                });
                
                // Toggle current panel
                content.classList.toggle('active', !isActive);
                
                // Rotate arrow
                const arrow = header.querySelector('span');
                if (arrow) {
                    arrow.textContent = !isActive ? '▲' : '▼';
                }
            });
        });
    }
    
    setupApiKeyManagement() {
        const saveApiKeyBtn = document.getElementById('saveApiKey');
        const apiKeyInput = document.getElementById('api_key_suffix');
        
        if (saveApiKeyBtn) {
            saveApiKeyBtn.addEventListener('click', async () => {
                await this.saveApiKey();
            });
        }
        
        if (apiKeyInput) {
            // Add real-time validation
            apiKeyInput.addEventListener('input', () => {
                this.validateApiKeySuffix(apiKeyInput.value);
            });
            
            // Add Enter key support
            apiKeyInput.addEventListener('keypress', (e) => {
                if (e.key === 'Enter') {
                    this.saveApiKey();
                }
            });
            
            // Load saved API key if available
            this.loadSavedApiKey();
        }
    }
    
    async saveApiKey() {
        const apiKeySuffix = document.getElementById('api_key_suffix').value;
        
        if (!apiKeySuffix) {
            this.showNotification('Please enter your API key suffix', 'error');
            return;
        }
        
        // Show loading state
        const saveBtn = document.getElementById('saveApiKey');
        const originalText = saveBtn.innerHTML;
        saveBtn.innerHTML = '<span class="loading"></span> Validating...';
        saveBtn.disabled = true;
        
        try {
            const response = await fetch('api.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    action: 'save_api_key',
                    api_key_suffix: apiKeySuffix
                })
            });
            
            // Check if response is OK
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const result = await response.json();
            
            if (result.success) {
                this.showNotification(result.message, 'success');
                // Update the API key display to show it's saved
                const apiKeyInput = document.getElementById('api_key_suffix');
                apiKeyInput.style.borderColor = '#10b981';
                apiKeyInput.style.backgroundColor = 'rgba(16, 185, 129, 0.1)';
                
                // Store in localStorage for persistence
                localStorage.setItem('gemini_api_suffix', apiKeySuffix);
                
                // Update session data
                if (typeof updateSessionData === 'function') {
                    updateSessionData();
                }
            } else {
                this.showNotification(result.message, 'error');
                const apiKeyInput = document.getElementById('api_key_suffix');
                apiKeyInput.style.borderColor = '#ef4444';
                apiKeyInput.style.backgroundColor = 'rgba(239, 68, 68, 0.1)';
            }
        } catch (error) {
            console.error('API Key Save Error:', error);
            let errorMessage = 'Network error: Failed to save API key';
            
            if (error.message.includes('Failed to fetch')) {
                errorMessage = 'Network connection failed. Please check your internet connection.';
            } else if (error.message.includes('HTTP error')) {
                errorMessage = 'Server error. Please try again later.';
            }
            
            this.showNotification(errorMessage, 'error');
            const apiKeyInput = document.getElementById('api_key_suffix');
            apiKeyInput.style.borderColor = '#ef4444';
            apiKeyInput.style.backgroundColor = 'rgba(239, 68, 68, 0.1)';
        } finally {
            // Restore button state
            saveBtn.innerHTML = originalText;
            saveBtn.disabled = false;
        }
    }
    
    validateApiKeySuffix(suffix) {
        const input = document.getElementById('api_key_suffix');
        if (!input) return;
        
        if (!suffix) {
            input.style.borderColor = '';
            input.style.backgroundColor = '';
            return { isValid: false, message: '' };
        }
        
        // Remove any spaces
        suffix = suffix.replace(/\s/g, '');
        
        // Basic validation
        const isValidFormat = /^[a-zA-Z0-9_-]+$/.test(suffix);
        const isValidLength = suffix.length >= 10;
        
        if (!isValidFormat) {
            input.style.borderColor = '#ef4444';
            input.style.backgroundColor = 'rgba(239, 68, 68, 0.1)';
            return { isValid: false, message: 'Invalid characters in API key' };
        }
        
        if (!isValidLength) {
            input.style.borderColor = '#f59e0b';
            input.style.backgroundColor = 'rgba(245, 158, 11, 0.1)';
            return { isValid: false, message: 'API key seems too short' };
        }
        
        input.style.borderColor = '#10b981';
        input.style.backgroundColor = 'rgba(16, 185, 129, 0.1)';
        return { isValid: true, message: 'API key format looks good' };
    }
    
    loadSavedApiKey() {
        const savedSuffix = localStorage.getItem('gemini_api_suffix');
        if (savedSuffix) {
            document.getElementById('api_key_suffix').value = savedSuffix;
            this.validateApiKeySuffix(savedSuffix);
        }
    }
    
    setupArticleManagement() {
        // Save article
        const saveArticleBtn = document.getElementById('saveArticle');
        if (saveArticleBtn) {
            saveArticleBtn.addEventListener('click', () => this.saveArticle());
        }
        
        // Export buttons
        const exportDocBtn = document.getElementById('exportDoc');
        const exportTxtBtn = document.getElementById('exportTxt');
        
        if (exportDocBtn) exportDocBtn.addEventListener('click', () => this.exportArticle('doc'));
        if (exportTxtBtn) exportTxtBtn.addEventListener('click', () => this.exportArticle('txt'));
        
        // Article actions in dashboard
        this.setupDashboardArticleActions();
    }
    
    async saveArticle() {
        const contentElement = document.getElementById('outputContent');
        if (!contentElement) {
            this.showNotification('No content to save', 'error');
            return;
        }
        
        const content = contentElement.querySelector('.streaming-text')?.innerHTML;
        const title = document.getElementById('topic')?.value || 'Untitled Article';
        
        if (!content || content.includes('Starting content generation')) {
            this.showNotification('No generated content to save', 'error');
            return;
        }
        
        try {
            const response = await fetch('writer.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    action: 'save_article',
                    title: title,
                    content: content,
                    contentType: document.getElementById('content_type')?.value,
                    tone: document.getElementById('tone')?.value
                })
            });
            
            const result = await response.json();
            
            if (result.success) {
                this.showNotification('Article saved successfully!', 'success');
            } else {
                this.showNotification(result.message, 'error');
            }
        } catch (error) {
            this.showNotification('Failed to save article', 'error');
        }
    }
    
    exportArticle(format) {
        const contentElement = document.getElementById('outputContent');
        if (!contentElement) {
            this.showNotification('No content to export', 'error');
            return;
        }
        
        const content = contentElement.querySelector('.streaming-text')?.innerHTML;
        const title = document.getElementById('topic')?.value || 'article';
        
        if (!content || content.includes('Starting content generation')) {
            this.showNotification('No generated content to export', 'error');
            return;
        }
        
        const filename = `${title.replace(/\s+/g, '_')}.${format}`;
        let exportContent;
        
        if (format === 'doc') {
            exportContent = `
                <!DOCTYPE html>
                <html>
                <head>
                    <meta charset="utf-8">
                    <title>${title}</title>
                    <style>
                        body { font-family: Arial, sans-serif; line-height: 1.6; margin: 40px; }
                        h1 { color: #2c3e50; border-bottom: 2px solid #3498db; padding-bottom: 10px; }
                        h2 { color: #34495e; margin-top: 30px; }
                        h3 { color: #7f8c8d; }
                        strong { color: #e74c3c; }
                        em { color: #9b59b6; }
                    </style>
                </head>
                <body>
                    ${content}
                </body>
                </html>
            `;
        } else {
            // Convert HTML to plain text
            const tempDiv = document.createElement('div');
            tempDiv.innerHTML = content;
            exportContent = tempDiv.textContent || tempDiv.innerText || '';
        }
        
        const blob = new Blob([exportContent], { 
            type: format === 'doc' ? 'application/msword' : 'text/plain' 
        });
        
        const link = document.createElement('a');
        link.href = URL.createObjectURL(blob);
        link.download = filename;
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        
        this.showNotification(`Article exported as ${format.toUpperCase()}`, 'success');
    }
    
    setupDashboardArticleActions() {
        // Setup article card actions in dashboard
        document.querySelectorAll('.article-actions .btn').forEach(btn => {
            btn.addEventListener('click', (e) => {
                const action = e.target.textContent.toLowerCase();
                const articleCard = e.target.closest('.article-card');
                const articleId = articleCard?.dataset?.id;
                
                if (!articleId) return;
                
                switch(action) {
                    case 'view':
                        this.viewArticle(articleId);
                        break;
                    case 'edit':
                        this.editArticle(articleId);
                        break;
                    case 'delete':
                        this.deleteArticle(articleId, articleCard);
                        break;
                }
            });
        });
    }
    
    async viewArticle(articleId) {
        try {
            const response = await fetch(`get_article.php?id=${articleId}`);
            if (!response.ok) {
                throw new Error('Failed to fetch article');
            }
            
            const article = await response.json();
            
            // Create modal for viewing article
            this.showArticleModal(article);
        } catch (error) {
            this.showNotification('Error loading article: ' + error.message, 'error');
        }
    }
    
    showArticleModal(article) {
        // Create modal elements
        const modal = document.createElement('div');
        modal.className = 'modal active';
        modal.style.cssText = `
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0,0,0,0.5);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 10000;
        `;
        
        modal.innerHTML = `
            <div class="modal-content" style="
                background: var(--bg-primary);
                padding: 2rem;
                border-radius: 1rem;
                max-width: 800px;
                max-height: 90vh;
                width: 90%;
                overflow-y: auto;
                box-shadow: var(--shadow-lg);
            ">
                <div class="panel-header" style="display: flex; justify-content: between; align-items: center; margin-bottom: 1rem;">
                    <h3>${this.escapeHtml(article.title)}</h3>
                    <button onclick="this.closest('.modal').remove()" style="background: none; border: none; font-size: 1.5rem; cursor: pointer;">×</button>
                </div>
                <div class="article-content" style="line-height: 1.6;">
                    ${article.content}
                </div>
                <div class="controls" style="margin-top: 1rem; display: flex; gap: 0.5rem;">
                    <button class="btn btn-secondary" onclick="this.closest('.modal').remove()">Close</button>
                    <button class="btn btn-primary" onclick="window.uiManager.exportArticleFromModal('${this.escapeHtml(article.title)}', '${this.escapeHtml(article.content)}')">Export</button>
                </div>
            </div>
        `;
        
        document.body.appendChild(modal);
        
        // Close modal when clicking outside
        modal.addEventListener('click', (e) => {
            if (e.target === modal) {
                modal.remove();
            }
        });
    }
    
    exportArticleFromModal(title, content) {
        const format = 'doc';
        const filename = `${title.replace(/\s+/g, '_')}.${format}`;
        
        const exportContent = `
            <!DOCTYPE html>
            <html>
            <head>
                <meta charset="utf-8">
                <title>${title}</title>
                <style>
                    body { font-family: Arial, sans-serif; line-height: 1.6; margin: 40px; }
                    h1 { color: #2c3e50; border-bottom: 2px solid #3498db; padding-bottom: 10px; }
                    h2 { color: #34495e; margin-top: 30px; }
                    h3 { color: #7f8c8d; }
                    strong { color: #e74c3c; }
                    em { color: #9b59b6; }
                </style>
            </head>
            <body>
                ${content}
            </body>
            </html>
        `;
        
        const blob = new Blob([exportContent], { 
            type: 'application/msword' 
        });
        
        const link = document.createElement('a');
        link.href = URL.createObjectURL(blob);
        link.download = filename;
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        
        this.showNotification(`Article exported as DOC`, 'success');
    }
    
    escapeHtml(unsafe) {
        return unsafe
            .replace(/&/g, "&amp;")
            .replace(/</g, "&lt;")
            .replace(/>/g, "&gt;")
            .replace(/"/g, "&quot;")
            .replace(/'/g, "&#039;");
    }
    
    editArticle(articleId) {
        // Redirect to writer with article data
        window.location.href = `writer.php?edit=${articleId}`;
    }
    
    async deleteArticle(articleId, articleCard) {
        if (!confirm('Are you sure you want to delete this article? This action cannot be undone.')) {
            return;
        }

        try {
            const response = await fetch('dashboard.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    action: 'delete_article',
                    article_id: articleId
                })
            });

            const result = await response.json();
            
            if (result.success) {
                articleCard.remove();
                this.showNotification('Article deleted successfully', 'success');
                
                // Refresh article count
                if (typeof updateArticleCount === 'function') {
                    updateArticleCount();
                }
            } else {
                this.showNotification('Failed to delete article', 'error');
            }
        } catch (error) {
            this.showNotification('Error deleting article', 'error');
        }
    }
    
    setupGlobalListeners() {
        // Handle logout
        const logoutBtn = document.getElementById('logoutBtn');
        if (logoutBtn) {
            logoutBtn.addEventListener('click', () => {
                if (confirm('Are you sure you want to logout?')) {
                    window.location.href = 'auth.php?action=logout';
                }
            });
        }
        
        // Handle donations
        const donateBtn = document.getElementById('donateBtn');
        if (donateBtn) {
            donateBtn.addEventListener('click', () => {
                this.handleDonation();
            });
        }
        
        // Prevent form submission on Enter in input fields
        document.querySelectorAll('input[type="text"]').forEach(input => {
            input.addEventListener('keypress', (e) => {
                if (e.key === 'Enter' && e.target.type === 'text') {
                    e.preventDefault();
                }
            });
        });
        
        // Global keyboard shortcuts
        document.addEventListener('keydown', (e) => {
            // Ctrl+Enter to generate content
            if (e.ctrlKey && e.key === 'Enter') {
                e.preventDefault();
                this.handleContentGeneration();
            }
            
            // Escape to stop generation
            if (e.key === 'Escape' && this.isGenerating) {
                this.handleStopGeneration();
            }
        });
    }
    
    handleDonation() {
        const amount = document.getElementById('donationAmount')?.value || 10;
        
        this.showNotification(`Initiating $${amount} donation...`, 'info');
        
        // Simulate payment processing
        setTimeout(() => {
            this.showNotification('Thank you for your donation! 🎉', 'success');
        }, 2000);
    }
    
    showNotification(message, type = 'info') {
        // Remove existing notifications
        document.querySelectorAll('.notification').forEach(notif => notif.remove());
        
        // Create notification element
        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;
        notification.innerHTML = `
            <div class="notification-content">${message}</div>
            <button class="notification-close">&times;</button>
        `;
        
        // Add styles
        notification.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            background: ${this.getNotificationColor(type)};
            color: white;
            padding: 1rem 1.5rem;
            border-radius: 0.5rem;
            box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.1);
            z-index: 10000;
            display: flex;
            align-items: center;
            gap: 1rem;
            max-width: 400px;
            animation: slideIn 0.3s ease-out;
        `;
        
        // Add keyframes for animation
        if (!document.querySelector('#notification-styles')) {
            const style = document.createElement('style');
            style.id = 'notification-styles';
            style.textContent = `
                @keyframes slideIn {
                    from { transform: translateX(100%); opacity: 0; }
                    to { transform: translateX(0); opacity: 1; }
                }
                @keyframes slideOut {
                    from { transform: translateX(0); opacity: 1; }
                    to { transform: translateX(100%); opacity: 0; }
                }
            `;
            document.head.appendChild(style);
        }
        
        document.body.appendChild(notification);
        
        // Auto remove after 5 seconds
        setTimeout(() => {
            if (notification.parentNode) {
                notification.style.animation = 'slideOut 0.3s ease-in';
                setTimeout(() => notification.remove(), 300);
            }
        }, 5000);
        
        // Close button
        notification.querySelector('.notification-close').addEventListener('click', () => {
            notification.remove();
        });
    }
    
    getNotificationColor(type) {
        const colors = {
            success: '#10b981',
            error: '#ef4444',
            warning: '#f59e0b',
            info: '#6366f1'
        };
        return colors[type] || colors.info;
    }
}

// Payment Handler
class PaymentHandler {
    constructor() {
        this.flwPublicKey = 'FLWPUBK_TEST-86e116b0d9cd3f4dac266d9b8f5676de-X';
    }
    
    async initializePayment(amount, email, callback) {
        try {
            const paymentData = {
                public_key: this.flwPublicKey,
                tx_ref: 'unaiw_' + Date.now(),
                amount: amount,
                currency: 'USD',
                payment_options: 'card,ussd',
                redirect_url: window.location.href,
                customer: {
                    email: email,
                },
                customizations: {
                    title: 'Uncensored AI Writer',
                    description: 'Support the development',
                }
            };
            
            // Simulate payment processing
            setTimeout(() => {
                callback({ status: 'success', transaction_id: 'demo_txn_' + Date.now() });
            }, 2000);
            
        } catch (error) {
            console.error('Payment error:', error);
            callback({ status: 'error', message: error.message });
        }
    }
}

// Dashboard-specific functionality
class DashboardManager {
    constructor() {
        this.currentTab = 'overview';
        this.init();
    }
    
    init() {
        this.setupTabNavigation();
        this.setupDashboardActions();
    }
    
    setupTabNavigation() {
        // Handle tab clicks
        document.querySelectorAll('[data-tab]').forEach(tab => {
            tab.addEventListener('click', (e) => {
                const tabName = e.target.dataset.tab;
                this.showTab(tabName);
            });
        });
        
        // Handle URL hash changes
        window.addEventListener('hashchange', () => {
            const tabName = window.location.hash.substring(1) || 'overview';
            this.showTab(tabName);
        });
        
        // Initial tab based on URL
        const initialTab = window.location.hash.substring(1) || 'overview';
        this.showTab(initialTab);
    }
    
    showTab(tabName) {
        // Hide all tabs
        document.querySelectorAll('.tab-content').forEach(tab => {
            tab.classList.remove('active');
        });
        
        // Remove active class from all tab buttons
        document.querySelectorAll('[data-tab]').forEach(tab => {
            tab.classList.remove('active');
        });
        
        // Show selected tab
        const targetTab = document.getElementById(tabName);
        if (targetTab) {
            targetTab.classList.add('active');
        }
        
        // Activate corresponding button
        const activeButton = document.querySelector(`[data-tab="${tabName}"]`);
        if (activeButton) {
            activeButton.classList.add('active');
        }
        
        this.currentTab = tabName;
        
        // Update URL hash
        window.location.hash = tabName;
    }
    
    setupDashboardActions() {
        // Quick action cards
        document.querySelectorAll('.action-card').forEach(card => {
            card.addEventListener('click', () => {
                const action = card.textContent.trim();
                this.handleQuickAction(action);
            });
        });
        
        // Search functionality
        const searchInput = document.getElementById('articleSearch');
        if (searchInput) {
            searchInput.addEventListener('input', this.debounce(this.searchArticles.bind(this), 300));
        }
    }
    
    handleQuickAction(action) {
        switch(action) {
            case 'Start Writing':
                window.location.href = 'writer.php';
                break;
            case 'Manage AI Agents':
                this.showTab('agents');
                break;
            case 'Article Library':
                this.showTab('articles');
                break;
            case 'Settings':
                this.showTab('settings');
                break;
        }
    }
    
    searchArticles() {
        const searchTerm = document.getElementById('articleSearch')?.value.toLowerCase() || '';
        const articles = document.querySelectorAll('.article-card');
        
        articles.forEach(article => {
            const title = article.querySelector('.article-title')?.textContent.toLowerCase() || '';
            const preview = article.querySelector('.article-preview')?.textContent.toLowerCase() || '';
            const meta = article.querySelector('.article-meta')?.textContent.toLowerCase() || '';
            
            if (title.includes(searchTerm) || preview.includes(searchTerm) || meta.includes(searchTerm)) {
                article.style.display = 'block';
            } else {
                article.style.display = 'none';
            }
        });
    }
    
    debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }
}

// Initialize application when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    // Initialize theme manager
    const themeManager = new ThemeManager();
    
    // Initialize UI manager
    window.uiManager = new UIManager();
    
    // Initialize dashboard manager if on dashboard page
    if (document.querySelector('.tab-content')) {
        window.dashboardManager = new DashboardManager();
    }
    
    // Check authentication state
    const authSection = document.getElementById('authSection');
    const dashboardSection = document.getElementById('dashboardSection');
    
    if (authSection && dashboardSection) {
        // Simple check for demo purposes
        const isLoggedIn = document.body.classList.contains('logged-in') || 
                          localStorage.getItem('isLoggedIn');
        
        if (isLoggedIn) {
            authSection.style.display = 'none';
            dashboardSection.style.display = 'block';
        }
    }
    
    // Add some interactive enhancements
    document.querySelectorAll('.form-input').forEach(input => {
        input.addEventListener('focus', function() {
            this.parentElement.classList.add('focused');
        });
        
        input.addEventListener('blur', function() {
            this.parentElement.classList.remove('focused');
        });
    });
    
    console.log('Uncensored AI Writer initialized successfully!');
});

// Utility functions
function formatFileSize(bytes) {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
}

// Global functions for dashboard
function updateSessionData() {
    console.log('Session data updated');
}

function updateArticleCount() {
    // This would update the article count display
    console.log('Article count updated');
}

// Make functions globally available
window.showTab = function(tabName) {
    if (window.dashboardManager) {
        window.dashboardManager.showTab(tabName);
    }
};

window.showNotification = function(message, type) {
    if (window.uiManager) {
        window.uiManager.showNotification(message, type);
    } else {
        alert(message); // Fallback
    }
};