<?php
require_once 'auth.php';
$auth = new Auth();

if (!$auth->isLoggedIn()) {
    header('Location: index.php');
    exit;
}

// Initialize database
initializeDatabase();

// Get user data
$pdo = getDBConnection();
$userId = $_SESSION['user_id'];
$currentUser = $auth->getCurrentUser();

// Handle article saving
if ($_POST && isset($_POST['action'])) {
    header('Content-Type: application/json');
    
    if ($_POST['action'] === 'save_article') {
        $title = $_POST['title'] ?? 'Untitled Article';
        $content = $_POST['content'] ?? '';
        $contentType = $_POST['contentType'] ?? 'Article';
        $tone = $_POST['tone'] ?? 'Professional';
        
        if (empty($content)) {
            echo json_encode(['success' => false, 'message' => 'No content to save']);
            exit;
        }
        
        $stmt = $pdo->prepare("INSERT INTO articles (user_id, title, content, content_type, tone, word_count) VALUES (?, ?, ?, ?, ?, ?)");
        
        // Calculate word count
        $wordCount = str_word_count(strip_tags($content));
        
        $result = $stmt->execute([
            $userId,
            $title,
            $content,
            $contentType,
            $tone,
            $wordCount
        ]);
        
        echo json_encode(['success' => $result, 'message' => $result ? 'Article saved successfully!' : 'Failed to save article']);
        exit;
    }
    
    // Handle AI content generation
    if ($_POST['action'] === 'generate_content') {
        $apiKey = $_SESSION['api_key'] ?? '';
        
        if (empty($apiKey)) {
            echo json_encode(['success' => false, 'message' => 'Please configure your API key first']);
            exit;
        }
        
        $prompt = $_POST['prompt'] ?? '';
        if (empty($prompt)) {
            echo json_encode(['success' => false, 'message' => 'No prompt provided']);
            exit;
        }
        
        // Generate content using Gemini API
        $result = generateGeminiContent($prompt, $apiKey);
        echo json_encode($result);
        exit;
    }
}

// Function to generate content using Gemini API
function generateGeminiContent($prompt, $apiKey) {
    try {
        $url = 'https://generativelanguage.googleapis.com/v1beta/models/gemini-2.0-flash:generateContent?key=' . urlencode($apiKey);
        
        $data = [
            'contents' => [
                [
                    'parts' => [
                        ['text' => $prompt]
                    ]
                ]
            ],
            'generationConfig' => [
                'temperature' => 0.7,
                'topK' => 40,
                'topP' => 0.95,
                'maxOutputTokens' => 8192,
            ]
        ];
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json'
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 60);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            return ['success' => false, 'message' => 'Connection error: ' . $error];
        }
        
        if ($httpCode !== 200) {
            if ($httpCode === 400) {
                return ['success' => false, 'message' => 'Invalid API request. Please check your API key.'];
            } elseif ($httpCode === 403) {
                return ['success' => false, 'message' => 'API key is invalid or has insufficient permissions'];
            } elseif ($httpCode === 429) {
                return ['success' => false, 'message' => 'API rate limit exceeded. Please try again later.'];
            } else {
                return ['success' => false, 'message' => 'API error (HTTP ' . $httpCode . ')'];
            }
        }
        
        $responseData = json_decode($response, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            return ['success' => false, 'message' => 'Invalid response from API'];
        }
        
        if (isset($responseData['error'])) {
            return ['success' => false, 'message' => 'API error: ' . ($responseData['error']['message'] ?? 'Unknown error')];
        }
        
        if (!isset($responseData['candidates'][0]['content']['parts'][0]['text'])) {
            return ['success' => false, 'message' => 'Unexpected API response format'];
        }
        
        $content = $responseData['candidates'][0]['content']['parts'][0]['text'];
        
        // Format the content with basic HTML
        $formattedContent = formatContent($content);
        
        return [
            'success' => true, 
            'content' => $formattedContent,
            'raw_content' => $content
        ];
        
    } catch (Exception $e) {
        return ['success' => false, 'message' => 'Generation failed: ' . $e->getMessage()];
    }
}

function formatContent($content) {
    if (empty($content)) return '';
    
    // Basic HTML formatting
    $formatted = $content;
    
    // Convert markdown-style headers to HTML
    $formatted = preg_replace('/^# (.*)$/m', '<h1>$1</h1>', $formatted);
    $formatted = preg_replace('/^## (.*)$/m', '<h2>$1</h2>', $formatted);
    $formatted = preg_replace('/^### (.*)$/m', '<h3>$1</h3>', $formatted);
    
    // Convert bold and italic
    $formatted = preg_replace('/\*\*(.*?)\*\*/', '<strong>$1</strong>', $formatted);
    $formatted = preg_replace('/\*(.*?)\*/', '<em>$1</em>', $formatted);
    
    // Convert line breaks
    $formatted = nl2br($formatted);
    
    // Convert paragraphs (double line breaks)
    $formatted = preg_replace('/([^\n])\n\n([^\n])/', '$1</p><p>$2', $formatted);
    $formatted = '<p>' . $formatted . '</p>';
    
    // Convert lists
    $formatted = preg_replace('/^- (.*)$/m', '<li>$1</li>', $formatted);
    $formatted = preg_replace('/(<li>.*<\/li>)/s', '<ul>$1</ul>', $formatted);
    
    $formatted = preg_replace('/^\d+\. (.*)$/m', '<li>$1</li>', $formatted);
    $formatted = preg_replace('/(<li>.*<\/li>)/s', '<ol>$1</ol>', $formatted);
    
    return $formatted;
}

// Get user's AI agents
$stmt = $pdo->prepare("SELECT * FROM ai_agents WHERE user_id = ? ORDER BY name");
$stmt->execute([$userId]);
$userAgents = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Check if API key is configured
$apiKeyConfigured = !empty($_SESSION['api_key']);
$apiKeySuffix = $apiKeyConfigured ? str_replace('AIzaSy', '', $_SESSION['api_key']) : '';
?>
<!DOCTYPE html>
<html lang="en" data-theme="light">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>AI Writer - Uncensored AI Writer</title>
    <link rel="stylesheet" href="styles.css">
    <style>
        .api-status {
            padding: 0.5rem 1rem;
            border-radius: 0.25rem;
            font-size: 0.875rem;
            font-weight: 500;
            margin-left: 1rem;
        }
        .api-status.connected {
            background: #dcfce7;
            color: #166534;
            border: 1px solid #bbf7d0;
        }
        .api-status.disconnected {
            background: #fef2f2;
            color: #dc2626;
            border: 1px solid #fecaca;
        }
        .generation-status {
            display: none;
            padding: 1rem;
            background: var(--bg-secondary);
            border-radius: 0.5rem;
            margin: 1rem 0;
        }
        .generation-status.active {
            display: block;
        }
        .progress-bar {
            width: 100%;
            height: 4px;
            background: var(--border-color);
            border-radius: 2px;
            overflow: hidden;
            margin: 0.5rem 0;
        }
        .progress-fill {
            height: 100%;
            background: var(--primary-color);
            width: 0%;
            transition: width 0.3s ease;
        }
    </style>
</head>
<body>
    <header class="header">
        <nav class="nav container">
            <a href="dashboard.php" class="logo">Uncensored AI Writer</a>
            <div class="nav-links">
                <a href="dashboard.php" class="nav-link">Dashboard</a>
                <a href="writer.php" class="nav-link active">AI Writer</a>
                <a href="#" class="nav-link" onclick="showTab('articles')">My Articles</a>
                <a href="#" class="nav-link" onclick="showTab('agents')">AI Agents</a>
                <span class="api-status <?php echo $apiKeyConfigured ? 'connected' : 'disconnected'; ?>">
                    <?php echo $apiKeyConfigured ? '✓ API Connected' : '✗ API Disconnected'; ?>
                </span>
                <button id="themeToggle" class="theme-toggle">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor">
                        <path d="M21 12.79A9 9 0 1 1 11.21 3 7 7 0 0 0 21 12.79z"></path>
                    </svg>
                </button>
                <button id="logoutBtn" class="btn btn-secondary">Logout</button>
            </div>
        </nav>
    </header>

    <main class="container" style="padding: 2rem 0;">
        <!-- API Key Warning -->
        <?php if (!$apiKeyConfigured): ?>
        <div style="background: #fef2f2; border: 1px solid #fecaca; color: #dc2626; padding: 1rem; border-radius: 0.5rem; margin-bottom: 2rem;">
            <strong>⚠️ API Key Required:</strong> Please configure your Google Gemini API key in the Dashboard to generate content.
            <a href="dashboard.php" class="btn btn-primary" style="margin-left: 1rem; padding: 0.5rem 1rem;">Configure API Key</a>
        </div>
        <?php endif; ?>

        <div class="writer-container">
            <!-- Input Panel -->
            <div class="input-panel">
                <div class="panel-header">
                    <h2 class="panel-title">Content Parameters</h2>
                </div>
                
                <form id="writerForm">
                    <div class="form-group">
                        <label class="form-label" for="topic">Topic/Idea *</label>
                        <input type="text" class="form-input" id="topic" name="topic" 
                               placeholder="Enter your article topic..." required
                               value="<?php echo $_GET['topic'] ?? ''; ?>">
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="content_type">Content Type</label>
                        <select class="form-input" id="content_type" name="content_type">
                            <option value="">Select content type...</option>
                            <option value="Articles">Articles</option>
                            <option value="Blog Posts">Blog Posts</option>
                            <option value="News Posts">News Posts</option>
                            <option value="Product Descriptions">Product Descriptions</option>
                            <option value="Social Media Posts">Social Media Posts</option>
                            <option value="Email Newsletters">Email Newsletters</option>
                            <option value="Press Releases">Press Releases</option>
                            <option value="Case Studies">Case Studies</option>
                            <option value="White Papers">White Papers</option>
                            <option value="How-To Guides">How-To Guides</option>
                            <option value="Tutorials">Tutorials</option>
                            <option value="FAQs">FAQs</option>
                            <option value="Landing Pages">Landing Pages</option>
                            <option value="Video Scripts">Video Scripts</option>
                            <option value="Podcast Outlines">Podcast Outlines</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="tone">Tone/Voice</label>
                        <select class="form-input" id="tone" name="tone">
                            <option value="">Select tone...</option>
                            <option value="Professional">Professional</option>
                            <option value="Casual">Casual</option>
                            <option value="Friendly">Friendly</option>
                            <option value="Formal">Formal</option>
                            <option value="Authoritative">Authoritative</option>
                            <option value="Conversational">Conversational</option>
                            <option value="Humorous">Humorous</option>
                            <option value="Persuasive">Persuasive</option>
                            <option value="Inspirational">Inspirational</option>
                            <option value="Technical">Technical</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="word_count">Word Count</label>
                        <input type="number" class="form-input" id="word_count" name="word_count" 
                               placeholder="e.g., 1000" value="1000" min="100" max="10000">
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="focus_keyword">Focus Keyword</label>
                        <input type="text" class="form-input" id="focus_keyword" name="focus_keyword" 
                               placeholder="Primary keyword for SEO">
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="other_keywords">Other Keywords</label>
                        <textarea class="form-input" id="other_keywords" name="other_keywords" 
                                  placeholder="Secondary keywords, comma separated" rows="2"></textarea>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="ai_agent">AI Writer Agent</label>
                        <select class="form-input" id="ai_agent" name="ai_agent">
                            <option value="">-- Select Agent --</option>
                            <option value="Lily SEO" selected>Lily SEO (Default)</option>
                            <option value="Technical Expert">Technical Expert</option>
                            <option value="Creative Writer">Creative Writer</option>
                            <option value="Business Strategist">Business Strategist</option>
                            <option value="Marketing Guru">Marketing Guru</option>
                            <?php foreach ($userAgents as $agent): ?>
                                <option value="<?php echo htmlspecialchars($agent['name']); ?>">
                                    <?php echo htmlspecialchars($agent['name']); ?> (Custom)
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="target_audience">Target Audience</label>
                        <input type="text" class="form-input" id="target_audience" name="target_audience" 
                               placeholder="e.g., Marketing professionals, Tech enthusiasts">
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="content_outline">Content Outline</label>
                        <textarea class="form-input" id="content_outline" name="content_outline" 
                                  placeholder="Brief outline or key points to cover" rows="3"></textarea>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="extra_instructions">Extra Instructions</label>
                        <textarea class="form-input" id="extra_instructions" name="extra_instructions" 
                                  placeholder="Any specific requirements or style preferences" rows="3"></textarea>
                    </div>
                    
                    <!-- Add-ins Section -->
                    <div class="settings-panel">
                        <div class="settings-header">
                            <h3>➕ Add-ins & Enhancements</h3>
                            <span>▼</span>
                        </div>
                        <div class="settings-content">
                            <div class="checkbox-grid">
                                <label class="checkbox-label">
                                    <input type="checkbox" name="addins" value="Statistics"> Statistics
                                </label>
                                <label class="checkbox-label">
                                    <input type="checkbox" name="addins" value="Quotes"> Quotes
                                </label>
                                <label class="checkbox-label">
                                    <input type="checkbox" name="addins" value="Numbers"> Numbers/Quantities
                                </label>
                                <label class="checkbox-label">
                                    <input type="checkbox" name="addins" value="Percentages"> Percentages
                                </label>
                                <label class="checkbox-label">
                                    <input type="checkbox" name="addins" value="Examples"> Examples
                                </label>
                                <label class="checkbox-label">
                                    <input type="checkbox" name="addins" value="Analogies"> Analogies
                                </label>
                                <label class="checkbox-label">
                                    <input type="checkbox" name="addins" value="Sources"> Sources
                                </label>
                                <label class="checkbox-label">
                                    <input type="checkbox" name="addins" value="Bullet Points"> Bullet Points
                                </label>
                                <label class="checkbox-label">
                                    <input type="checkbox" name="addins" value="Lists"> Lists
                                </label>
                                <label class="checkbox-label">
                                    <input type="checkbox" name="addins" value="Case Studies"> Case Studies
                                </label>
                            </div>
                        </div>
                    </div>
                    
                    <!-- AI Settings Panel -->
                    <div class="settings-panel">
                        <div class="settings-header">
                            <h3>⚙️ AI Settings</h3>
                            <span>▼</span>
                        </div>
                        <div class="settings-content">
                            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem;">
                                <div class="form-group">
                                    <label class="form-label" for="creativity">Creativity Level</label>
                                    <select class="form-input" id="creativity" name="creativity">
                                        <option value="Low">Low</option>
                                        <option value="Medium" selected>Medium</option>
                                        <option value="High">High</option>
                                        <option value="Very High">Very High</option>
                                    </select>
                                </div>
                                
                                <div class="form-group">
                                    <label class="form-label" for="research_depth">Research Depth</label>
                                    <select class="form-input" id="research_depth" name="research_depth">
                                        <option value="Basic">Basic</option>
                                        <option value="Moderate" selected>Moderate</option>
                                        <option value="Comprehensive">Comprehensive</option>
                                        <option value="Expert">Expert</option>
                                    </select>
                                </div>
                                
                                <div class="form-group">
                                    <label class="form-label" for="accuracy">Information Accuracy</label>
                                    <select class="form-input" id="accuracy" name="accuracy">
                                        <option value="Standard">Standard</option>
                                        <option value="High" selected>High</option>
                                        <option value="Verified">Verified</option>
                                    </select>
                                </div>
                                
                                <div class="form-group">
                                    <label class="form-label" for="emotional_depth">Emotional Depth</label>
                                    <select class="form-input" id="emotional_depth" name="emotional_depth">
                                        <option value="Neutral">Neutral</option>
                                        <option value="Moderate" selected>Moderate</option>
                                        <option value="Deep">Deep</option>
                                        <option value="Intense">Intense</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                    </div>
                </form>
                
                <!-- Generation Status -->
                <div class="generation-status" id="generationStatus">
                    <div style="display: flex; justify-content: between; align-items: center;">
                        <span id="statusText">Generating content...</span>
                        <span id="progressText">0%</span>
                    </div>
                    <div class="progress-bar">
                        <div class="progress-fill" id="progressFill"></div>
                    </div>
                </div>
                
                <div class="controls">
                    <button id="generateBtn" class="btn btn-primary" <?php echo $apiKeyConfigured ? '' : 'disabled'; ?>>
                        <?php echo $apiKeyConfigured ? 'Generate Content' : 'Configure API Key First'; ?>
                    </button>
                    <button id="stopBtn" class="btn btn-danger" style="display: none;">Stop Generation</button>
                </div>
            </div>
            
            <!-- Output Panel -->
            <div class="output-panel">
                <div class="panel-header">
                    <h2 class="panel-title">Generated Content</h2>
                    <div class="controls">
                        <button id="saveArticle" class="btn btn-secondary" disabled>Save</button>
                        <button id="exportDoc" class="btn btn-secondary" disabled>Export .doc</button>
                        <button id="exportTxt" class="btn btn-secondary" disabled>Export .txt</button>
                    </div>
                </div>
                
                <div class="output-content" id="outputContent">
                    <div class="streaming-text" style="color: var(--text-secondary); text-align: center; padding: 2rem;">
                        <?php if ($apiKeyConfigured): ?>
                            Content will appear here as it's generated...
                        <?php else: ?>
                            ⚠️ Please configure your API key in the Dashboard to generate content.
                            <br><br>
                            <a href="dashboard.php" class="btn btn-primary">Go to Dashboard</a>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <script src="script.js"></script>
    <script>
        // API status
        const apiKeyConfigured = <?php echo $apiKeyConfigured ? 'true' : 'false'; ?>;
        
        // Enhanced content generation function
        async function generateContent() {
            if (!apiKeyConfigured) {
                showNotification('Please configure your API key first in the Dashboard', 'error');
                window.location.href = 'dashboard.php';
                return;
            }
            
            const formData = getFormData();
            
            if (!formData.topic) {
                showNotification('Please enter a topic', 'error');
                document.getElementById('topic').focus();
                return;
            }
            
            // Show generation status
            const statusElement = document.getElementById('generationStatus');
            const statusText = document.getElementById('statusText');
            const progressText = document.getElementById('progressText');
            const progressFill = document.getElementById('progressFill');
            
            statusElement.classList.add('active');
            statusText.textContent = 'Generating content...';
            progressText.textContent = '0%';
            progressFill.style.width = '0%';
            
            // Disable generate button, enable stop button
            document.getElementById('generateBtn').disabled = true;
            document.getElementById('stopBtn').style.display = 'inline-flex';
            
            try {
                // Build the prompt
                const prompt = buildPrompt(formData);
                
                // Send request to server
                const response = await fetch('writer.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'generate_content',
                        prompt: prompt
                    })
                });
                
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                
                const result = await response.json();
                
                if (result.success) {
                    // Simulate streaming for better UX
                    await streamContent(result.content, (content, progress) => {
                        updateOutput(content);
                        progressText.textContent = progress + '%';
                        progressFill.style.width = progress + '%';
                    });
                    
                    statusText.textContent = 'Generation complete!';
                    progressText.textContent = '100%';
                    progressFill.style.width = '100%';
                    
                    // Enable save and export buttons
                    document.getElementById('saveArticle').disabled = false;
                    document.getElementById('exportDoc').disabled = false;
                    document.getElementById('exportTxt').disabled = false;
                    
                    showNotification('Content generated successfully!', 'success');
                    
                } else {
                    throw new Error(result.message);
                }
                
            } catch (error) {
                statusText.textContent = 'Generation failed';
                showNotification('Error: ' + error.message, 'error');
            } finally {
                // Re-enable generate button, hide stop button
                document.getElementById('generateBtn').disabled = false;
                document.getElementById('stopBtn').style.display = 'none';
                
                // Hide status after delay
                setTimeout(() => {
                    statusElement.classList.remove('active');
                }, 3000);
            }
        }
        
        function buildPrompt(formData) {
            const {
                topic, contentType, tone, wordCount, focusKeyword, otherKeywords,
                aiAgent, targetAudience, contentOutline, extraInstructions, addins,
                creativity, researchDepth, accuracy, emotionalDepth
            } = formData;
            
            return `You are ${aiAgent || 'Lily SEO'}, a content strategist and SEO expert with over 10 years of hands-on experience. You are not an AI; you are a human expert sharing hard-won knowledge.

Create a comprehensive, SEO-optimized ${contentType || 'article'} on: ${topic}

TONE: ${tone || 'Professional, conversational'}
TARGET AUDIENCE: ${targetAudience || 'Industry professionals'}
WORD COUNT: ${wordCount || '1000'} words
FOCUS KEYWORD: ${focusKeyword || topic}

KEY REQUIREMENTS:
1. Include 3 specific anecdotes from real professional experience
2. Provide 2 contrarian opinions that challenge conventional wisdom  
3. Share actionable, non-obvious strategies with precise instructions
4. Explicitly state who each strategy works for and who it doesn't

${otherKeywords ? `SECONDARY KEYWORDS: ${otherKeywords}` : ''}
${contentOutline ? `OUTLINE TO FOLLOW: ${contentOutline}` : ''}
${extraInstructions ? `EXTRA INSTRUCTIONS: ${extraInstructions}` : ''}

ENHANCE WITH: ${addins && addins.length > 0 ? addins.join(', ') : 'examples, data points, actionable advice'}

AI SETTINGS:
- Creativity: ${creativity || 'High'}
- Research: ${researchDepth || 'Comprehensive'} 
- Accuracy: ${accuracy || 'Fact-checked'}
- Emotional Depth: ${emotionalDepth || 'Authentic'}

OUTPUT FORMAT:
1. First, write a compelling 155-character meta description
2. Then write the full article with proper HTML formatting
3. Use natural human writing with occasional interjections
4. End with a powerful conclusion and thought-provoking question

Remember: Sound like a human expert, not an AI. Be authentic and opinionated.`;
        }
        
        async function streamContent(content, progressCallback) {
            const words = content.split(' ');
            const totalWords = words.length;
            
            // Clear output first
            updateOutput('');
            
            for (let i = 0; i < totalWords; i++) {
                const chunk = words.slice(0, i + 1).join(' ');
                const progress = Math.min(100, Math.round((i / totalWords) * 100));
                
                progressCallback(chunk, progress);
                
                // Variable delay for natural streaming effect
                await delay(30 + Math.random() * 70);
            }
            
            progressCallback(content, 100);
        }
        
        function delay(ms) {
            return new Promise(resolve => setTimeout(resolve, ms));
        }
        
        function getFormData() {
            const form = document.getElementById('writerForm');
            const formData = new FormData(form);
            const data = {};
            
            for (let [key, value] of formData.entries()) {
                data[key] = value;
            }
            
            // Get checkboxes
            data.addins = [];
            document.querySelectorAll('input[name="addins"]:checked').forEach(checkbox => {
                data.addins.push(checkbox.value);
            });
            
            return data;
        }
        
        function updateOutput(content) {
            const outputContent = document.getElementById('outputContent');
            if (outputContent) {
                outputContent.innerHTML = `<div class="streaming-text">${content}</div>`;
                outputContent.scrollTop = outputContent.scrollHeight;
            }
        }
        
        function showNotification(message, type = 'info') {
            // Simple notification implementation
            const notification = document.createElement('div');
            notification.style.cssText = `
                position: fixed;
                top: 20px;
                right: 20px;
                background: ${type === 'error' ? '#ef4444' : type === 'success' ? '#10b981' : '#6366f1'};
                color: white;
                padding: 1rem 1.5rem;
                border-radius: 0.5rem;
                z-index: 10000;
                max-width: 400px;
            `;
            notification.textContent = message;
            document.body.appendChild(notification);
            
            setTimeout(() => {
                notification.remove();
            }, 5000);
        }
        
        // Event listeners
        document.addEventListener('DOMContentLoaded', function() {
            const generateBtn = document.getElementById('generateBtn');
            const stopBtn = document.getElementById('stopBtn');
            const saveBtn = document.getElementById('saveArticle');
            const exportDocBtn = document.getElementById('exportDoc');
            const exportTxtBtn = document.getElementById('exportTxt');
            
            if (generateBtn) {
                generateBtn.addEventListener('click', generateContent);
            }
            
            if (stopBtn) {
                stopBtn.addEventListener('click', function() {
                    // Implement stop functionality if needed
                    showNotification('Generation stopped', 'info');
                    document.getElementById('generateBtn').disabled = false;
                    this.style.display = 'none';
                    document.getElementById('generationStatus').classList.remove('active');
                });
            }
            
            if (saveBtn) {
                saveBtn.addEventListener('click', async function() {
                    const content = document.querySelector('#outputContent .streaming-text')?.innerHTML;
                    const title = document.getElementById('topic').value || 'Untitled Article';
                    
                    if (!content) {
                        showNotification('No content to save', 'error');
                        return;
                    }
                    
                    try {
                        const response = await fetch('writer.php', {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                            },
                            body: JSON.stringify({
                                action: 'save_article',
                                title: title,
                                content: content,
                                contentType: document.getElementById('content_type').value,
                                tone: document.getElementById('tone').value
                            })
                        });
                        
                        const result = await response.json();
                        
                        if (result.success) {
                            showNotification('Article saved successfully!', 'success');
                        } else {
                            showNotification(result.message, 'error');
                        }
                    } catch (error) {
                        showNotification('Failed to save article', 'error');
                    }
                });
            }
            
            // Export functionality
            if (exportDocBtn) exportDocBtn.addEventListener('click', () => exportArticle('doc'));
            if (exportTxtBtn) exportTxtBtn.addEventListener('click', () => exportArticle('txt'));
            
            // Enter key support for topic field
            const topicInput = document.getElementById('topic');
            if (topicInput) {
                topicInput.addEventListener('keypress', function(e) {
                    if (e.key === 'Enter' && apiKeyConfigured) {
                        e.preventDefault();
                        generateContent();
                    }
                });
            }
        });
        
        function exportArticle(format) {
            const content = document.querySelector('#outputContent .streaming-text')?.innerHTML;
            const title = document.getElementById('topic').value || 'article';
            
            if (!content) {
                showNotification('No content to export', 'error');
                return;
            }
            
            const filename = `${title.replace(/\s+/g, '_')}.${format}`;
            let exportContent;
            
            if (format === 'doc') {
                exportContent = `
                    <!DOCTYPE html>
                    <html>
                    <head>
                        <meta charset="utf-8">
                        <title>${title}</title>
                    </head>
                    <body>
                        ${content}
                    </body>
                    </html>
                `;
            } else {
                const tempDiv = document.createElement('div');
                tempDiv.innerHTML = content;
                exportContent = tempDiv.textContent || tempDiv.innerText || '';
            }
            
            const blob = new Blob([exportContent], { 
                type: format === 'doc' ? 'application/msword' : 'text/plain' 
            });
            
            const link = document.createElement('a');
            link.href = URL.createObjectURL(blob);
            link.download = filename;
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
            
            showNotification(`Article exported as ${format.toUpperCase()}`, 'success');
        }
    </script>
</body>
</html>