// script.js - Main Application JavaScript
document.addEventListener('DOMContentLoaded', function() {
    initializeApp();
});

function initializeApp() {
    // Theme management
    initializeTheme();
    
    // Navigation
    initializeNavigation();
    
    // Tab functionality for auth pages
    initializeTabs();
    
    // AI Content Generation
    if (document.getElementById('content-form')) {
        initializeContentGeneration();
    }
    
    // Payment functionality
    if (document.getElementById('donation-form')) {
        initializeDonations();
    }
    
    // Admin functionality
    if (document.getElementById('admin-panel')) {
        initializeAdminPanel();
    }
}

// Theme Management
function initializeTheme() {
    const themeToggle = document.getElementById('theme-toggle');
    const savedTheme = localStorage.getItem('theme') || 'light';
    
    document.body.classList.toggle('dark-theme', savedTheme === 'dark');
    updateThemeToggleText();
    
    if (themeToggle) {
        themeToggle.addEventListener('click', function() {
            document.body.classList.toggle('dark-theme');
            const newTheme = document.body.classList.contains('dark-theme') ? 'dark' : 'light';
            localStorage.setItem('theme', newTheme);
            updateThemeToggleText();
        });
    }
}

function updateThemeToggleText() {
    const themeToggle = document.getElementById('theme-toggle');
    if (themeToggle) {
        themeToggle.textContent = document.body.classList.contains('dark-theme') ? '☀️ Light Mode' : '🌙 Dark Mode';
    }
}

// Navigation
function initializeNavigation() {
    const navToggle = document.querySelector('.nav-toggle');
    const navMenu = document.querySelector('.nav-menu');
    
    if (navToggle && navMenu) {
        navToggle.addEventListener('click', function() {
            navMenu.classList.toggle('active');
            navToggle.classList.toggle('active');
        });
    }
}

// Tab Functionality
function initializeTabs() {
    const tabButtons = document.querySelectorAll('.tab-button');
    
    tabButtons.forEach(button => {
        button.addEventListener('click', function() {
            const targetTab = this.getAttribute('data-tab');
            
            // Update active tab button
            tabButtons.forEach(btn => btn.classList.remove('active'));
            this.classList.add('active');
            
            // Show target tab content
            document.querySelectorAll('.tab-content').forEach(content => {
                content.classList.remove('active');
            });
            document.getElementById(`${targetTab}-tab`).classList.add('active');
        });
    });
}

// AI Content Generation
function initializeContentGeneration() {
    const form = document.getElementById('content-form');
    const generateBtn = document.getElementById('generate-btn');
    const outputContent = document.getElementById('output-content');
    
    // Advanced settings toggle
    const settingsToggle = document.querySelector('.advanced-settings h3');
    if (settingsToggle) {
        settingsToggle.addEventListener('click', function() {
            const panel = this.nextElementSibling;
            panel.style.display = panel.style.display === 'none' ? 'block' : 'none';
            this.querySelector('.toggle-icon').textContent = 
                panel.style.display === 'none' ? '▼' : '▲';
        });
    }
    
    form.addEventListener('submit', async function(e) {
        e.preventDefault();
        await generateContent();
    });
    
    // Output controls
    document.getElementById('copy-output').addEventListener('click', copyOutput);
    document.getElementById('clear-output').addEventListener('click', clearOutput);
}

async function generateContent() {
    const generateBtn = document.getElementById('generate-btn');
    const outputContent = document.getElementById('output-content');
    const formData = new FormData(document.getElementById('content-form'));
    
    generateBtn.disabled = true;
    generateBtn.textContent = 'Generating...';
    outputContent.innerHTML = '<div class="loading">Generating content...<div class="spinner"></div></div>';
    
    try {
        const prompt = buildAIPrompt(formData);
        const response = await callGeminiAPI(prompt);
        
        // Stream the response
        await streamResponse(response, outputContent);
        
    } catch (error) {
        outputContent.innerHTML = `<div class="error">Error: ${error.message}</div>`;
    } finally {
        generateBtn.disabled = false;
        generateBtn.textContent = 'Generate Content';
    }
}

function buildAIPrompt(formData) {
    const topic = formData.get('topic');
    const tone = formData.get('tone');
    const wordCount = formData.get('word_count');
    const agent = formData.get('ai_agent');
    
    let basePrompt = agent === 'lily-seo' ? window.AI_PROMPTS.LILY_SEO : window.AI_PROMPTS.DEFAULT;
    
    return basePrompt.replace('[Insert Exact Article Topic Here]', topic)
                     .replace('[Industry Expert]', tone)
                     + `\n\nWord Count: ${wordCount}\nTone: ${tone}`;
}

async function callGeminiAPI(prompt) {
    const apiUrl = window.APP_CONFIG.AI_API.endpoint + '?key=' + window.APP_CONFIG.AI_API.key;
    
    const response = await fetch(apiUrl, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            contents: [{
                parts: [{
                    text: prompt
                }]
            }]
        })
    });
    
    if (!response.ok) {
        throw new Error(`API Error: ${response.status}`);
    }
    
    return response.json();
}

async function streamResponse(apiResponse, outputElement) {
    outputElement.innerHTML = '';
    
    if (apiResponse.candidates && apiResponse.candidates[0].content.parts[0].text) {
        const text = apiResponse.candidates[0].content.parts[0].text;
        let index = 0;
        
        function typeWriter() {
            if (index < text.length) {
                outputElement.innerHTML += text.charAt(index);
                index++;
                outputElement.scrollTop = outputElement.scrollHeight;
                setTimeout(typeWriter, 20);
            }
        }
        
        typeWriter();
    } else {
        outputElement.innerHTML = '<div class="error">No content generated. Please try again.</div>';
    }
}

function copyOutput() {
    const outputText = document.getElementById('output-content').textContent;
    navigator.clipboard.writeText(outputText).then(() => {
        showNotification('Content copied to clipboard!');
    });
}

function clearOutput() {
    document.getElementById('output-content').innerHTML = 
        '<p class="placeholder">Your generated content will appear here...</p>';
}

// Donation System
function initializeDonations() {
    const donationForm = document.getElementById('donation-form');
    const amountButtons = document.querySelectorAll('.amount-btn');
    
    amountButtons.forEach(btn => {
        btn.addEventListener('click', function() {
            amountButtons.forEach(b => b.classList.remove('active'));
            this.classList.add('active');
            document.getElementById('custom-amount').value = this.dataset.amount;
        });
    });
    
    donationForm.addEventListener('submit', processDonation);
}

async function processDonation(e) {
    e.preventDefault();
    
    const amount = document.getElementById('custom-amount').value;
    const email = document.getElementById('donor-email').value;
    
    // Show processing state
    const submitBtn = document.querySelector('#donation-form button[type="submit"]');
    submitBtn.disabled = true;
    submitBtn.textContent = 'Processing...';
    
    try {
        // Simulate payment processing
        await simulatePayment(amount, email);
        
        // Show success message
        showDonationSuccess(amount);
        
    } catch (error) {
        showNotification('Payment failed: ' + error.message, 'error');
    } finally {
        submitBtn.disabled = false;
        submitBtn.textContent = 'Donate Now';
    }
}

function simulatePayment(amount, email) {
    return new Promise((resolve, reject) => {
        setTimeout(() => {
            // Simulate 90% success rate
            if (Math.random() > 0.1) {
                resolve({ transactionId: 'TX_' + Date.now(), amount: amount });
            } else {
                reject(new Error('Payment processing failed'));
            }
        }, 2000);
    });
}

function showDonationSuccess(amount) {
    const modal = document.createElement('div');
    modal.className = 'donation-success-modal';
    modal.innerHTML = `
        <div class="modal-content">
            <span class="close-modal">&times;</span>
            <h3>Thank You for Your Generous Support! 💝</h3>
            <p>Your donation of $${amount} has been received successfully.</p>
            <div class="success-message">
                ${generateThankYouMessage(amount)}
            </div>
            <button class="btn-primary" onclick="this.closest('.donation-success-modal').remove()">Close</button>
        </div>
    `;
    
    document.body.appendChild(modal);
    
    // Close modal functionality
    modal.querySelector('.close-modal').addEventListener('click', function() {
        modal.remove();
    });
}

function generateThankYouMessage(amount) {
    const messages = [
        `Your support means the world to us! With your contribution of $${amount}, we can continue to improve the platform and add even more powerful features.`,
        `We're incredibly grateful for your $${amount} donation. Supporters like you make it possible to keep this service running and accessible to everyone.`,
        `Thank you for believing in our mission! Your $${amount} donation helps us maintain the infrastructure and develop new innovations for the platform.`
    ];
    
    return messages[Math.floor(Math.random() * messages.length)];
}

// Utility Functions
function showNotification(message, type = 'success') {
    const notification = document.createElement('div');
    notification.className = `notification ${type}`;
    notification.textContent = message;
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.remove();
    }, 3000);
}

// File Management Functions
async function saveContent(title, content) {
    const filename = `article_${Date.now()}.json`;
    const fileData = {
        title: title,
        content: content,
        created: new Date().toISOString(),
        author: document.querySelector('.user-menu span').textContent.replace('Welcome, ', '')
    };
    
    // In a real implementation, this would save to the server
    localStorage.setItem(filename, JSON.stringify(fileData));
    showNotification('Content saved successfully!');
}

function loadSavedArticles() {
    // Load from localStorage (in real app, would be server-side)
    const articles = [];
    for (let i = 0; i < localStorage.length; i++) {
        const key = localStorage.key(i);
        if (key.startsWith('article_')) {
            articles.push(JSON.parse(localStorage.getItem(key)));
        }
    }
    return articles;
}