<?php
require_once 'config.php';
require_once 'auth.php';

if (!$auth->isLoggedIn()) {
    header('Location: membership.php');
    exit;
}

$user = $_SESSION['user'];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>AI Writer Dashboard - Uncensored AI Writer</title>
    <link rel="stylesheet" href="styles.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        /* Enhanced Writer Page Styles */
        .writer-container {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
        }

        .writer-layout {
            display: grid;
            grid-template-columns: 280px 1fr;
            gap: 0;
            min-height: calc(100vh - 70px);
        }

        .sidebar {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            padding: 2rem 1.5rem;
            border-right: 1px solid rgba(255, 255, 255, 0.2);
            box-shadow: 5px 0 15px rgba(0, 0, 0, 0.1);
            overflow-y: auto;
        }

        .dark-theme .sidebar {
            background: rgba(45, 55, 72, 0.95);
            border-right: 1px solid rgba(255, 255, 255, 0.1);
        }

        .main-content {
            background: #f8fafc;
            padding: 2rem;
            overflow-y: auto;
        }

        .dark-theme .main-content {
            background: #1a202c;
        }

        .content-generation {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 2rem;
            height: 100%;
            max-height: calc(100vh - 140px);
        }

        @media (max-width: 1200px) {
            .content-generation {
                grid-template-columns: 1fr;
                gap: 1.5rem;
            }
        }

        .form-section, .output-section {
            background: white;
            border-radius: 20px;
            padding: 2rem;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
            border: 1px solid rgba(255, 255, 255, 0.2);
            backdrop-filter: blur(10px);
        }

        .dark-theme .form-section,
        .dark-theme .output-section {
            background: rgba(45, 55, 72, 0.9);
            border: 1px solid rgba(255, 255, 255, 0.1);
        }

        .section-header {
            display: flex;
            justify-content: between;
            align-items: center;
            margin-bottom: 1.5rem;
            padding-bottom: 1rem;
            border-bottom: 2px solid #e2e8f0;
        }

        .dark-theme .section-header {
            border-bottom-color: #4a5568;
        }

        .section-header h2 {
            margin: 0;
            color: #2d3748;
            font-size: 1.5rem;
            font-weight: 700;
        }

        .dark-theme .section-header h2 {
            color: #e2e8f0;
        }

        /* Form Styles */
        .form-group {
            margin-bottom: 1.5rem;
        }

        .form-group label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 600;
            color: #4a5568;
            font-size: 0.9rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .dark-theme .form-group label {
            color: #a0aec0;
        }

        .form-group input,
        .form-group select,
        .form-group textarea {
            width: 100%;
            padding: 0.75rem 1rem;
            border: 2px solid #e2e8f0;
            border-radius: 12px;
            font-size: 1rem;
            transition: all 0.3s ease;
            background: white;
        }

        .dark-theme .form-group input,
        .dark-theme .form-group select,
        .dark-theme .form-group textarea {
            background: #2d3748;
            border-color: #4a5568;
            color: #e2e8f0;
        }

        .form-group input:focus,
        .form-group select:focus,
        .form-group textarea:focus {
            outline: none;
            border-color: #667eea;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }

        .form-row {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 1rem;
        }

        @media (max-width: 768px) {
            .form-row {
                grid-template-columns: 1fr;
            }
        }

        /* Advanced Settings */
        .advanced-settings {
            background: #f7fafc;
            border-radius: 12px;
            padding: 1.5rem;
            margin: 2rem 0;
            border: 1px solid #e2e8f0;
        }

        .dark-theme .advanced-settings {
            background: #2d3748;
            border-color: #4a5568;
        }

        .settings-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            cursor: pointer;
            margin-bottom: 1rem;
        }

        .settings-header h3 {
            margin: 0;
            color: #2d3748;
            font-size: 1.1rem;
        }

        .dark-theme .settings-header h3 {
            color: #e2e8f0;
        }

        .toggle-icon {
            font-size: 1.2rem;
            transition: transform 0.3s ease;
        }

        .settings-panel {
            display: none;
        }

        .settings-panel.active {
            display: block;
            animation: slideDown 0.3s ease;
        }

        @keyframes slideDown {
            from {
                opacity: 0;
                transform: translateY(-10px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .settings-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1.5rem;
            margin-bottom: 1.5rem;
        }

        .setting-item {
            margin-bottom: 1rem;
        }

        .setting-item label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 600;
            color: #4a5568;
        }

        .dark-theme .setting-item label {
            color: #a0aec0;
        }

        .range-container {
            display: flex;
            align-items: center;
            gap: 1rem;
        }

        .range-value {
            min-width: 40px;
            text-align: center;
            font-weight: 600;
            color: #667eea;
        }

        input[type="range"] {
            flex: 1;
            height: 6px;
            border-radius: 3px;
            background: #e2e8f0;
            outline: none;
        }

        .dark-theme input[type="range"] {
            background: #4a5568;
        }

        input[type="range"]::-webkit-slider-thumb {
            appearance: none;
            width: 20px;
            height: 20px;
            border-radius: 50%;
            background: #667eea;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        input[type="range"]::-webkit-slider-thumb:hover {
            background: #5a6fd8;
            transform: scale(1.1);
        }

        .add-ins {
            margin-top: 1.5rem;
        }

        .add-ins h4 {
            margin-bottom: 1rem;
            color: #2d3748;
        }

        .dark-theme .add-ins h4 {
            color: #e2e8f0;
        }

        .checkbox-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 0.75rem;
        }

        .checkbox-grid label {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            cursor: pointer;
            padding: 0.5rem;
            border-radius: 8px;
            transition: background 0.3s ease;
        }

        .checkbox-grid label:hover {
            background: rgba(102, 126, 234, 0.1);
        }

        .checkbox-grid input[type="checkbox"] {
            width: 18px;
            height: 18px;
            border-radius: 4px;
            border: 2px solid #cbd5e0;
            appearance: none;
            cursor: pointer;
            position: relative;
            transition: all 0.3s ease;
        }

        .checkbox-grid input[type="checkbox"]:checked {
            background: #667eea;
            border-color: #667eea;
        }

        .checkbox-grid input[type="checkbox"]:checked::after {
            content: '✓';
            position: absolute;
            color: white;
            font-size: 12px;
            font-weight: bold;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
        }

        /* Output Section */
        .output-controls {
            display: flex;
            gap: 0.5rem;
            margin-bottom: 1rem;
            flex-wrap: wrap;
        }

        .output-content {
            min-height: 400px;
            max-height: 500px;
            overflow-y: auto;
            background: #f7fafc;
            border: 2px solid #e2e8f0;
            border-radius: 12px;
            padding: 1.5rem;
            font-family: 'Georgia', serif;
            line-height: 1.8;
            color: #2d3748;
        }

        .dark-theme .output-content {
            background: #2d3748;
            border-color: #4a5568;
            color: #e2e8f0;
        }

        .output-content h1, .output-content h2, .output-content h3 {
            color: #2d3748;
            margin: 1.5rem 0 1rem 0;
        }

        .dark-theme .output-content h1,
        .dark-theme .output-content h2,
        .dark-theme .output-content h3 {
            color: #e2e8f0;
        }

        .output-content p {
            margin-bottom: 1rem;
        }

        .output-content ul, .output-content ol {
            margin: 1rem 0;
            padding-left: 2rem;
        }

        .output-content blockquote {
            border-left: 4px solid #667eea;
            padding-left: 1rem;
            margin: 1rem 0;
            font-style: italic;
            color: #4a5568;
        }

        .dark-theme .output-content blockquote {
            color: #a0aec0;
        }

        .placeholder {
            color: #a0aec0;
            font-style: italic;
            text-align: center;
            margin: 2rem 0;
        }

        /* Loading Animation */
        .loading {
            text-align: center;
            padding: 2rem;
            color: #667eea;
        }

        .spinner {
            border: 3px solid #f3f3f3;
            border-top: 3px solid #667eea;
            border-radius: 50%;
            width: 40px;
            height: 40px;
            animation: spin 1s linear infinite;
            margin: 1rem auto;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        /* Buttons */
        .btn-primary {
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
            border: none;
            padding: 1rem 2rem;
            border-radius: 12px;
            font-size: 1.1rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
        }

        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(102, 126, 234, 0.4);
        }

        .btn-primary:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none;
        }

        .btn-secondary {
            background: transparent;
            color: #667eea;
            border: 2px solid #667eea;
            padding: 0.5rem 1rem;
            border-radius: 8px;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .btn-secondary:hover {
            background: #667eea;
            color: white;
        }

        .btn-full {
            width: 100%;
        }

        .btn-large {
            padding: 1.25rem 2.5rem;
            font-size: 1.2rem;
        }

        /* Sidebar Styles */
        .sidebar-section {
            margin-bottom: 2.5rem;
        }

        .sidebar-section h3 {
            color: #2d3748;
            font-size: 1.1rem;
            margin-bottom: 1rem;
            padding-bottom: 0.5rem;
            border-bottom: 2px solid #e2e8f0;
        }

        .dark-theme .sidebar-section h3 {
            color: #e2e8f0;
            border-bottom-color: #4a5568;
        }

        .sidebar-btn {
            width: 100%;
            padding: 1rem;
            margin-bottom: 0.75rem;
            background: white;
            border: 2px solid #e2e8f0;
            border-radius: 12px;
            cursor: pointer;
            transition: all 0.3s ease;
            text-align: left;
            font-weight: 600;
            color: #4a5568;
            display: flex;
            align-items: center;
            gap: 0.5rem;
        }

        .dark-theme .sidebar-btn {
            background: #2d3748;
            border-color: #4a5568;
            color: #e2e8f0;
        }

        .sidebar-btn:hover {
            border-color: #667eea;
            color: #667eea;
            transform: translateX(5px);
        }

        .sidebar-btn i {
            font-size: 1.1rem;
        }

        .saved-item, .agent-item {
            padding: 0.75rem;
            margin-bottom: 0.5rem;
            background: #f7fafc;
            border-radius: 8px;
            cursor: pointer;
            transition: all 0.3s ease;
            border-left: 3px solid #667eea;
        }

        .dark-theme .saved-item,
        .dark-theme .agent-item {
            background: #2d3748;
        }

        .saved-item:hover, .agent-item:hover {
            background: #edf2f7;
            transform: translateX(3px);
        }

        .dark-theme .saved-item:hover,
        .dark-theme .agent-item:hover {
            background: #4a5568;
        }

        .saved-item h4, .agent-item h4 {
            margin: 0 0 0.25rem 0;
            font-size: 0.9rem;
            color: #2d3748;
        }

        .dark-theme .saved-item h4,
        .dark-theme .agent-item h4 {
            color: #e2e8f0;
        }

        .saved-item p, .agent-item p {
            margin: 0;
            font-size: 0.8rem;
            color: #718096;
        }

        /* Modals */
        .modal {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.7);
            backdrop-filter: blur(5px);
        }

        .modal-content {
            background: white;
            margin: 5% auto;
            padding: 2.5rem;
            border-radius: 20px;
            width: 90%;
            max-width: 500px;
            position: relative;
            box-shadow: 0 20px 40px rgba(0, 0, 0, 0.3);
            animation: modalSlideIn 0.3s ease;
        }

        .dark-theme .modal-content {
            background: #2d3748;
        }

        @keyframes modalSlideIn {
            from {
                opacity: 0;
                transform: translateY(-50px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        .close-modal {
            position: absolute;
            right: 1.5rem;
            top: 1.5rem;
            font-size: 1.5rem;
            cursor: pointer;
            color: #a0aec0;
            transition: color 0.3s ease;
        }

        .close-modal:hover {
            color: #667eea;
        }

        /* Responsive Design */
        @media (max-width: 1024px) {
            .writer-layout {
                grid-template-columns: 1fr;
            }
            
            .sidebar {
                order: 2;
                border-right: none;
                border-top: 1px solid #e2e8f0;
            }
            
            .main-content {
                order: 1;
            }
        }

        @media (max-width: 768px) {
            .content-generation {
                grid-template-columns: 1fr;
                gap: 1.5rem;
            }
            
            .form-section, .output-section {
                padding: 1.5rem;
            }
            
            .settings-grid {
                grid-template-columns: 1fr;
            }
        }

        /* User Menu */
        .user-menu {
            display: flex;
            align-items: center;
            gap: 1rem;
        }

        .user-menu span {
            color: white;
            font-weight: 600;
        }

        .admin-link {
            background: rgba(255, 255, 255, 0.2);
            padding: 0.5rem 1rem;
            border-radius: 8px;
            color: white;
            text-decoration: none;
            transition: background 0.3s ease;
        }

        .admin-link:hover {
            background: rgba(255, 255, 255, 0.3);
        }

        .logout-link {
            color: white;
            text-decoration: none;
            transition: opacity 0.3s ease;
        }

        .logout-link:hover {
            opacity: 0.8;
        }

        /* Real-time typing effect */
        .typing-cursor {
            display: inline-block;
            width: 2px;
            height: 1.2em;
            background: #667eea;
            margin-left: 2px;
            animation: blink 1s infinite;
        }

        @keyframes blink {
            0%, 50% { opacity: 1; }
            51%, 100% { opacity: 0; }
        }

        /* Error states */
        .error {
            background: #fed7d7;
            color: #c53030;
            padding: 1rem;
            border-radius: 8px;
            margin: 1rem 0;
            border-left: 4px solid #f56565;
        }

        .dark-theme .error {
            background: #742a2a;
            color: #fc8181;
        }

        .success {
            background: #c6f6d5;
            color: #276749;
            padding: 1rem;
            border-radius: 8px;
            margin: 1rem 0;
            border-left: 4px solid #48bb78;
        }

        .dark-theme .success {
            background: #22543d;
            color: #68d391;
        }
    </style>
</head>
<body class="writer-page">
    <header class="header">
        <nav class="nav">
            <div class="nav-brand">
                <h1><i class="fas fa-feather-alt"></i> Uncensored AI Writer</h1>
            </div>
            <ul class="nav-menu">
                <li><a href="#" id="theme-toggle"><i class="fas fa-moon"></i> Dark Mode</a></li>
                <li><a href="donate.php"><i class="fas fa-heart"></i> Support Developer</a></li>
                <li class="user-menu">
                    <span><i class="fas fa-user"></i> Welcome, <?php echo htmlspecialchars($user['username']); ?></span>
                    <?php if ($auth->isAdmin()): ?>
                        <a href="admin/admin.php" class="admin-link"><i class="fas fa-cog"></i> Admin Panel</a>
                    <?php endif; ?>
                    <a href="logout.php" class="logout-link"><i class="fas fa-sign-out-alt"></i> Logout</a>
                </li>
            </ul>
            <div class="nav-toggle">
                <span></span>
                <span></span>
                <span></span>
            </div>
        </nav>
    </header>

    <main class="writer-container">
        <div class="writer-layout">
            <!-- Sidebar -->
            <aside class="sidebar">
                <div class="sidebar-section">
                    <h3><i class="fas fa-bolt"></i> Quick Actions</h3>
                    <button class="sidebar-btn" id="new-content">
                        <i class="fas fa-plus"></i> New Content
                    </button>
                    <button class="sidebar-btn" id="save-content">
                        <i class="fas fa-save"></i> Save Article
                    </button>
                    <button class="sidebar-btn" id="export-doc">
                        <i class="fas fa-file-word"></i> Export DOC
                    </button>
                    <button class="sidebar-btn" id="export-txt">
                        <i class="fas fa-file-alt"></i> Export TXT
                    </button>
                </div>

                <div class="sidebar-section">
                    <h3><i class="fas fa-history"></i> Recent Articles</h3>
                    <div id="saved-articles-list">
                        <div class="saved-item">
                            <h4>Getting Started Guide</h4>
                            <p>Yesterday, 845 words</p>
                        </div>
                        <div class="saved-item">
                            <h4>Advanced SEO Techniques</h4>
                            <p>2 days ago, 1,234 words</p>
                        </div>
                    </div>
                </div>

                <div class="sidebar-section">
                    <h3><i class="fas fa-robot"></i> AI Agents</h3>
                    <button class="sidebar-btn" id="create-agent">
                        <i class="fas fa-plus-circle"></i> Create New Agent
                    </button>
                    <div id="ai-agents-list">
                        <div class="agent-item">
                            <h4>Lily SEO</h4>
                            <p>SEO Content Specialist</p>
                        </div>
                        <div class="agent-item">
                            <h4>Tech Writer Pro</h4>
                            <p>Technical Content Expert</p>
                        </div>
                        <div class="agent-item">
                            <h4>Creative Storyteller</h4>
                            <p>Narrative Content Creator</p>
                        </div>
                    </div>
                </div>

                <div class="sidebar-section">
                    <h3><i class="fas fa-chart-line"></i> Statistics</h3>
                    <div class="stats-info">
                        <div class="stat-item">
                            <span class="stat-label">Articles Generated:</span>
                            <span class="stat-value">47</span>
                        </div>
                        <div class="stat-item">
                            <span class="stat-label">Total Words:</span>
                            <span class="stat-value">28,451</span>
                        </div>
                        <div class="stat-item">
                            <span class="stat-label">AI Detection Rate:</span>
                            <span class="stat-value success">0.3%</span>
                        </div>
                    </div>
                </div>
            </aside>

            <!-- Main Content Area -->
            <div class="main-content">
                <div class="content-generation">
                    <!-- Content Form -->
                    <div class="form-section">
                        <div class="section-header">
                            <h2><i class="fas fa-magic"></i> Content Generation</h2>
                        </div>
                        <form id="content-form">
                            <div class="form-row">
                                <div class="form-group">
                                    <label for="topic"><i class="fas fa-lightbulb"></i> Topic/Idea *</label>
                                    <input type="text" id="topic" name="topic" required placeholder="Enter your content topic or idea...">
                                </div>
                                <div class="form-group">
                                    <label for="content-type"><i class="fas fa-file-alt"></i> Content Type</label>
                                    <select id="content-type" name="content_type">
                                        <option value="Article">📝 Article</option>
                                        <option value="Blog Post">✍️ Blog Post</option>
                                        <option value="Listicle">📋 Listicle</option>
                                        <option value="News Post">📰 News Post</option>
                                        <option value="Product Description">🛍️ Product Description</option>
                                        <option value="Social Media Post">💬 Social Media Post</option>
                                        <option value="Email Newsletter">📧 Email Newsletter</option>
                                        <option value="Press Release">📢 Press Release</option>
                                        <option value="Case Study">🔬 Case Study</option>
                                        <option value="White Paper">📄 White Paper</option>
                                        <option value="How-To Guide">🔧 How-To Guide</option>
                                        <option value="Tutorial">🎓 Tutorial</option>
                                        <option value="FAQs">❓ FAQs</option>
                                        <option value="Landing Page">🌐 Landing Page</option>
                                        <option value="About Us Page">👥 About Us Page</option>
                                        <option value="Mission Statement">🎯 Mission Statement</option>
                                        <option value="Video Script">🎥 Video Script</option>
                                        <option value="Podcast Outline">🎙️ Podcast Outline</option>
                                        <option value="Interview Questions">💼 Interview Questions</option>
                                        <option value="Book Summary">📚 Book Summary</option>
                                        <option value="Research Paper">🔍 Research Paper</option>
                                        <option value="Business Proposal">💼 Business Proposal</option>
                                        <option value="Meeting Agendas">📅 Meeting Agendas</option>
                                        <option value="Job Descriptions">👔 Job Descriptions</option>
                                        <option value="Resume Bullet Points">📄 Resume Bullet Points</option>
                                        <option value="Cover Letters">💌 Cover Letters</option>
                                        <option value="Advertising Copy">📢 Advertising Copy</option>
                                        <option value="Meta Descriptions">🔍 Meta Descriptions</option>
                                        <option value="Instagram Captions">📸 Instagram Captions</option>
                                        <option value="Twitter Threads">🐦 Twitter Threads</option>
                                    </select>
                                </div>
                            </div>

                            <div class="form-row">
                                <div class="form-group">
                                    <label for="tone"><i class="fas fa-volume-up"></i> Tone/Voice</label>
                                    <select id="tone" name="tone">
                                        <option value="Professional">💼 Professional</option>
                                        <option value="Casual">😊 Casual</option>
                                        <option value="Friendly">🤝 Friendly</option>
                                        <option value="Formal">🎩 Formal</option>
                                        <option value="Authoritative">👑 Authoritative</option>
                                        <option value="Conversational">💬 Conversational</option>
                                        <option value="Humorous">😄 Humorous</option>
                                        <option value="Witty">🎭 Witty</option>
                                        <option value="Sarcastic">😏 Sarcastic</option>
                                        <option value="Empathetic">❤️ Empathetic</option>
                                        <option value="Persuasive">🎯 Persuasive</option>
                                        <option value="Inspirational">✨ Inspirational</option>
                                        <option value="Confident">💪 Confident</option>
                                        <option value="Neutral">😐 Neutral</option>
                                        <option value="Academic">🎓 Academic</option>
                                        <option value="Technical">🔧 Technical</option>
                                        <option value="Storytelling">📖 Storytelling</option>
                                        <option value="Dramatic">🎬 Dramatic</option>
                                        <option value="Optimistic">🌈 Optimistic</option>
                                        <option value="Urgent">⚡ Urgent</option>
                                        <option value="Calming">🌊 Calming</option>
                                        <option value="Corporate">🏢 Corporate</option>
                                        <option value="Salesy">💰 Salesy</option>
                                        <option value="Educational">📚 Educational</option>
                                        <option value="Youthful">🎉 Youthful</option>
                                        <option value="Sophisticated">🍷 Sophisticated</option>
                                        <option value="Bold">🔥 Bold</option>
                                        <option value="Conservative">📊 Conservative</option>
                                        <option value="Creative">🎨 Creative</option>
                                        <option value="Minimalist">⚪ Minimalist</option>
                                    </select>
                                </div>
                                <div class="form-group">
                                    <label for="word-count"><i class="fas fa-ruler"></i> Word Count</label>
                                    <input type="number" id="word-count" name="word_count" value="1000" min="100" max="10000" step="50">
                                </div>
                            </div>

                            <div class="form-group">
                                <label for="keywords"><i class="fas fa-key"></i> Keywords (comma separated)</label>
                                <textarea id="keywords" name="keywords" placeholder="primary keyword, secondary keyword, supporting terms..." rows="2"></textarea>
                            </div>

                            <div class="form-group">
                                <label for="ai-agent"><i class="fas fa-robot"></i> AI Writer Agent</label>
                                <select id="ai-agent" name="ai_agent">
                                    <option value="lily-seo">👩‍💼 Lily SEO - Content Strategist</option>
                                    <option value="tech-expert">👨‍💻 Tech Expert - Technical Writer</option>
                                    <option value="creative-writer">👩‍🎨 Creative Writer - Storyteller</option>
                                    <option value="business-analyst">👨‍💼 Business Analyst - Professional</option>
                                    <option value="marketing-guru">👩‍💼 Marketing Guru - Persuasive</option>
                                </select>
                            </div>

                            <div class="form-group">
                                <label for="target-audience"><i class="fas fa-users"></i> Target Audience</label>
                                <input type="text" id="target-audience" name="target_audience" placeholder="e.g., Small business owners, Tech enthusiasts...">
                            </div>

                            <div class="form-group">
                                <label for="content-outline"><i class="fas fa-list-ol"></i> Content Outline (Optional)</label>
                                <textarea id="content-outline" name="content_outline" placeholder="Brief outline or key points to cover..." rows="3"></textarea>
                            </div>

                            <div class="form-group">
                                <label for="extra-instructions"><i class="fas fa-sticky-note"></i> Extra Instructions</label>
                                <textarea id="extra-instructions" name="extra_instructions" placeholder="Any specific requirements or special instructions..." rows="3"></textarea>
                            </div>

                            <!-- Advanced AI Settings -->
                            <div class="advanced-settings">
                                <div class="settings-header">
                                    <h3><i class="fas fa-cogs"></i> Advanced AI Settings</h3>
                                    <span class="toggle-icon">▼</span>
                                </div>
                                <div class="settings-panel">
                                    <div class="settings-grid">
                                        <!-- 33 Advanced Settings as before -->
                                        <div class="setting-item">
                                            <label for="creativity">Creativity Level: <span class="range-value" id="creativity-value">75%</span></label>
                                            <div class="range-container">
                                                <input type="range" id="creativity" name="creativity" min="0" max="100" value="75">
                                            </div>
                                        </div>
                                        <!-- ... (all 33 settings from previous code) ... -->
                                    </div>

                                    <div class="add-ins">
                                        <h4><i class="fas fa-plus-circle"></i> Content Add-ins</h4>
                                        <div class="checkbox-grid">
                                            <label><input type="checkbox" name="addins[]" value="statistics"> 📊 Statistics</label>
                                            <label><input type="checkbox" name="addins[]" value="quotes"> 💬 Quotes</label>
                                            <label><input type="checkbox" name="addins[]" value="numbers" checked> 🔢 Numbers/Quantities</label>
                                            <label><input type="checkbox" name="addins[]" value="percentages"> 📈 Percentages</label>
                                            <label><input type="checkbox" name="addins[]" value="examples"> 💡 Examples</label>
                                            <label><input type="checkbox" name="addins[]" value="analogies"> 🔄 Analogies</label>
                                            <label><input type="checkbox" name="addins[]" value="sources"> 📚 Sources</label>
                                            <label><input type="checkbox" name="addins[]" value="bullet-points" checked> • Bullet Points</label>
                                            <label><input type="checkbox" name="addins[]" value="lists"> 📋 Lists</label>
                                            <label><input type="checkbox" name="addins[]" value="case-studies"> 🔬 Case Studies</label>
                                            <label><input type="checkbox" name="addins[]" value="comparisons"> ⚖️ Comparisons</label>
                                            <label><input type="checkbox" name="addins[]" value="definitions"> 📖 Definitions</label>
                                        </div>
                                    </div>

                                    <div class="advanced-options">
                                        <h4><i class="fas fa-sliders-h"></i> Advanced Options</h4>
                                        <div class="checkbox-grid">
                                            <label><input type="checkbox" name="options[]" value="unrestricted-web" checked> 🌐 Unrestricted Web Access</label>
                                            <label><input type="checkbox" name="options[]" value="real-time-info"> ⏰ Real-time Information</label>
                                            <label><input type="checkbox" name="options[]" value="multilingual"> 🈁 Multilingual Support</label>
                                            <label><input type="checkbox" name="options[]" value="technical-knowledge"> 🔧 Technical Knowledge</label>
                                            <label><input type="checkbox" name="options[]" value="creative-writing"> 🎨 Creative Writing</label>
                                            <label><input type="checkbox" name="options[]" value="analytical-skills"> 📊 Analytical Skills</label>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <button type="submit" class="btn-primary btn-large btn-full" id="generate-btn">
                                <i class="fas fa-bolt"></i> Generate Human-Like Content
                            </button>
                        </form>
                    </div>

                    <!-- Output Area -->
                    <div class="output-section">
                        <div class="section-header">
                            <h2><i class="fas fa-file-alt"></i> Generated Content</h2>
                            <div class="output-controls">
                                <button id="copy-output" class="btn-secondary" title="Copy to Clipboard">
                                    <i class="fas fa-copy"></i> Copy
                                </button>
                                <button id="clear-output" class="btn-secondary" title="Clear Output">
                                    <i class="fas fa-trash"></i> Clear
                                </button>
                                <button id="format-output" class="btn-secondary" title="Format Text">
                                    <i class="fas fa-paragraph"></i> Format
                                </button>
                            </div>
                        </div>
                        <div id="output-content" class="output-content">
                            <p class="placeholder">Your AI-generated content will appear here with real-time streaming from Quillfy AI...</p>
                        </div>
                        <div class="output-stats" id="output-stats" style="display: none; margin-top: 1rem; padding: 1rem; background: #f7fafc; border-radius: 8px;">
                            <div class="stats-row">
                                <span class="stat"><strong>Words:</strong> <span id="word-count-stat">0</span></span>
                                <span class="stat"><strong>Characters:</strong> <span id="char-count-stat">0</span></span>
                                <span class="stat"><strong>Reading Time:</strong> <span id="reading-time-stat">0 min</span></span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </main>

    <!-- AI Agent Creation Modal -->
    <div id="agent-modal" class="modal">
        <div class="modal-content">
            <span class="close-modal">&times;</span>
            <h3><i class="fas fa-robot"></i> Create New AI Agent</h3>
            <form id="agent-form">
                <div class="form-group">
                    <label for="agent-name">Agent Name</label>
                    <input type="text" id="agent-name" required placeholder="e.g., Marketing Maverick">
                </div>
                <div class="form-group">
                    <label for="agent-specialty">Specialty</label>
                    <select id="agent-specialty" required>
                        <option value="">Select a specialty...</option>
                        <option value="seo">SEO Content</option>
                        <option value="technical">Technical Writing</option>
                        <option value="creative">Creative Writing</option>
                        <option value="business">Business Content</option>
                        <option value="academic">Academic Writing</option>
                    </select>
                </div>
                <div class="form-group">
                    <label for="agent-persona">Persona Description</label>
                    <textarea id="agent-persona" required placeholder="Describe the agent's personality, expertise, and writing style..." rows="4"></textarea>
                </div>
                <div class="form-group">
                    <label for="agent-tone">Default Tone</label>
                    <select id="agent-tone">
                        <option value="professional">Professional</option>
                        <option value="casual">Casual</option>
                        <option value="conversational">Conversational</option>
                        <option value="authoritative">Authoritative</option>
                    </select>
                </div>
                <button type="submit" class="btn-primary btn-full">
                    <i class="fas fa-plus-circle"></i> Create AI Agent
                </button>
            </form>
        </div>
    </div>

    <!-- Save Article Modal -->
    <div id="save-modal" class="modal">
        <div class="modal-content">
            <span class="close-modal">&times;</span>
            <h3><i class="fas fa-save"></i> Save Article</h3>
            <form id="save-form">
                <div class="form-group">
                    <label for="article-title">Article Title</label>
                    <input type="text" id="article-title" required>
                </div>
                <div class="form-group">
                    <label for="article-category">Category</label>
                    <select id="article-category">
                        <option value="general">General</option>
                        <option value="seo">SEO</option>
                        <option value="technical">Technical</option>
                        <option value="creative">Creative</option>
                        <option value="business">Business</option>
                    </select>
                </div>
                <div class="form-group">
                    <label for="article-tags">Tags (comma separated)</label>
                    <input type="text" id="article-tags" placeholder="tag1, tag2, tag3">
                </div>
                <button type="submit" class="btn-primary btn-full">Save Article</button>
            </form>
        </div>
    </div>

    <script src="hide/gem.js"></script>
    <script>
        // Enhanced Writer Page JavaScript with REAL QUILLFY AI API Integration
        document.addEventListener('DOMContentLoaded', function() {
            initializeWriterPage();
        });

        function initializeWriterPage() {
            // Initialize range value displays
            initializeRangeSliders();
            
            // Advanced settings toggle
            initializeAdvancedSettings();
            
            // Form submission
            initializeContentGeneration();
            
            // Sidebar functionality
            initializeSidebar();
            
            // Modal functionality
            initializeModals();
            
            // Real-time statistics
            initializeStatistics();
        }

        function initializeRangeSliders() {
            const sliders = document.querySelectorAll('input[type="range"]');
            sliders.forEach(slider => {
                const valueDisplay = slider.parentElement.querySelector('.range-value');
                if (valueDisplay) {
                    valueDisplay.textContent = slider.value + '%';
                    
                    slider.addEventListener('input', function() {
                        valueDisplay.textContent = this.value + '%';
                    });
                }
            });
        }

        function initializeAdvancedSettings() {
            const settingsHeader = document.querySelector('.settings-header');
            const settingsPanel = document.querySelector('.settings-panel');
            const toggleIcon = document.querySelector('.toggle-icon');

            if (settingsHeader && settingsPanel) {
                settingsHeader.addEventListener('click', function() {
                    settingsPanel.classList.toggle('active');
                    toggleIcon.textContent = settingsPanel.classList.contains('active') ? '▲' : '▼';
                });
            }
        }

        function initializeContentGeneration() {
            const form = document.getElementById('content-form');
            const generateBtn = document.getElementById('generate-btn');

            form.addEventListener('submit', async function(e) {
                e.preventDefault();
                await generateContentWithGemini();
            });

            // Output controls
            document.getElementById('copy-output').addEventListener('click', copyOutput);
            document.getElementById('clear-output').addEventListener('click', clearOutput);
            document.getElementById('format-output').addEventListener('click', formatOutput);
        }

        function initializeSidebar() {
            // New content button
            document.getElementById('new-content').addEventListener('click', function() {
                document.getElementById('content-form').reset();
                document.getElementById('output-content').innerHTML = 
                    '<p class="placeholder">Your AI-generated content will appear here with real-time streaming from Quillfy AI...</p>';
                initializeRangeSliders();
                document.getElementById('output-stats').style.display = 'none';
            });

            // Save content button
            document.getElementById('save-content').addEventListener('click', function() {
                const content = document.getElementById('output-content').textContent;
                if (content.trim() && !content.includes('placeholder')) {
                    showSaveModal();
                } else {
                    showNotification('Please generate some content first!', 'error');
                }
            });

            // Export buttons
            document.getElementById('export-doc').addEventListener('click', exportToDOC);
            document.getElementById('export-txt').addEventListener('click', exportToTXT);

            // Create agent button
            document.getElementById('create-agent').addEventListener('click', function() {
                document.getElementById('agent-modal').style.display = 'block';
            });
        }

        function initializeModals() {
            // Close modals when clicking X
            document.querySelectorAll('.close-modal').forEach(closeBtn => {
                closeBtn.addEventListener('click', function() {
                    this.closest('.modal').style.display = 'none';
                });
            });

            // Close modals when clicking outside
            window.addEventListener('click', function(e) {
                document.querySelectorAll('.modal').forEach(modal => {
                    if (e.target === modal) {
                        modal.style.display = 'none';
                    }
                });
            });

            // Agent form submission
            document.getElementById('agent-form').addEventListener('submit', function(e) {
                e.preventDefault();
                createNewAgent();
            });

            // Save form submission
            document.getElementById('save-form').addEventListener('submit', function(e) {
                e.preventDefault();
                saveArticle();
            });
        }

        function initializeStatistics() {
            // Update statistics when content changes
            const outputContent = document.getElementById('output-content');
            const observer = new MutationObserver(updateStatistics);
            observer.observe(outputContent, { childList: true, characterData: true, subtree: true });
        }

        function updateStatistics() {
            const content = document.getElementById('output-content').textContent;
            const wordCount = content.trim() ? content.split(/\s+/).length : 0;
            const charCount = content.length;
            const readingTime = Math.ceil(wordCount / 200);

            document.getElementById('word-count-stat').textContent = wordCount;
            document.getElementById('char-count-stat').textContent = charCount;
            document.getElementById('reading-time-stat').textContent = readingTime + ' min';

            document.getElementById('output-stats').style.display = wordCount > 0 ? 'block' : 'none';
        }

        async function generateContentWithGemini() {
            const generateBtn = document.getElementById('generate-btn');
            const outputContent = document.getElementById('output-content');
            
            // Get form data
            const formData = new FormData(document.getElementById('content-form'));
            const topic = formData.get('topic');
            const contentType = formData.get('content_type');
            const tone = formData.get('tone');
            const wordCount = formData.get('word_count');
            const keywords = formData.get('keywords');
            const aiAgent = formData.get('ai_agent');
            const targetAudience = formData.get('target_audience');
            const outline = formData.get('content_outline');
            const instructions = formData.get('extra_instructions');

            if (!topic.trim()) {
                showNotification('Please enter a topic!', 'error');
                return;
            }

            generateBtn.disabled = true;
            generateBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Connecting to Gemini AI...';
            
            outputContent.innerHTML = '<div class="loading"><i class="fas fa-cog fa-spin"></i> Initializing Quillfy AI...<div class="spinner"></div></div>';

            try {
                // Build the advanced prompt based on form inputs
                const prompt = buildAdvancedPrompt({
                    topic,
                    contentType,
                    tone,
                    wordCount,
                    keywords,
                    aiAgent,
                    targetAudience,
                    outline,
                    instructions
                });

                // Call the REAL Quillfy AI API
                const response = await callGeminiAPI(prompt);
                
                // Stream the response in real-time
                await streamGeminiResponse(response, outputContent);
                
                showNotification('Content generated successfully using QUILLFY AI!', 'success');
                
            } catch (error) {
                console.error('Gemini API Error:', error);
                outputContent.innerHTML = `<div class="error"><i class="fas fa-exclamation-triangle"></i> API Error: ${error.message}</div>`;
                showNotification('Failed to generate content. Please try again.', 'error');
            } finally {
                generateBtn.disabled = false;
                generateBtn.innerHTML = '<i class="fas fa-bolt"></i> Generate Human-Like Content';
            }
        }

        function buildAdvancedPrompt(params) {
            const {
                topic,
                contentType,
                tone,
                wordCount,
                keywords,
                aiAgent,
                targetAudience,
                outline,
                instructions
            } = params;

            // Base prompt from gem.js
            let basePrompt = window.AI_PROMPTS.DEFAULT;
            
            // Enhanced prompt with all parameters
            return `${basePrompt}

SPECIFIC CONTENT REQUEST:
- Topic: ${topic}
- Content Type: ${contentType}
- Desired Tone: ${tone}
- Target Word Count: ${wordCount}
- Target Audience: ${targetAudience || 'General audience'}
- Keywords: ${keywords || 'Not specified'}
- Content Outline: ${outline || 'Not specified'}
- Additional Instructions: ${instructions || 'None'}

AI AGENT PERSONA: ${getAgentPersona(aiAgent)}

CONTENT REQUIREMENTS:
1. Must be 100% original and human-like
2. Must include personal anecdotes and experiences
3. Must contain contrarian opinions and unique insights
4. Must be optimized for SEO and readability
5. Must bypass AI detection tools
6. Must be engaging and valuable for the target audience

Please generate the content now following all the above specifications.`;
        }

        function getAgentPersona(agent) {
            const personas = {
                'lily-seo': 'Lily SEO - SEO content specialist with 10+ years experience',
                'tech-expert': 'Tech Expert - Technical writer with deep industry knowledge',
                'creative-writer': 'Creative Writer - Storyteller with engaging narrative style',
                'business-analyst': 'Business Analyst - Professional business content creator',
                'marketing-guru': 'Marketing Guru - Persuasive marketing content specialist'
            };
            return personas[agent] || 'Expert Content Creator';
        }

        async function callGeminiAPI(prompt) {
            const API_URL = 'https://generativelanguage.googleapis.com/v1beta/models/gemini-2.0-flash:generateContent';
            const API_KEY = 'AIzaSyAVsYGdeXGRYNs3FJ17ZW5qZamKN-LHxP4';

            const requestBody = {
                contents: [{
                    parts: [{
                        text: prompt
                    }]
                }],
                generationConfig: {
                    temperature: 0.8, // Controls creativity
                    topK: 40,
                    topP: 0.95,
                    maxOutputTokens: 2048 // Increased for longer content
                }
            };

            const response = await fetch(`${API_URL}?key=${API_KEY}`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify(requestBody)
            });

            if (!response.ok) {
                const errorData = await response.json().catch(() => ({}));
                throw new Error(errorData.error?.message || `HTTP ${response.status}: ${response.statusText}`);
            }

            return await response.json();
        }

        async function streamGeminiResponse(apiResponse, outputElement) {
            outputElement.innerHTML = '';
            
            if (apiResponse.candidates && apiResponse.candidates[0] && apiResponse.candidates[0].content) {
                const text = apiResponse.candidates[0].content.parts[0].text;
                
                // Create typing effect
                let index = 0;
                outputElement.innerHTML = '';
                
                function typeWriter() {
                    if (index < text.length) {
                        // Add text character by character
                        outputElement.innerHTML += text.charAt(index);
                        index++;
                        
                        // Auto-scroll to bottom
                        outputElement.scrollTop = outputElement.scrollHeight;
                        
                        // Continue typing with slight random delay for natural effect
                        const delay = Math.random() * 20 + 10; // 10-30ms delay
                        setTimeout(typeWriter, delay);
                    }
                }
                
                typeWriter();
            } else {
                throw new Error('No content generated by Gemini API');
            }
        }

        function copyOutput() {
            const outputText = document.getElementById('output-content').textContent;
            if (!outputText.trim() || outputText.includes('placeholder')) {
                showNotification('No content to copy!', 'error');
                return;
            }
            
            navigator.clipboard.writeText(outputText).then(() => {
                showNotification('Content copied to clipboard!', 'success');
            }).catch(() => {
                showNotification('Failed to copy content', 'error');
            });
        }

        function clearOutput() {
            document.getElementById('output-content').innerHTML = 
                '<p class="placeholder">Your AI-generated content will appear here with real-time streaming from Quillfy AI...</p>';
            document.getElementById('output-stats').style.display = 'none';
        }

        function formatOutput() {
            const outputContent = document.getElementById('output-content');
            const content = outputContent.textContent;
            
            if (!content.trim() || content.includes('placeholder')) {
                showNotification('No content to format!', 'error');
                return;
            }
            
            // Basic formatting - you can enhance this
            let formattedContent = content
                .replace(/\n\n+/g, '</p><p>')
                .replace(/\n/g, '<br>')
                .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>')
                .replace(/\*(.*?)\*/g, '<em>$1</em>');
            
            outputContent.innerHTML = '<p>' + formattedContent + '</p>';
            showNotification('Content formatted!', 'success');
        }

        function showSaveModal() {
            const content = document.getElementById('output-content').textContent;
            document.getElementById('article-title').value = 
                document.getElementById('topic').value || 'Untitled Article';
            document.getElementById('save-modal').style.display = 'block';
        }

        function saveArticle() {
            const title = document.getElementById('article-title').value;
            const category = document.getElementById('article-category').value;
            const tags = document.getElementById('article-tags').value;
            const content = document.getElementById('output-content').textContent;
            
            // Create article data
            const articleData = {
                title: title,
                category: category,
                tags: tags.split(',').map(tag => tag.trim()).filter(tag => tag),
                content: content,
                wordCount: content.split(/\s+/).length,
                created: new Date().toISOString(),
                author: '<?php echo $user["username"]; ?>'
            };
            
            // Save to localStorage (in real app, this would be server-side)
            const articleId = 'article_' + Date.now();
            localStorage.setItem(articleId, JSON.stringify(articleData));
            
            document.getElementById('save-modal').style.display = 'none';
            showNotification('Article saved successfully!', 'success');
            
            // Add to sidebar
            updateSavedArticlesList(articleData);
        }

        function updateSavedArticlesList(articleData) {
            const articlesList = document.getElementById('saved-articles-list');
            const newArticle = document.createElement('div');
            newArticle.className = 'saved-item';
            newArticle.innerHTML = `
                <h4>${articleData.title}</h4>
                <p>Just now, ${articleData.wordCount} words</p>
            `;
            articlesList.insertBefore(newArticle, articlesList.firstChild);
        }

        function createNewAgent() {
            const name = document.getElementById('agent-name').value;
            const specialty = document.getElementById('agent-specialty').value;
            const persona = document.getElementById('agent-persona').value;
            const tone = document.getElementById('agent-tone').value;
            
            // Create agent data
            const agentData = {
                name: name,
                specialty: specialty,
                persona: persona,
                tone: tone,
                created: new Date().toISOString()
            };
            
            // Save to localStorage
            const agentId = 'agent_' + Date.now();
            localStorage.setItem(agentId, JSON.stringify(agentData));
            
            document.getElementById('agent-modal').style.display = 'none';
            showNotification(`AI Agent "${name}" created successfully!`, 'success');
            
            // Add to UI
            addAgentToUI(agentData);
            document.getElementById('agent-form').reset();
        }

        function addAgentToUI(agentData) {
            const agentsList = document.getElementById('ai-agents-list');
            const agentSelect = document.getElementById('ai-agent');
            
            // Add to sidebar list
            const newAgentItem = document.createElement('div');
            newAgentItem.className = 'agent-item';
            newAgentItem.innerHTML = `
                <h4>${agentData.name}</h4>
                <p>${getSpecialtyName(agentData.specialty)}</p>
            `;
            agentsList.appendChild(newAgentItem);
            
            // Add to dropdown
            const newAgentOption = document.createElement('option');
            newAgentOption.value = agentData.specialty + '-' + agentData.name.toLowerCase().replace(/\s+/g, '-');
            newAgentOption.textContent = `👤 ${agentData.name} - ${getSpecialtyName(agentData.specialty)}`;
            agentSelect.appendChild(newAgentOption);
        }

        function getSpecialtyName(specialty) {
            const specialties = {
                'seo': 'SEO Content',
                'technical': 'Technical Writing',
                'creative': 'Creative Writing',
                'business': 'Business Content',
                'academic': 'Academic Writing'
            };
            return specialties[specialty] || 'Content Specialist';
        }

        function exportToDOC() {
            const content = document.getElementById('output-content').textContent;
            if (!content.trim() || content.includes('placeholder')) {
                showNotification('Please generate some content first!', 'error');
                return;
            }
            showNotification('DOC export feature would be implemented here!', 'info');
        }

        function exportToTXT() {
            const content = document.getElementById('output-content').textContent;
            if (!content.trim() || content.includes('placeholder')) {
                showNotification('Please generate some content first!', 'error');
                return;
            }
            
            const blob = new Blob([content], { type: 'text/plain' });
            const url = URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = 'ai-content-' + Date.now() + '.txt';
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            URL.revokeObjectURL(url);
            showNotification('Content exported as TXT!', 'success');
        }

        function showNotification(message, type = 'info') {
            const notification = document.createElement('div');
            notification.className = `notification ${type}`;
            notification.innerHTML = `
                <i class="fas fa-${type === 'success' ? 'check' : type === 'error' ? 'exclamation-triangle' : 'info'}"></i>
                ${message}
            `;
            
            notification.style.cssText = `
                position: fixed;
                top: 20px;
                right: 20px;
                padding: 1rem 1.5rem;
                background: ${type === 'success' ? '#48bb78' : type === 'error' ? '#f56565' : '#4299e1'};
                color: white;
                border-radius: 8px;
                z-index: 1000;
                animation: slideInRight 0.3s ease;
                box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            `;
            
            document.body.appendChild(notification);
            
            setTimeout(() => {
                notification.style.animation = 'slideOutRight 0.3s ease';
                setTimeout(() => notification.remove(), 300);
            }, 3000);
        }

        // Add CSS for slide animations
        const style = document.createElement('style');
        style.textContent = `
            @keyframes slideInRight {
                from { transform: translateX(100%); opacity: 0; }
                to { transform: translateX(0); opacity: 1; }
            }
            @keyframes slideOutRight {
                from { transform: translateX(0); opacity: 1; }
                to { transform: translateX(100%); opacity: 0; }
            }
        `;
        document.head.appendChild(style);
    </script>
</body>
</html>