<?php
// api.php - Central API Router

require_once 'config.php';
require_once 'inc/database.inc.php';
require_once 'inc/user.inc.php';
require_once 'inc/ai-core.inc.php';

header('Content-Type: text/plain; charset=utf-8');
header('Cache-Control: no-cache');
header('X-Accel-Buffering: no');

$db = new Database();
$user = new User($db);
$ai_core = new AICore($db);

if (!$user->isLoggedIn()) {
    http_response_code(401);
    echo "data: " . json_encode(['error' => 'Authentication required']) . "\n\n";
    exit;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    switch ($action) {
        case 'generate_content':
            handleContentGeneration($user, $ai_core);
            break;
        default:
            http_response_code(400);
            echo "data: " . json_encode(['error' => 'Invalid action']) . "\n\n";
            break;
    }
}

function handleContentGeneration($user, $ai_core) {
    // Check subscription or free trial
    if (!$user->hasActiveSubscription() && $user->getRemainingFreeArticles() <= 0) {
        sendError('No remaining free articles. Please upgrade your subscription.');
        return;
    }

    // Get form data
    $topic = filter_var($_POST['topic'] ?? '', FILTER_SANITIZE_STRING);
    $content_type = filter_var($_POST['content_type'] ?? 'article', FILTER_SANITIZE_STRING);
    $tone = filter_var($_POST['tone'] ?? 'professional', FILTER_SANITIZE_STRING);
    $word_count = filter_var($_POST['word_count'] ?? 1000, FILTER_VALIDATE_INT);

    if (empty($topic)) {
        sendError('Topic is required');
        return;
    }

    // Generate content using AI core
    $result = $ai_core->generateContent($user, [
        'topic' => $topic,
        'content_type' => $content_type,
        'tone' => $tone,
        'word_count' => $word_count
    ]);

    if ($result['success']) {
        sendContent($result['content']);
    } else {
        sendError($result['message']);
    }
}

function sendContent($content) {
    // Simulate streaming by sending content in chunks
    $words = explode(' ', $content);
    $chunk_size = 3; // Words per chunk
    
    for ($i = 0; $i < count($words); $i += $chunk_size) {
        $chunk = implode(' ', array_slice($words, $i, $chunk_size));
        echo "data: " . json_encode(['content' => $chunk . ' ']) . "\n\n";
        ob_flush();
        flush();
        
        // Small delay to simulate streaming
        usleep(100000); // 100ms
    }
    
    echo "data: [DONE]\n\n";
    ob_flush();
    flush();
}

function sendError($message) {
    echo "data: " . json_encode(['error' => $message]) . "\n\n";
    ob_flush();
    flush();
}
?>