// ai-writer.js - Advanced AI Content Generation Handler
class AIWriter {
    constructor() {
        this.isGenerating = false;
        this.currentContent = '';
        this.generatedTitle = '';
        this.advancedSettings = new AdvancedAISettings();
        this.init();
    }

    init() {
        this.setupEventListeners();
        this.loadUserPreferences();
        this.initializeAIAgents();
    }

    setupEventListeners() {
        // Main generation form
        const form = document.getElementById('aiWriterForm');
        if (form) {
            form.addEventListener('submit', (e) => {
                this.handleGeneration(e);
            });
        }

        // Content actions
        document.getElementById('copyContent')?.addEventListener('click', () => {
            this.copyContent();
        });

        document.getElementById('downloadContent')?.addEventListener('click', () => {
            this.downloadContent();
        });

        document.getElementById('publishWordPress')?.addEventListener('click', () => {
            this.publishToWordPress();
        });

        document.getElementById('proofreadContent')?.addEventListener('click', () => {
            this.proofreadContent();
        });

        document.getElementById('rewriteContent')?.addEventListener('click', () => {
            this.rewriteContent();
        });

        document.getElementById('expandContent')?.addEventListener('click', () => {
            this.expandContent();
        });

        document.getElementById('shortenContent')?.addEventListener('click', () => {
            this.shortenContent();
        });

        // Keyword suggestions
        document.getElementById('suggestKeywords')?.addEventListener('click', () => {
            this.suggestKeywords();
        });

        // Template selection
        document.getElementById('contentTemplate')?.addEventListener('change', (e) => {
            this.loadTemplate(e.target.value);
        });

        // AI agent selection
        document.getElementById('aiAgent')?.addEventListener('change', (e) => {
            this.updateAIAgent(e.target.value);
        });

        // Real-time preview
        const topicInput = document.getElementById('topic');
        if (topicInput) {
            topicInput.addEventListener('input', this.debounce(() => {
                this.generateOutlinePreview();
            }, 500));
        }

        // Advanced settings toggles
        document.querySelectorAll('.settings-toggle').forEach(toggle => {
            toggle.addEventListener('click', (e) => {
                this.toggleSettingsSection(e.target.dataset.section);
            });
        });

        // Word count tracking
        document.getElementById('wordCount')?.addEventListener('input', (e) => {
            this.updateWordCountPreview(e.target.value);
        });

        // Tone preview
        document.getElementById('tone')?.addEventListener('change', (e) => {
            this.updateTonePreview(e.target.value);
        });
    }

    async handleGeneration(e) {
        e.preventDefault();
        if (this.isGenerating) {
            this.showNotification('Content generation already in progress', 'warning');
            return;
        }

        const formData = new FormData(e.target);
        const validation = this.validateFormData(formData);
        
        if (!validation.valid) {
            this.showNotification(validation.message, 'error');
            return;
        }

        this.isGenerating = true;
        
        // Update UI state
        const submitBtn = e.target.querySelector('button[type="submit"]');
        const originalText = submitBtn.textContent;
        submitBtn.innerHTML = '<span class="loading"></span> Generating...';
        submitBtn.disabled = true;

        // Show progress indicator
        this.showGenerationProgress();

        // Clear previous content
        const outputElement = document.getElementById('contentOutput');
        outputElement.innerHTML = '<div class="generation-starting">Starting content generation...</div>';

        try {
            await this.streamContent(formData, outputElement);
            this.showNotification('Content generated successfully!', 'success');
        } catch (error) {
            console.error('Generation error:', error);
            this.showError('Content generation failed: ' + error.message);
        } finally {
            this.isGenerating = false;
            submitBtn.textContent = originalText;
            submitBtn.disabled = false;
            this.hideGenerationProgress();
        }
    }

    validateFormData(formData) {
        const topic = formData.get('topic');
        const wordCount = parseInt(formData.get('word_count'));
        const contentType = formData.get('content_type');

        if (!topic || topic.trim().length < 10) {
            return { valid: false, message: 'Please provide a detailed topic (minimum 10 characters)' };
        }

        if (wordCount < 100 || wordCount > 10000) {
            return { valid: false, message: 'Word count must be between 100 and 10,000' };
        }

        if (!contentType) {
            return { valid: false, message: 'Please select a content type' };
        }

        return { valid: true };
    }

    async streamContent(formData, outputElement) {
        // Add advanced settings to form data
        const settings = this.advancedSettings.getSettingsForAPI();
        Object.keys(settings).forEach(key => {
            formData.append(`settings[${key}]`, settings[key]);
        });

        // Add action and metadata
        formData.append('action', 'generate_content');
        formData.append('stream', 'true');
        formData.append('timestamp', Date.now());

        const response = await fetch('api.php', {
            method: 'POST',
            headers: {
                'X-Requested-With': 'XMLHttpRequest'
            },
            body: formData
        });

        if (!response.ok) {
            throw new Error(`HTTP ${response.status}: ${response.statusText}`);
        }

        if (!response.body) {
            throw new Error('Response body is not available');
        }

        const reader = response.body.getReader();
        const decoder = new TextDecoder();
        this.currentContent = '';
        this.generatedTitle = '';

        outputElement.innerHTML = '<div class="content-stream"></div>';
        const streamContainer = outputElement.querySelector('.content-stream');

        let buffer = '';
        let isFirstChunk = true;

        try {
            while (true) {
                const { done, value } = await reader.read();
                
                if (done) break;

                buffer += decoder.decode(value, { stream: true });
                const lines = buffer.split('\n');
                buffer = lines.pop() || ''; // Keep incomplete line in buffer

                for (const line of lines) {
                    if (line.startsWith('data: ')) {
                        const dataStr = line.slice(6);
                        
                        if (dataStr === '[DONE]') {
                            this.finalizeContent(streamContainer);
                            return;
                        }

                        try {
                            const data = JSON.parse(dataStr);
                            
                            if (data.error) {
                                throw new Error(data.error);
                            }

                            if (data.title && !this.generatedTitle) {
                                this.generatedTitle = data.title;
                                this.displayTitle(streamContainer, data.title);
                            }

                            if (data.content) {
                                this.currentContent += data.content;
                                this.displayContentChunk(streamContainer, data.content, isFirstChunk);
                                isFirstChunk = false;
                            }

                            if (data.metadata) {
                                this.updateGenerationMetrics(data.metadata);
                            }

                        } catch (e) {
                            console.warn('Failed to parse JSON:', dataStr, e);
                            continue;
                        }
                    }
                }
            }
        } finally {
            reader.releaseLock();
        }
    }

    displayTitle(container, title) {
        const titleElement = document.createElement('h3');
        titleElement.className = 'generated-title';
        titleElement.textContent = title;
        container.appendChild(titleElement);

        // Update the title input field if it exists
        const titleInput = document.querySelector('input[name="title"]');
        if (titleInput) {
            titleInput.value = title;
        }
    }

    displayContentChunk(container, chunk, isFirstChunk) {
        if (isFirstChunk) {
            const contentElement = document.createElement('div');
            contentElement.className = 'generated-content';
            contentElement.innerHTML = this.formatContent(chunk);
            container.appendChild(contentElement);
        } else {
            const contentElement = container.querySelector('.generated-content');
            if (contentElement) {
                contentElement.innerHTML += this.formatContent(chunk);
                // Auto-scroll to bottom
                contentElement.scrollTop = contentElement.scrollHeight;
            }
        }

        // Update word count in real-time
        this.updateRealTimeWordCount();
    }

    formatContent(content) {
        // Enhanced content formatting
        return content
            .split('\n')
            .map(paragraph => {
                const trimmed = paragraph.trim();
                if (!trimmed) return '';

                // Detect headings (lines that are short and end with colon or are in title case)
                if (trimmed.length < 60 && (trimmed.endsWith(':') || this.looksLikeHeading(trimmed))) {
                    return `<h4>${trimmed.replace(':', '')}</h4>`;
                }

                // Detect lists
                if (trimmed.match(/^[\d•\-]\s/)) {
                    return `<li>${trimmed.replace(/^[\d•\-]\s/, '')}</li>`;
                }

                // Regular paragraphs with enhanced formatting
                return `<p>${this.enhanceParagraph(trimmed)}</p>`;
            })
            .join('')
            .replace(/(<li>.*?<\/li>)/gs, '<ul>$1</ul>')
            .replace(/<\/ul>\s*<ul>/g, '');
    }

    looksLikeHeading(text) {
        const words = text.split(' ');
        return words.length <= 8 && text === text.toUpperCase() || 
               (text[0] === text[0].toUpperCase() && words.length <= 6);
    }

    enhanceParagraph(text) {
        // Add emphasis to important phrases
        let enhanced = text.replace(/\b(crucial|important|essential|key|significant)\b/gi, '<strong>$1</strong>');
        
        // Highlight numbers and statistics
        enhanced = enhanced.replace(/(\d+%?|\d+\.\d+%?)/g, '<span class="statistic">$1</span>');
        
        // Add subtle emphasis to transitional phrases
        const transitions = ['however', 'therefore', 'moreover', 'furthermore', 'consequently'];
        transitions.forEach(transition => {
            enhanced = enhanced.replace(new RegExp(`\\b${transition}\\b`, 'gi'), `<em>${transition}</em>`);
        });

        return enhanced;
    }

    finalizeContent(container) {
        // Add completion marker and actions
        const completionMarker = document.createElement('div');
        completionMarker.className = 'generation-complete';
        completionMarker.innerHTML = `
            <div class="completion-badge">✓ Generation Complete</div>
            <div class="content-stats">
                <span>Words: ${this.getWordCount()}</span>
                <span>Characters: ${this.currentContent.length}</span>
                <span>Reading Time: ${this.calculateReadingTime()} min</span>
            </div>
        `;
        container.appendChild(completionMarker);

        // Enable post-generation actions
        this.enableContentActions();
    }

    getWordCount() {
        return this.currentContent.split(/\s+/).filter(word => word.length > 0).length;
    }

    calculateReadingTime() {
        const wordsPerMinute = 200;
        const wordCount = this.getWordCount();
        return Math.ceil(wordCount / wordsPerMinute);
    }

    updateRealTimeWordCount() {
        const wordCountElement = document.getElementById('realTimeWordCount');
        if (wordCountElement) {
            wordCountElement.textContent = this.getWordCount();
        }
    }

    updateGenerationMetrics(metadata) {
        // Update any progress bars or metrics
        const progressElement = document.getElementById('generationProgress');
        if (progressElement && metadata.progress) {
            progressElement.style.width = `${metadata.progress}%`;
            progressElement.textContent = `${metadata.progress}%`;
        }
    }

    showGenerationProgress() {
        // Create or show progress overlay
        let progressOverlay = document.getElementById('generationProgressOverlay');
        if (!progressOverlay) {
            progressOverlay = document.createElement('div');
            progressOverlay.id = 'generationProgressOverlay';
            progressOverlay.innerHTML = `
                <div class="progress-container">
                    <div class="progress-bar">
                        <div class="progress-fill" id="generationProgress"></div>
                    </div>
                    <div class="progress-text">Generating content...</div>
                    <div class="progress-stats">
                        <span id="realTimeWordCount">0</span> words generated
                    </div>
                </div>
            `;
            document.body.appendChild(progressOverlay);
        }
        progressOverlay.style.display = 'flex';
    }

    hideGenerationProgress() {
        const progressOverlay = document.getElementById('generationProgressOverlay');
        if (progressOverlay) {
            progressOverlay.style.display = 'none';
        }
    }

    copyContent() {
        if (!this.currentContent) {
            this.showNotification('No content to copy', 'warning');
            return;
        }

        navigator.clipboard.writeText(this.currentContent).then(() => {
            this.showNotification('Content copied to clipboard!', 'success');
        }).catch(() => {
            // Fallback for older browsers
            const textArea = document.createElement('textarea');
            textArea.value = this.currentContent;
            document.body.appendChild(textArea);
            textArea.select();
            document.execCommand('copy');
            document.body.removeChild(textArea);
            this.showNotification('Content copied to clipboard!', 'success');
        });
    }

    downloadContent() {
        if (!this.currentContent) {
            this.showNotification('No content to download', 'warning');
            return;
        }

        const title = this.generatedTitle || 'ai-generated-content';
        const filename = title.toLowerCase().replace(/[^a-z0-9]/g, '-') + '.txt';
        
        const blob = new Blob([this.currentContent], { type: 'text/plain;charset=utf-8' });
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = filename;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        URL.revokeObjectURL(url);
        
        this.showNotification('Content downloaded!', 'success');
    }

    async publishToWordPress() {
        if (!this.currentContent) {
            this.showNotification('No content to publish', 'warning');
            return;
        }

        const title = this.generatedTitle || prompt('Enter a title for your post:');
        if (!title) return;

        try {
            const formData = new FormData();
            formData.append('action', 'publish_wordpress');
            formData.append('title', title);
            formData.append('content', this.currentContent);
            formData.append('status', document.getElementById('publishStatus')?.value || 'draft');

            const response = await fetch('api.php', {
                method: 'POST',
                body: formData
            });

            const result = await response.json();

            if (result.success) {
                this.showNotification('Content published to WordPress successfully!', 'success');
            } else {
                this.showNotification('Publishing failed: ' + result.message, 'error');
            }
        } catch (error) {
            this.showNotification('Publishing failed: ' + error.message, 'error');
        }
    }

    async proofreadContent() {
        if (!this.currentContent) {
            this.showNotification('No content to proofread', 'warning');
            return;
        }

        this.showNotification('Proofreading content...', 'info');

        try {
            const formData = new FormData();
            formData.append('action', 'proofread_content');
            formData.append('content', this.currentContent);

            const response = await fetch('api.php', {
                method: 'POST',
                body: formData
            });

            const result = await response.json();

            if (result.success) {
                this.displayProofreadResults(result.corrections);
            } else {
                this.showNotification('Proofreading failed: ' + result.message, 'error');
            }
        } catch (error) {
            this.showNotification('Proofreading failed: ' + error.message, 'error');
        }
    }

    displayProofreadResults(corrections) {
        const outputElement = document.getElementById('contentOutput');
        let proofreadContent = this.currentContent;

        corrections.forEach(correction => {
            proofreadContent = proofreadContent.replace(
                correction.original,
                `<span class="correction" title="${correction.suggestion}">${correction.corrected}</span>`
            );
        });

        outputElement.innerHTML = `
            <div class="proofread-results">
                <h4>Proofread Results</h4>
                <div class="correction-stats">Found ${corrections.length} suggestions</div>
                <div class="proofread-content">${proofreadContent}</div>
                <div class="proofread-actions">
                    <button onclick="aiWriter.applyProofreadCorrections()" class="btn-primary">Apply All Corrections</button>
                    <button onclick="aiWriter.revertProofread()" class="btn-secondary">Revert</button>
                </div>
            </div>
        `;
    }

    applyProofreadCorrections() {
        const proofreadContent = document.querySelector('.proofread-content');
        if (proofreadContent) {
            this.currentContent = proofreadContent.textContent;
            this.showNotification('Corrections applied successfully!', 'success');
            this.displayCurrentContent();
        }
    }

    revertProofread() {
        this.displayCurrentContent();
        this.showNotification('Proofread changes reverted', 'info');
    }

    displayCurrentContent() {
        const outputElement = document.getElementById('contentOutput');
        outputElement.innerHTML = `
            <div class="generated-content">${this.formatContent(this.currentContent)}</div>
            <div class="generation-complete">
                <div class="completion-badge">✓ Content Ready</div>
            </div>
        `;
    }

    async rewriteContent() {
        if (!this.currentContent) {
            this.showNotification('No content to rewrite', 'warning');
            return;
        }

        const tone = prompt('Enter new tone (e.g., more professional, casual, persuasive):') || 'improved';
        
        try {
            const formData = new FormData();
            formData.append('action', 'rewrite_content');
            formData.append('content', this.currentContent);
            formData.append('tone', tone);

            const response = await fetch('api.php', {
                method: 'POST',
                body: formData
            });

            const result = await response.json();

            if (result.success) {
                this.currentContent = result.rewritten_content;
                this.displayCurrentContent();
                this.showNotification('Content rewritten successfully!', 'success');
            } else {
                this.showNotification('Rewriting failed: ' + result.message, 'error');
            }
        } catch (error) {
            this.showNotification('Rewriting failed: ' + error.message, 'error');
        }
    }

    async expandContent() {
        if (!this.currentContent) {
            this.showNotification('No content to expand', 'warning');
            return;
        }

        try {
            const formData = new FormData();
            formData.append('action', 'expand_content');
            formData.append('content', this.currentContent);

            const response = await fetch('api.php', {
                method: 'POST',
                body: formData
            });

            const result = await response.json();

            if (result.success) {
                this.currentContent = result.expanded_content;
                this.displayCurrentContent();
                this.showNotification('Content expanded successfully!', 'success');
            } else {
                this.showNotification('Expansion failed: ' + result.message, 'error');
            }
        } catch (error) {
            this.showNotification('Expansion failed: ' + error.message, 'error');
        }
    }

    async shortenContent() {
        if (!this.currentContent) {
            this.showNotification('No content to shorten', 'warning');
            return;
        }

        const targetLength = prompt('Target word count:', Math.floor(this.getWordCount() / 2));
        if (!targetLength) return;

        try {
            const formData = new FormData();
            formData.append('action', 'shorten_content');
            formData.append('content', this.currentContent);
            formData.append('target_length', targetLength);

            const response = await fetch('api.php', {
                method: 'POST',
                body: formData
            });

            const result = await response.json();

            if (result.success) {
                this.currentContent = result.shortened_content;
                this.displayCurrentContent();
                this.showNotification('Content shortened successfully!', 'success');
            } else {
                this.showNotification('Shortening failed: ' + result.message, 'error');
            }
        } catch (error) {
            this.showNotification('Shortening failed: ' + error.message, 'error');
        }
    }

    async suggestKeywords() {
        const topic = document.getElementById('topic')?.value;
        if (!topic) {
            this.showNotification('Please enter a topic first', 'warning');
            return;
        }

        try {
            const formData = new FormData();
            formData.append('action', 'suggest_keywords');
            formData.append('topic', topic);

            const response = await fetch('api.php', {
                method: 'POST',
                body: formData
            });

            const result = await response.json();

            if (result.success) {
                this.displayKeywordSuggestions(result.keywords);
            } else {
                this.showNotification('Keyword suggestion failed: ' + result.message, 'error');
            }
        } catch (error) {
            this.showNotification('Keyword suggestion failed: ' + error.message, 'error');
        }
    }

    displayKeywordSuggestions(keywords) {
        const suggestionsHTML = keywords.map(group => `
            <div class="keyword-group">
                <h5>${group.intent}</h5>
                <div class="keyword-list">
                    ${group.keywords.map(kw => `<span class="keyword-tag">${kw}</span>`).join('')}
                </div>
            </div>
        `).join('');

        const modal = this.createModal('Keyword Suggestions', suggestionsHTML);
        document.body.appendChild(modal);

        // Add click handlers for keywords
        modal.querySelectorAll('.keyword-tag').forEach(tag => {
            tag.addEventListener('click', () => {
                this.addKeywordToInput(tag.textContent);
            });
        });
    }

    addKeywordToInput(keyword) {
        const keywordsInput = document.querySelector('textarea[name="keywords"]');
        if (keywordsInput) {
            const currentKeywords = keywordsInput.value.split(',').map(k => k.trim()).filter(k => k);
            if (!currentKeywords.includes(keyword)) {
                currentKeywords.push(keyword);
                keywordsInput.value = currentKeywords.join(', ');
            }
        }
    }

    createModal(title, content) {
        const modal = document.createElement('div');
        modal.className = 'modal-overlay';
        modal.innerHTML = `
            <div class="modal">
                <div class="modal-header">
                    <h4>${title}</h4>
                    <button class="modal-close">&times;</button>
                </div>
                <div class="modal-content">${content}</div>
            </div>
        `;

        modal.querySelector('.modal-close').addEventListener('click', () => {
            document.body.removeChild(modal);
        });

        modal.addEventListener('click', (e) => {
            if (e.target === modal) {
                document.body.removeChild(modal);
            }
        });

        return modal;
    }

    loadTemplate(templateId) {
        const templates = {
            'blog-post': {
                topic: 'How to [Topic] - A Complete Guide',
                tone: 'informative',
                word_count: 1500,
                outline: 'Introduction, Main Points, Conclusion'
            },
            'product-review': {
                topic: '[Product] Review: Honest Assessment After Testing',
                tone: 'balanced',
                word_count: 1200,
                outline: 'Overview, Pros and Cons, Verdict'
            },
            'listicle': {
                topic: 'X Best Ways to [Achieve Goal]',
                tone: 'engaging',
                word_count: 1000,
                outline: 'Introduction, List Items, Summary'
            }
        };

        const template = templates[templateId];
        if (template) {
            document.getElementById('topic').value = template.topic;
            document.getElementById('tone').value = template.tone;
            document.getElementById('wordCount').value = template.word_count;
            document.getElementById('outline').value = template.outline;
        }
    }

    updateAIAgent(agentId) {
        this.showNotification(`AI Agent changed to: ${agentId}`, 'info');
        // Additional agent-specific configuration can be applied here
    }

    toggleSettingsSection(sectionId) {
        const section = document.getElementById(sectionId);
        if (section) {
            section.classList.toggle('expanded');
        }
    }

    updateWordCountPreview(count) {
        const preview = document.getElementById('wordCountPreview');
        if (preview) {
            preview.textContent = `Approximately ${count} words`;
        }
    }

    updateTonePreview(tone) {
        const preview = document.getElementById('tonePreview');
        if (preview) {
            const descriptions = {
                'professional': 'Formal, business-appropriate language',
                'casual': 'Conversational, friendly tone',
                'authoritative': 'Expert, confident voice',
                'sarcastic': 'Ironic, humorous approach'
            };
            preview.textContent = descriptions[tone] || 'Custom tone selected';
        }
    }

    async generateOutlinePreview() {
        const topic = document.getElementById('topic')?.value;
        if (!topic || topic.length < 10) return;

        // Debounced outline generation
        clearTimeout(this.outlineTimeout);
        this.outlineTimeout = setTimeout(async () => {
            try {
                const formData = new FormData();
                formData.append('action', 'generate_outline');
                formData.append('topic', topic);

                const response = await fetch('api.php', {
                    method: 'POST',
                    body: formData
                });

                const result = await response.json();

                if (result.success) {
                    this.displayOutlinePreview(result.outline);
                }
            } catch (error) {
                console.error('Outline generation failed:', error);
            }
        }, 1000);
    }

    displayOutlinePreview(outline) {
        const outlineElement = document.getElementById('outlinePreview');
        if (outlineElement) {
            outlineElement.innerHTML = `
                <h5>Suggested Outline:</h5>
                <div class="outline-suggestion">${outline}</div>
                <button onclick="aiWriter.applyOutlineSuggestion()" class="btn-small">Apply Outline</button>
            `;
        }
    }

    applyOutlineSuggestion() {
        const outlineSuggestion = document.querySelector('.outline-suggestion');
        if (outlineSuggestion) {
            document.getElementById('outline').value = outlineSuggestion.textContent;
            this.showNotification('Outline applied successfully!', 'success');
        }
    }

    enableContentActions() {
        document.querySelectorAll('.content-action').forEach(button => {
            button.disabled = false;
        });
    }

    loadUserPreferences() {
        const preferences = JSON.parse(localStorage.getItem('aiWriterPreferences') || '{}');
        
        // Apply saved preferences to form fields
        Object.keys(preferences).forEach(key => {
            const element = document.querySelector(`[name="${key}"]`);
            if (element) {
                element.value = preferences[key];
            }
        });
    }

    saveUserPreferences() {
        const preferences = {};
        document.querySelectorAll('input, select, textarea').forEach(element => {
            if (element.name) {
                preferences[element.name] = element.value;
            }
        });
        localStorage.setItem('aiWriterPreferences', JSON.stringify(preferences));
    }

    debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }

    showNotification(message, type = 'info') {
        // Reuse or enhance the notification system from app.js
        if (window.UncensoredAIWriter) {
            window.UncensoredAIWriter.showNotification(message, type);
        } else {
            // Fallback notification
            alert(`${type.toUpperCase()}: ${message}`);
        }
    }

    showError(message) {
        const outputElement = document.getElementById('contentOutput');
        outputElement.innerHTML = `
            <div class="error-message">
                <h4>❌ Generation Error</h4>
                <p>${message}</p>
                <button onclick="aiWriter.retryGeneration()" class="btn-primary">Retry</button>
            </div>
        `;
    }

    retryGeneration() {
        const form = document.getElementById('aiWriterForm');
        if (form) {
            this.handleGeneration(new Event('submit'));
        }
    }
}

// Advanced AI Settings Class
class AdvancedAISettings {
    constructor() {
        this.settings = this.loadDefaultSettings();
        this.init();
    }

    init() {
        this.setupEventListeners();
        this.loadSavedSettings();
    }

    setupEventListeners() {
        // Slider controls
        document.querySelectorAll('.setting-slider').forEach(slider => {
            slider.addEventListener('input', (e) => {
                const valueElement = document.getElementById(e.target.id + 'Value');
                if (valueElement) {
                    valueElement.textContent = e.target.value;
                }
                this.settings[e.target.id] = parseFloat(e.target.value);
                this.saveSettings();
            });
        });

        // Checkbox controls
        document.querySelectorAll('.setting-checkbox').forEach(checkbox => {
            checkbox.addEventListener('change', (e) => {
                this.settings[e.target.id] = e.target.checked;
                this.saveSettings();
            });
        });

        // Select controls
        document.querySelectorAll('.setting-select').forEach(select => {
            select.addEventListener('change', (e) => {
                this.settings[e.target.id] = e.target.value;
                this.saveSettings();
            });
        });

        // Preset buttons
        document.querySelectorAll('.preset-btn').forEach(btn => {
            btn.addEventListener('click', (e) => {
                this.loadPreset(e.target.dataset.preset);
            });
        });
    }

    loadDefaultSettings() {
        return {
            creativity: 0.8,
            research_depth: 3,
            emotional_depth: 'medium',
            humor_level: 'low',
            formality: 'medium',
            point_of_view: 'third_person',
            factual_strictness: 'balanced',
            include_statistics: true,
            include_quotes: false,
            include_examples: true,
            use_anecdotes: true,
            add_contrarian_views: false,
            sentence_variety: 'high',
            paragraph_variety: 'medium',
            transition_style: 'smooth',
            keyword_density: 'natural',
            seo_optimization: 'balanced',
            content_freshness: 'current',
            citation_style: 'none',
            complexity_level: 'accessible',
            reading_grade: '8-10'
        };
    }

    loadSavedSettings() {
        const saved = localStorage.getItem('aiWriterAdvancedSettings');
        if (saved) {
            this.settings = { ...this.settings, ...JSON.parse(saved) };
            this.updateUI();
        }
    }

    saveSettings() {
        localStorage.setItem('aiWriterAdvancedSettings', JSON.stringify(this.settings));
    }

    updateUI() {
        Object.keys(this.settings).forEach(key => {
            const element = document.getElementById(key);
            if (element) {
                if (element.type === 'range') {
                    element.value = this.settings[key];
                    const valueElement = document.getElementById(key + 'Value');
                    if (valueElement) {
                        valueElement.textContent = this.settings[key];
                    }
                } else if (element.type === 'checkbox') {
                    element.checked = this.settings[key];
                } else {
                    element.value = this.settings[key];
                }
            }
        });
    }

    loadPreset(presetName) {
        const presets = {
            'seo-optimized': {
                creativity: 0.7,
                research_depth: 4,
                include_statistics: true,
                seo_optimization: 'high',
                keyword_density: 'optimized'
            },
            'creative-writing': {
                creativity: 0.9,
                emotional_depth: 'high',
                humor_level: 'medium',
                sentence_variety: 'very_high',
                use_anecdotes: true
            },
            'academic': {
                creativity: 0.5,
                formality: 'high',
                factual_strictness: 'strict',
                citation_style: 'academic',
                complexity_level: 'advanced'
            },
            'conversational': {
                creativity: 0.8,
                formality: 'low',
                point_of_view: 'first_person',
                sentence_variety: 'medium',
                transition_style: 'conversational'
            }
        };

        if (presets[presetName]) {
            this.settings = { ...this.settings, ...presets[presetName] };
            this.updateUI();
            this.saveSettings();
            
            // Show notification
            if (window.aiWriter) {
                window.aiWriter.showNotification(`Loaded "${presetName}" preset`, 'success');
            }
        }
    }

    getSettingsForAPI() {
        return {
            temperature: this.settings.creativity,
            max_tokens: 500 + (this.settings.research_depth * 300),
            presence_penalty: (this.settings.creativity - 0.5) * 2,
            frequency_penalty: this.settings.factual_strictness === 'strict' ? 0.5 : 0,
            // Additional settings for the AI model
            style: {
                formality: this.settings.formality,
                emotional_depth: this.settings.emotional_depth,
                humor: this.settings.humor_level,
                point_of_view: this.settings.point_of_view
            },
            content_requirements: {
                include_statistics: this.settings.include_statistics,
                include_quotes: this.settings.include_quotes,
                include_examples: this.settings.include_examples,
                use_anecdotes: this.settings.use_anecdotes,
                add_contrarian_views: this.settings.add_contrarian_views
            },
            seo: {
                optimization_level: this.settings.seo_optimization,
                keyword_density: this.settings.keyword_density
            }
        };
    }
}

// Initialize AI Writer when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.aiWriter = new AIWriter();
    
    // Save preferences when leaving the page
    window.addEventListener('beforeunload', () => {
        window.aiWriter.saveUserPreferences();
    });
});

// Export for global access
if (typeof module !== 'undefined' && module.exports) {
    module.exports = { AIWriter, AdvancedAISettings };
}