class AIChatModule {
    constructor() {
        this.chatHistory = [];
        this.currentAI = 'default';
        this.isTyping = false;
        this.init();
    }

    init() {
        this.setupEventListeners();
        this.loadChatHistory();
    }

    setupEventListeners() {
        // Chat form submission
        document.getElementById('chatForm')?.addEventListener('submit', (e) => {
            this.sendMessage(e);
        });

        // AI agent selection
        document.getElementById('aiAgentSelect')?.addEventListener('change', (e) => {
            this.currentAI = e.target.value;
            this.addSystemMessage(`AI agent changed to ${e.target.options[e.target.selectedIndex].text}`);
        });

        // Clear chat history
        document.getElementById('clearChat')?.addEventListener('click', () => {
            this.clearChat();
        });
    }

    async sendMessage(e) {
        e.preventDefault();
        if (this.isTyping) return;

        const messageInput = document.getElementById('chatMessage');
        const message = messageInput.value.trim();
        
        if (!message) return;

        // Add user message to chat
        this.addMessage('user', message);
        messageInput.value = '';

        // Show typing indicator
        this.showTypingIndicator();

        try {
            const response = await this.getAIResponse(message);
            this.addMessage('ai', response);
        } catch (error) {
            this.addMessage('system', 'Sorry, I encountered an error. Please try again.');
        } finally {
            this.hideTypingIndicator();
        }
    }

    async getAIResponse(message) {
        const formData = new FormData();
        formData.append('action', 'chat');
        formData.append('message', message);
        formData.append('ai_agent', this.currentAI);
        formData.append('chat_history', JSON.stringify(this.chatHistory.slice(-10))); // Last 10 messages

        const response = await fetch('api.php', {
            method: 'POST',
            body: formData
        });

        if (!response.ok) throw new Error('Network error');

        const data = await response.json();
        
        if (data.success) {
            this.chatHistory.push({ role: 'user', content: message });
            this.chatHistory.push({ role: 'assistant', content: data.response });
            this.saveChatHistory();
            return data.response;
        } else {
            throw new Error(data.message);
        }
    }

    addMessage(sender, content) {
        const chatMessages = document.getElementById('chatMessages');
        const messageDiv = document.createElement('div');
        messageDiv.className = `chat-message ${sender}-message`;
        
        const timestamp = new Date().toLocaleTimeString();
        messageDiv.innerHTML = `
            <div class="message-content">
                <div class="message-sender">${sender === 'user' ? 'You' : this.currentAI}</div>
                <div class="message-text">${this.formatMessage(content)}</div>
                <div class="message-time">${timestamp}</div>
            </div>
        `;
        
        chatMessages.appendChild(messageDiv);
        chatMessages.scrollTop = chatMessages.scrollHeight;
    }

    addSystemMessage(content) {
        this.addMessage('system', content);
    }

    formatMessage(content) {
        // Basic formatting for links and line breaks
        return content
            .replace(/\n/g, '<br>')
            .replace(/(https?:\/\/[^\s]+)/g, '<a href="$1" target="_blank">$1</a>');
    }

    showTypingIndicator() {
        this.isTyping = true;
        const chatMessages = document.getElementById('chatMessages');
        const typingDiv = document.createElement('div');
        typingDiv.id = 'typing-indicator';
        typingDiv.className = 'chat-message ai-message typing';
        typingDiv.innerHTML = `
            <div class="message-content">
                <div class="message-sender">${this.currentAI}</div>
                <div class="typing-dots">
                    <span></span>
                    <span></span>
                    <span></span>
                </div>
            </div>
        `;
        chatMessages.appendChild(typingDiv);
        chatMessages.scrollTop = chatMessages.scrollHeight;
    }

    hideTypingIndicator() {
        this.isTyping = false;
        const typingIndicator = document.getElementById('typing-indicator');
        if (typingIndicator) {
            typingIndicator.remove();
        }
    }

    saveChatHistory() {
        localStorage.setItem('ai_chat_history', JSON.stringify(this.chatHistory));
    }

    loadChatHistory() {
        const saved = localStorage.getItem('ai_chat_history');
        if (saved) {
            this.chatHistory = JSON.parse(saved);
            // Display last 10 messages
            this.chatHistory.slice(-10).forEach(msg => {
                this.addMessage(msg.role, msg.content);
            });
        }
    }

    clearChat() {
        this.chatHistory = [];
        localStorage.removeItem('ai_chat_history');
        document.getElementById('chatMessages').innerHTML = '';
        this.addSystemMessage('Chat history cleared');
    }
}