<?php
// ai-core.inc.php - AI Integration Core

class AICore {
    private $db;
    private $gemini_api_url;

    public function __construct($database) {
        $this->db = $database;
        $this->gemini_api_url = GEMINI_API_URL;
    }

    public function generateContent($user, $params) {
        $user_data = $user->getUserData();
        
        // Construct the prompt using advanced techniques
        $prompt = $this->buildAdvancedPrompt($params);
        
        // Get API key (user's key or default)
        $api_key = $this->getUserAPIKey($user_data);
        
        // Call Gemini API
        $response = $this->callGeminiAPI($prompt, $api_key);
        
        if ($response['success']) {
            // Save content to database
            $this->saveGeneratedContent($user_data['id'], $params, $response['content']);
            
            return [
                'success' => true,
                'content' => $response['content']
            ];
        }
        
        return [
            'success' => false,
            'message' => $response['error']
        ];
    }

    private function buildAdvancedPrompt($params) {
        // Advanced prompt engineering for human-like content
        $prompt = "You are an expert content writer specializing in creating highly engaging, human-like content that passes AI detection. 

CONTEXT:
- Topic: {$params['topic']}
- Content Type: {$params['content_type']}
- Tone: {$params['tone']}
- Target Length: {$params['word_count']} words

SPECIFIC INSTRUCTIONS:
1. Write in a completely natural, conversational style that mimics human thought processes
2. Include personal anecdotes and real-world examples where appropriate
3. Present balanced perspectives with occasional contrarian viewpoints
4. Use natural variations in sentence structure and length
5. Incorporate subtle humor, emotion, and personal reflection
6. Include occasional rhetorical questions and conversational asides
7. Use imperfect grammar occasionally to appear more human
8. Vary paragraph length naturally
9. Include transitional phrases that flow naturally
10. Avoid overly perfect or formulaic structure

CONTENT STRUCTURE:
- Start with an engaging hook that addresses the reader directly
- Develop ideas organically rather than using rigid outlines
- Include personal insights and unique perspectives
- End with a thoughtful conclusion that invites further reflection

IMPORTANT: This content must be indistinguishable from human-written text and bypass AI detection tools. Focus on authenticity over perfection.

Now, write the content following these guidelines:";

        return $prompt;
    }

    private function getUserAPIKey($user_data) {
        // If user has their own API key, use it
        if (!empty($user_data['api_key'])) {
            return 'AIzaSy' . $user_data['api_key'];
        }
        
        // Otherwise use default key
        return DEFAULT_GEMINI_KEY;
    }

    private function callGeminiAPI($prompt, $api_key) {
        $data = [
            'contents' => [
                [
                    'parts' => [
                        ['text' => $prompt]
                    ]
                ]
            ],
            'generationConfig' => [
                'temperature' => 0.8,
                'topK' => 40,
                'topP' => 0.95,
                'maxOutputTokens' => 8192,
            ]
        ];

        $url = $this->gemini_api_url . '?key=' . $api_key;

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 60);

        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($http_code === 200) {
            $result = json_decode($response, true);
            
            if (isset($result['candidates'][0]['content']['parts'][0]['text'])) {
                return [
                    'success' => true,
                    'content' => $result['candidates'][0]['content']['parts'][0]['text']
                ];
            }
        }

        return [
            'success' => false,
            'error' => 'AI API request failed. Please check your API key or try again later.'
        ];
    }

    private function saveGeneratedContent($user_id, $params, $content) {
        $word_count = str_word_count($content);
        
        $this->db->query(
            "INSERT INTO ai_content (user_id, title, content, word_count, ai_agent, settings) 
             VALUES (?, ?, ?, ?, ?, ?)",
            [
                $user_id,
                $params['topic'],
                $content,
                $word_count,
                'Default Agent',
                json_encode($params)
            ]
        );
    }
}
?>