<?php
// database.inc.php - Database Connection Class

class Database {
    private $pdo;
    private $error;

    public function __construct() {
        try {
            $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
            $this->pdo = new PDO($dsn, DB_USER, DB_PASS);
            $this->pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $this->pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            $this->error = "Connection failed: " . $e->getMessage();
            error_log($this->error);
        }
    }

    public function getConnection() {
        return $this->pdo;
    }

    public function query($sql, $params = []) {
        try {
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            return $stmt;
        } catch (PDOException $e) {
            error_log("Query Error: " . $e->getMessage());
            return false;
        }
    }

    public function lastInsertId() {
        return $this->pdo->lastInsertId();
    }

    // Initialize database tables
    public function initializeDatabase() {
        $tables = [
            "CREATE TABLE IF NOT EXISTS users (
                id INT AUTO_INCREMENT PRIMARY KEY,
                email VARCHAR(255) UNIQUE NOT NULL,
                password VARCHAR(255) NOT NULL,
                full_name VARCHAR(255),
                is_verified BOOLEAN DEFAULT FALSE,
                verification_code VARCHAR(10),
                api_key VARCHAR(255),
                subscription_tier ENUM('free', 'basic', 'pro') DEFAULT 'free',
                subscription_expires DATETIME,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                last_login DATETIME,
                ip_address VARCHAR(45)
            )",

            "CREATE TABLE IF NOT EXISTS subscriptions (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id INT,
                plan_name VARCHAR(100),
                price DECIMAL(10,2),
                duration_days INT,
                status ENUM('active', 'expired', 'cancelled') DEFAULT 'active',
                starts_at DATETIME,
                expires_at DATETIME,
                payment_reference VARCHAR(255),
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (user_id) REFERENCES users(id)
            )",

            "CREATE TABLE IF NOT EXISTS ai_content (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id INT,
                title VARCHAR(500),
                content LONGTEXT,
                word_count INT,
                ai_agent VARCHAR(100),
                settings JSON,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (user_id) REFERENCES users(id)
            )",

            "CREATE TABLE IF NOT EXISTS ai_agents (
                id INT AUTO_INCREMENT PRIMARY KEY,
                name VARCHAR(100) UNIQUE,
                description TEXT,
                system_prompt LONGTEXT,
                is_default BOOLEAN DEFAULT FALSE,
                created_by INT,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )",

            "CREATE TABLE IF NOT EXISTS payments (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id INT,
                amount DECIMAL(10,2),
                currency VARCHAR(10) DEFAULT 'USD',
                payment_gateway VARCHAR(50),
                transaction_id VARCHAR(255),
                status VARCHAR(50),
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (user_id) REFERENCES users(id)
            )",

            "CREATE TABLE IF NOT EXISTS admin_logs (
                id INT AUTO_INCREMENT PRIMARY KEY,
                admin_id INT,
                action VARCHAR(500),
                target_user_id INT,
                ip_address VARCHAR(45),
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )"
        ];

        foreach ($tables as $table) {
            $this->query($table);
        }

        // Create default admin user if not exists
        $this->createDefaultAdmin();
    }

    private function createDefaultAdmin() {
        $checkAdmin = $this->query("SELECT id FROM users WHERE email = 'aiadmin@system'");
        if (!$checkAdmin->fetch()) {
            $hashedPassword = password_hash('Eni23!ht', PASSWORD_DEFAULT);
            $this->query(
                "INSERT INTO users (email, password, full_name, is_verified, subscription_tier) 
                 VALUES (?, ?, ?, TRUE, 'pro')",
                ['aiadmin@system', $hashedPassword, 'System Administrator']
            );
        }
    }
}
?>