<?php
// payment.inc.php - Unified Payment Handler
class PaymentHandler {
    private $db;
    private $flutterwave_public_key;
    private $flutterwave_secret_key;
    private $stripe_secret_key;

    public function __construct($database) {
        $this->db = $database;
        $this->flutterwave_public_key = FLW_PUBLIC_KEY;
        $this->flutterwave_secret_key = FLW_SECRET_KEY;
        $this->stripe_secret_key = 'sk_test_your_stripe_key_here'; // Add to config.php
    }

    public function initializeFlutterwavePayment($payment_data) {
        // Prepare Flutterwave payment :cite[1]:cite[6]
        $tx_ref = uniqid('flw_');
        $callback_url = BASE_URL . '/payment-webhook.php?gateway=flutterwave';

        $payment_data = [
            'tx_ref' => $tx_ref,
            'amount' => $payment_data['amount'],
            'currency' => $payment_data['currency'] ?? 'USD',
            'payment_options' => 'card,account,banktransfer',
            'redirect_url' => $callback_url,
            'customer' => [
                'email' => $payment_data['email'],
                'name' => $payment_data['customer_name']
            ],
            'customizations' => [
                'title' => APP_NAME,
                'description' => $payment_data['description'] ?? 'Subscription Payment'
            ]
        ];

        // Initialize cURL request to Flutterwave
        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => 'https://api.flutterwave.com/v3/payments',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'POST',
            CURLOPT_POSTFIELDS => json_encode($payment_data),
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->flutterwave_secret_key,
                'Content-Type: application/json'
            ],
        ]);

        $response = curl_exec($curl);
        $err = curl_error($curl);
        curl_close($curl);

        if ($err) {
            return ['success' => false, 'message' => 'Payment initialization failed'];
        }

        $response_data = json_decode($response, true);
        if ($response_data['status'] === 'success') {
            // Save payment record to database
            $this->savePaymentRecord([
                'user_id' => $payment_data['user_id'],
                'transaction_ref' => $tx_ref,
                'amount' => $payment_data['amount'],
                'currency' => $payment_data['currency'],
                'gateway' => 'flutterwave',
                'status' => 'pending'
            ]);

            return [
                'success' => true, 
                'payment_url' => $response_data['data']['link'],
                'transaction_ref' => $tx_ref
            ];
        }

        return ['success' => false, 'message' => $response_data['message'] ?? 'Payment failed'];
    }

    public function initializeStripePayment($payment_data) {
        // Initialize Stripe payment :cite[2]:cite[7]
        require_once 'vendor/autoload.php'; // Ensure Stripe PHP SDK is installed

        \Stripe\Stripe::setApiKey($this->stripe_secret_key);

        try {
            $checkout_session = \Stripe\Checkout\Session::create([
                'payment_method_types' => ['card'],
                'line_items' => [[
                    'price_data' => [
                        'currency' => $payment_data['currency'] ?? 'usd',
                        'product_data' => [
                            'name' => $payment_data['description'] ?? 'Subscription',
                        ],
                        'unit_amount' => $payment_data['amount'] * 100, // Amount in cents
                    ],
                    'quantity' => 1,
                ]],
                'mode' => 'payment',
                'success_url' => BASE_URL . '/payment-success.php?session_id={CHECKOUT_SESSION_ID}',
                'cancel_url' => BASE_URL . '/payment-cancel.php',
                'customer_email' => $payment_data['email'],
                'metadata' => [
                    'user_id' => $payment_data['user_id'],
                    'plan_id' => $payment_data['plan_id']
                ]
            ]);

            // Save payment record
            $this->savePaymentRecord([
                'user_id' => $payment_data['user_id'],
                'transaction_ref' => $checkout_session->id,
                'amount' => $payment_data['amount'],
                'currency' => $payment_data['currency'],
                'gateway' => 'stripe',
                'status' => 'pending'
            ]);

            return [
                'success' => true,
                'payment_url' => $checkout_session->url,
                'session_id' => $checkout_session->id
            ];

        } catch (Exception $e) {
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    private function savePaymentRecord($payment_data) {
        $this->db->query(
            "INSERT INTO payments (user_id, transaction_ref, amount, currency, gateway, status, created_at) 
             VALUES (?, ?, ?, ?, ?, ?, NOW())",
            [
                $payment_data['user_id'],
                $payment_data['transaction_ref'],
                $payment_data['amount'],
                $payment_data['currency'],
                $payment_data['gateway'],
                $payment_data['status']
            ]
        );
    }

    public function verifyFlutterwavePayment($transaction_ref) {
        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => "https://api.flutterwave.com/v3/transactions/{$transaction_ref}/verify",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => 'GET',
            CURLOPT_HTTPHEADER => [
                'Authorization: Bearer ' . $this->flutterwave_secret_key
            ],
        ]);

        $response = curl_exec($curl);
        $err = curl_error($curl);
        curl_close($curl);

        if (!$err) {
            $response_data = json_decode($response, true);
            if ($response_data['status'] === 'success' && $response_data['data']['status'] === 'successful') {
                return ['success' => true, 'data' => $response_data['data']];
            }
        }

        return ['success' => false];
    }
}
?>