<?php
class SubscriptionManager {
    private $db;

    public function __construct($database) {
        $this->db = $database;
    }

    public function createSubscriptionPlan($plan_data) {
        $this->db->query(
            "INSERT INTO subscription_plans (name, description, price, duration_days, features, max_articles, max_ai_chats, can_publish_wordpress) 
             VALUES (?, ?, ?, ?, ?, ?, ?, ?)",
            [
                $plan_data['name'],
                $plan_data['description'],
                $plan_data['price'],
                $plan_data['duration_days'],
                json_encode($plan_data['features']),
                $plan_data['max_articles'],
                $plan_data['max_ai_chats'],
                $plan_data['can_publish_wordpress'] ?? 0
            ]
        );
        return $this->db->lastInsertId();
    }

    public function activateUserSubscription($user_id, $plan_id, $transaction_ref) {
        // Get plan details
        $plan = $this->db->query("SELECT * FROM subscription_plans WHERE id = ?", [$plan_id])->fetch();

        // Calculate expiry date
        $expires_at = date('Y-m-d H:i:s', strtotime("+{$plan['duration_days']} days"));

        // Create subscription
        $this->db->query(
            "INSERT INTO subscriptions (user_id, plan_id, status, starts_at, expires_at, transaction_ref) 
             VALUES (?, ?, 'active', NOW(), ?, ?)",
            [$user_id, $plan_id, $expires_at, $transaction_ref]
        );

        // Update user tier
        $this->db->query(
            "UPDATE users SET subscription_tier = ?, subscription_expires = ? WHERE id = ?",
            [$plan['name'], $expires_at, $user_id]
        );

        return true;
    }

    public function checkSubscriptionStatus($user_id) {
        $subscription = $this->db->query(
            "SELECT s.*, p.name as plan_name, p.features, p.max_articles, p.max_ai_chats 
             FROM subscriptions s 
             JOIN subscription_plans p ON s.plan_id = p.id 
             WHERE s.user_id = ? AND s.status = 'active' AND s.expires_at > NOW() 
             ORDER BY s.created_at DESC LIMIT 1",
            [$user_id]
        )->fetch();

        return $subscription ?: false;
    }

    public function getUsageStatistics($user_id) {
        $today = date('Y-m-d');
        $month_start = date('Y-m-01');

        $usage = $this->db->query(
            "SELECT 
                COUNT(*) as total_articles,
                SUM(CASE WHERE DATE(created_at) = ? THEN 1 ELSE 0 END) as today_articles,
                SUM(CASE WHERE created_at >= ? THEN 1 ELSE 0 END) as month_articles
             FROM ai_content 
             WHERE user_id = ?",
            [$today, $month_start, $user_id]
        )->fetch();

        return $usage;
    }
}
?>