<?php
// user.inc.php - User Management Class

class User {
    private $db;
    private $user_id;
    private $user_data;

    public function __construct($database) {
        $this->db = $database;
        $this->checkAuthentication();
    }

    private function checkAuthentication() {
        if (isset($_SESSION['user_id'])) {
            $this->user_id = $_SESSION['user_id'];
            $this->loadUserData();
        }
    }

    private function loadUserData() {
        $stmt = $this->db->query("SELECT * FROM users WHERE id = ?", [$this->user_id]);
        $this->user_data = $stmt->fetch();
    }

    public function isLoggedIn() {
        return !empty($this->user_data);
    }

    public function register($email, $password, $full_name) {
        // Validate Gmail address
        if (!filter_var($email, FILTER_VALIDATE_EMAIL) || !str_contains($email, '@gmail.com')) {
            return ['success' => false, 'message' => 'Only Gmail addresses are allowed'];
        }

        // Check if email exists
        $stmt = $this->db->query("SELECT id FROM users WHERE email = ?", [$email]);
        if ($stmt->fetch()) {
            return ['success' => false, 'message' => 'Email already registered'];
        }

        // Generate verification code
        $verification_code = sprintf("%06d", mt_rand(1, 999999));

        // Create user
        $hashed_password = password_hash($password, PASSWORD_DEFAULT);
        $result = $this->db->query(
            "INSERT INTO users (email, password, full_name, verification_code, ip_address) 
             VALUES (?, ?, ?, ?, ?)",
            [$email, $hashed_password, $full_name, $verification_code, $_SERVER['REMOTE_ADDR']]
        );

        if ($result) {
            // Send verification email
            $this->sendVerificationEmail($email, $verification_code);
            return ['success' => true, 'message' => 'Registration successful. Please check your email for verification code.'];
        }

        return ['success' => false, 'message' => 'Registration failed'];
    }

    public function verifyAccount($email, $code) {
        $stmt = $this->db->query("SELECT id FROM users WHERE email = ? AND verification_code = ?", [$email, $code]);
        $user = $stmt->fetch();

        if ($user) {
            $this->db->query("UPDATE users SET is_verified = TRUE, verification_code = NULL WHERE id = ?", [$user['id']]);
            return ['success' => true, 'message' => 'Account verified successfully'];
        }

        return ['success' => false, 'message' => 'Invalid verification code'];
    }

    public function login($email, $password) {
        $stmt = $this->db->query("SELECT * FROM users WHERE email = ?", [$email]);
        $user = $stmt->fetch();

        if ($user && password_verify($password, $user['password'])) {
            if (!$user['is_verified']) {
                return ['success' => false, 'message' => 'Please verify your email first'];
            }

            $_SESSION['user_id'] = $user['id'];
            $this->user_id = $user['id'];
            $this->user_data = $user;

            // Update last login
            $this->db->query("UPDATE users SET last_login = NOW() WHERE id = ?", [$this->user_id]);

            return ['success' => true, 'message' => 'Login successful'];
        }

        return ['success' => false, 'message' => 'Invalid email or password'];
    }

    public function logout() {
        session_destroy();
        $this->user_id = null;
        $this->user_data = null;
    }

    public function getUserData() {
        return $this->user_data;
    }

    public function hasActiveSubscription() {
        if (!$this->isLoggedIn()) return false;

        $stmt = $this->db->query(
            "SELECT COUNT(*) as active FROM subscriptions 
             WHERE user_id = ? AND status = 'active' AND expires_at > NOW()",
            [$this->user_id]
        );
        $result = $stmt->fetch();
        return $result['active'] > 0;
    }

    public function getRemainingFreeArticles() {
        if (!$this->isLoggedIn()) return 0;

        $stmt = $this->db->query(
            "SELECT COUNT(*) as used FROM ai_content 
             WHERE user_id = ? AND created_at > DATE_SUB(NOW(), INTERVAL 24 HOUR)",
            [$this->user_id]
        );
        $result = $stmt->fetch();
        return max(0, FREE_TRIAL_ARTICLES - $result['used']);
    }

    private function sendVerificationEmail($email, $code) {
        // Basic email sending implementation
        $subject = APP_NAME . " - Verify Your Account";
        $message = "Your verification code is: " . $code;
        $headers = "From: " . SMTP_FROM . "\r\n";

        // In a real implementation, use PHPMailer or similar
        @mail($email, $subject, $message, $headers);
    }
}
?>