<?php
class WordPressPublisher {
    private $db;

    public function __construct($database) {
        $this->db = $database;
    }

    public function generateWordPressPlugin($user_id) {
        // Generate a custom WordPress plugin for the user
        $api_key = $this->generateAPIKey($user_id);
        
        $plugin_code = <<<EOT
<?php
/**
 * Plugin Name: Uncensored AI Writer Publisher
 * Description: Auto-publish content from Uncensored AI Writer
 * Version: 1.0
 * Author: VinesubTech
 */

// Prevent direct access
defined('ABSPATH') or die('No script kiddies please!');

class UncensoredAIPublisher {
    private \$api_key = '{$api_key}';
    private \$api_url = 'https://your-domain.com/api.php';

    public function __construct() {
        add_action('rest_api_init', [\$this, 'register_routes']);
        add_action('admin_menu', [\$this, 'add_admin_menu']);
    }

    public function register_routes() {
        register_rest_route('uncensored-ai/v1', '/publish', [
            'methods' => 'POST',
            'callback' => [\$this, 'handle_publish_request'],
            'permission_callback' => [\$this, 'verify_api_key']
        ]);
    }

    public function verify_api_key(\\WP_REST_Request \$request) {
        \$headers = \$request->get_headers();
        \$api_key = isset(\$headers['x-api-key']) ? \$headers['x-api-key'][0] : '';
        return \$api_key === \$this->api_key;
    }

    public function handle_publish_request(\\WP_REST_Request \$request) {
        \$params = \$request->get_params();
        
        \$post_data = [
            'post_title' => wp_strip_all_tags(\$params['title']),
            'post_content' => \$params['content'],
            'post_status' => \$params['status'] ?? 'draft',
            'post_type' => 'post',
            'post_category' => \$params['categories'] ?? []
        ];

        \$post_id = wp_insert_post(\$post_data);

        if (\$post_id) {
            // Handle featured image if provided
            if (!empty(\$params['featured_image'])) {
                \$this->upload_featured_image(\$params['featured_image'], \$post_id);
            }

            return ['success' => true, 'post_id' => \$post_id, 'url' => get_permalink(\$post_id)];
        }

        return ['success' => false, 'error' => 'Failed to create post'];
    }

    private function upload_featured_image(\$image_url, \$post_id) {
        require_once(ABSPATH . 'wp-admin/includes/image.php');
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/media.php');

        \$media = media_sideload_image(\$image_url, \$post_id);
        
        if (!is_wp_error(\$media)) {
            \$attachments = get_posts([
                'post_type' => 'attachment',
                'post_parent' => \$post_id,
                'numberposts' => 1,
                'orderby' => 'ID', 
                'order' => 'DESC'
            ]);

            if (!empty(\$attachments)) {
                set_post_thumbnail(\$post_id, \$attachments[0]->ID);
            }
        }
    }

    public function add_admin_menu() {
        add_options_page(
            'Uncensored AI Settings',
            'Uncensored AI',
            'manage_options',
            'uncensored-ai-settings',
            [\$this, 'admin_page']
        );
    }

    public function admin_page() {
        echo '<div class="wrap"><h1>Uncensored AI Publisher</h1>';
        echo '<p>Your API Key: <code>' . \$this->api_key . '</code></p>';
        echo '<p>Use this key to authenticate publishing requests from your Uncensored AI Writer account.</p>';
        echo '</div>';
    }
}

new UncensoredAIPublisher();
EOT;

        return $plugin_code;
    }

    private function generateAPIKey($user_id) {
        $api_key = 'uai_' . bin2hex(random_bytes(32));
        
        $this->db->query(
            "UPDATE users SET wordpress_api_key = ? WHERE id = ?",
            [$api_key, $user_id]
        );

        return $api_key;
    }

    public function publishToWordPress($user_id, $content_data, $wordpress_url) {
        $user = $this->db->query(
            "SELECT wordpress_api_key FROM users WHERE id = ?",
            [$user_id]
        )->fetch();

        if (!$user || empty($user['wordpress_api_key'])) {
            return ['success' => false, 'message' => 'WordPress API key not configured'];
        }

        // Prepare publish request
        $payload = [
            'title' => $content_data['title'],
            'content' => $content_data['content'],
            'status' => $content_data['status'] ?? 'draft',
            'categories' => $content_data['categories'] ?? []
        ];

        // Send to WordPress site
        $response = $this->makeWordPressRequest($wordpress_url, $payload, $user['wordpress_api_key']);
        return $response;
    }

    private function makeWordPressRequest($wordpress_url, $payload, $api_key) {
        $url = rtrim($wordpress_url, '/') . '/wp-json/uncensored-ai/v1/publish';

        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($payload),
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'X-API-Key: ' . $api_key
            ]
        ]);

        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($http_code === 200) {
            return json_decode($response, true);
        }

        return ['success' => false, 'message' => 'WordPress publishing failed'];
    }
}
?>