<?php
// index.php - Main Application Entry Point

require_once 'config.php';
require_once 'inc/database.inc.php';
require_once 'inc/user.inc.php';

$db = new Database();
$user = new User($db);

// Initialize database if not already done
$db->initializeDatabase();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo APP_NAME; ?></title>
    <link rel="stylesheet" href="assets/css/style.css">
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
</head>
<body>
    <div id="app">
        <!-- Navigation -->
        <nav class="navbar">
            <div class="nav-container">
                <div class="nav-brand">
                    <h1><?php echo APP_NAME; ?></h1>
                </div>
                <div class="nav-menu" id="navMenu">
                    <?php if ($user->isLoggedIn()): ?>
                        <a href="#dashboard" class="nav-link">Dashboard</a>
                        <a href="#ai-writer" class="nav-link">AI Writer</a>
                        <a href="#ai-chat" class="nav-link">AI Chat</a>
                        <a href="#subscription" class="nav-link">Subscription</a>
                        <a href="#" id="logoutBtn" class="nav-link">Logout</a>
                    <?php else: ?>
                        <a href="#login" class="nav-link">Login</a>
                        <a href="#register" class="nav-link">Register</a>
                    <?php endif; ?>
                </div>
                <div class="nav-actions">
                    <button id="themeToggle" class="theme-toggle">🌓</button>
                    <button class="nav-toggle" id="navToggle">☰</button>
                </div>
            </div>
        </nav>

        <!-- Main Content -->
        <main class="main-content">
            <?php if (!$user->isLoggedIn()): ?>
                <!-- Authentication Sections -->
                <section id="login" class="section active">
                    <div class="auth-container">
                        <h2>Login to Your Account</h2>
                        <form id="loginForm" class="auth-form">
                            <div class="form-group">
                                <label>Email</label>
                                <input type="email" name="email" required>
                            </div>
                            <div class="form-group">
                                <label>Password</label>
                                <input type="password" name="password" required>
                            </div>
                            <button type="submit" class="btn-primary">Login</button>
                        </form>
                    </div>
                </section>

                <section id="register" class="section">
                    <div class="auth-container">
                        <h2>Create New Account</h2>
                        <form id="registerForm" class="auth-form">
                            <div class="form-group">
                                <label>Full Name</label>
                                <input type="text" name="full_name" required>
                            </div>
                            <div class="form-group">
                                <label>Email (Gmail only)</label>
                                <input type="email" name="email" required>
                            </div>
                            <div class="form-group">
                                <label>Password</label>
                                <input type="password" name="password" required>
                            </div>
                            <button type="submit" class="btn-primary">Register</button>
                        </form>
                    </div>
                </section>
            <?php else: ?>
                <!-- Dashboard -->
                <section id="dashboard" class="section active">
                    <div class="dashboard-container">
                        <h2>Welcome, <?php echo htmlspecialchars($user->getUserData()['full_name']); ?></h2>
                        
                        <div class="stats-grid">
                            <div class="stat-card">
                                <h3>Subscription Status</h3>
                                <p class="stat-value"><?php echo $user->hasActiveSubscription() ? 'Active' : 'Inactive'; ?></p>
                            </div>
                            <div class="stat-card">
                                <h3>Free Articles Remaining</h3>
                                <p class="stat-value"><?php echo $user->getRemainingFreeArticles(); ?></p>
                            </div>
                        </div>
                    </div>
                </section>

                <!-- AI Writer Section -->
                <section id="ai-writer" class="section">
                    <div class="ai-writer-container">
                        <h2>AI Content Generator</h2>
                        <div class="ai-writer-grid">
                            <!-- Input Form -->
                            <div class="input-panel">
                                <form id="aiWriterForm">
                                    <div class="form-group">
                                        <label>Content Type</label>
                                        <select name="content_type" required>
                                            <option value="article">Article</option>
                                            <option value="blog_post">Blog Post</option>
                                            <option value="listicle">Listicle</option>
                                            <option value="product_review">Product Review</option>
                                        </select>
                                    </div>

                                    <div class="form-group">
                                        <label>Tone/Voice</label>
                                        <select name="tone" required>
                                            <option value="professional">Professional</option>
                                            <option value="casual">Casual</option>
                                            <option value="authoritative">Authoritative</option>
                                            <option value="sarcastic">Sarcastic</option>
                                        </select>
                                    </div>

                                    <div class="form-group">
                                        <label>Word Count</label>
                                        <input type="number" name="word_count" value="1000" min="100" max="5000" required>
                                    </div>

                                    <div class="form-group">
                                        <label>Topic/Idea</label>
                                        <textarea name="topic" placeholder="Describe what you want to write about..." required></textarea>
                                    </div>

                                    <button type="submit" class="btn-primary">Generate Content</button>
                                </form>
                            </div>

                            <!-- Output Panel -->
                            <div class="output-panel">
                                <div class="output-header">
                                    <h3>Generated Content</h3>
                                    <div class="output-actions">
                                        <button id="copyContent" class="btn-secondary">Copy</button>
                                        <button id="downloadContent" class="btn-secondary">Download</button>
                                    </div>
                                </div>
                                <div id="contentOutput" class="content-output">
                                    <p class="placeholder">Content will appear here...</p>
                                </div>
                            </div>
                        </div>
                    </div>
                </section>
            <?php endif; ?>
        </main>
    </div>

    <!-- JavaScript -->
    <script src="assets/js/app.js"></script>
    <?php if ($user->isLoggedIn()): ?>
        <script src="assets/js/ai-writer.js"></script>
    <?php endif; ?>
</body>
</html>