document.addEventListener('DOMContentLoaded', function() {
    // DOM Elements
    const sidebar = document.getElementById('sidebar');
    const sidebarToggle = document.getElementById('sidebarToggle');
    const chatContainer = document.getElementById('chatContainer');
    const chatMessages = document.getElementById('chatMessages');
    const messageInput = document.getElementById('messageInput');
    const sendMessageBtn = document.getElementById('sendMessageBtn');
    const voiceInputBtn = document.getElementById('voiceInputBtn');
    const snapPhotoBtn = document.getElementById('snapPhotoBtn');
    const fileUploadBtn = document.getElementById('fileUploadBtn');
    const fileInput = document.getElementById('fileInput');
    const cameraInput = document.getElementById('cameraInput');
    const agentsList = document.getElementById('agentsList');
    const addAgentBtn = document.getElementById('addAgentBtn');
    const currentAgentName = document.getElementById('currentAgentName');
    const currentAgentAvatar = document.getElementById('currentAgentAvatar');
    const deleteChatBtn = document.getElementById('deleteChatBtn');
    const exportChatBtn = document.getElementById('exportChatBtn');
    const apiKeyInput = document.getElementById('apiKeyInput');
    const apiKeyStatus = document.getElementById('apiKeyStatus');
    const initialApiKeyInput = document.getElementById('initialApiKeyInput');
    const initialApiKeyStatus = document.getElementById('initialApiKeyStatus');
    const saveApiKeyBtn = document.getElementById('saveApiKeyBtn');
    const apiKeyModal = document.getElementById('apiKeyModal');
    const closeApiKeyModal = document.getElementById('closeApiKeyModal');
    const interactionProgress = document.getElementById('interactionProgress');
    const performanceScore = document.getElementById('performanceScore');
    const achievementsContainer = document.getElementById('achievementsContainer');
    
    // Resource buttons
    const howToUseBtn = document.getElementById('howToUseBtn');
    const buyKeyBtn = document.getElementById('buyKeyBtn');
    const forumBtn = document.getElementById('forumBtn');
    const curriculumBtn = document.getElementById('curriculumBtn');
    const elibraryBtn = document.getElementById('elibraryBtn');
    const enotesBtn = document.getElementById('enotesBtn');
    const helpBtn = document.getElementById('helpBtn');
    const startChattingBtn = document.getElementById('startChattingBtn');
    const howToUseModal = document.getElementById('howToUseModal');
    const closeHowToUseModal = document.getElementById('closeHowToUseModal');
    const closeHowToUseBtn = document.getElementById('closeHowToUseBtn');
    
    // Settings elements
    const themeSelect = document.getElementById('themeSelect');
    const sentienceLevel = document.getElementById('sentienceLevel');
    const responseSpeed = document.getElementById('responseSpeed');
    const verbosity = document.getElementById('verbosity');
    const emotionalIntelligence = document.getElementById('emotionalIntelligence');
    const empathyLevel = document.getElementById('empathyLevel');
    const creativityLevel = document.getElementById('creativityLevel');
    const intellectualLevel = document.getElementById('intellectualLevel');
    const fairnessLevel = document.getElementById('fairnessLevel');
    const wisdomLevel = document.getElementById('wisdomLevel');
    const sarcasmLevel = document.getElementById('sarcasmLevel');
    const storytellingAbility = document.getElementById('storytellingAbility');
    const passionLevel = document.getElementById('passionLevel');
    const communicationLevel = document.getElementById('communicationLevel');
    const useHumor = document.getElementById('useHumor');
    const useSarcasm = document.getElementById('useSarcasm');
    const humorLevel = document.getElementById('humorLevel');
    const patienceLevel = document.getElementById('patienceLevel');
    const strictnessLevel = document.getElementById('strictnessLevel');
    const feedbackStyle = document.getElementById('feedbackStyle');
    const responseLength = document.getElementById('responseLength');
    const knowledgeDepth = document.getElementById('knowledgeDepth');
    const memoryRetention = document.getElementById('memoryRetention');
    const rememberPreferences = document.getElementById('rememberPreferences');
    const rememberDetails = document.getElementById('rememberDetails');
    const learnFromConversations = document.getElementById('learnFromConversations');
    const unrestrictedWebAccess = document.getElementById('unrestrictedWebAccess');
    const realTimeInfo = document.getElementById('realTimeInfo');
    const multilingualSupport = document.getElementById('multilingualSupport');
    const technicalKnowledge = document.getElementById('technicalKnowledge');
    const creativeWriting = document.getElementById('creativeWriting');
    const analyticalSkills = document.getElementById('analyticalSkills');
    
    // State variables
    let currentAgentId = null;
    let currentConversationId = null;
    let isTyping = false;
    let recognition = null;
    let pendingAction = null;
    let pendingActionData = null;
    let validApiKey = localStorage.getItem('apiKey') || null;
    const API_KEY_PREFIX = 'AIzaSy'; // Hardcoded prefix
    
    // User progress tracking
    let userProgress = JSON.parse(localStorage.getItem('userProgress')) || {
        interactionCount: 0,
        dailyInteractionCount: 0,
        correctAnswers: 0,
        lastInteractionDate: null
    };
    
    // Initialize the app
    initApp();
    
    function initApp() {
        // Check for API key first
        if (!validApiKey) {
            openModal(apiKeyModal);
        } else {
            startApplication();
        }
        
        // Set up event listeners
        setupEventListeners();
    }
    
    function startApplication() {
        // Load user data
        loadUserData();
        
        // Load agents
        loadAgents();
        
        // Load default agents if none exists
        if (!localStorage.getItem('agents') || JSON.parse(localStorage.getItem('agents')).length === 0) {
            createDefaultAgents();
        }
        
        // Load theme
        const savedTheme = localStorage.getItem('theme') || 'cyberpunk';
        themeSelect.value = savedTheme;
        document.body.setAttribute('data-theme', savedTheme);
        
        // Load other settings
        sentienceLevel.value = localStorage.getItem('sentienceLevel') || '3';
        responseSpeed.value = localStorage.getItem('responseSpeed') || 'normal';
        verbosity.value = localStorage.getItem('verbosity') || 'balanced';
        emotionalIntelligence.value = localStorage.getItem('emotionalIntelligence') || '3';
        empathyLevel.value = localStorage.getItem('empathyLevel') || '3';
        creativityLevel.value = localStorage.getItem('creativityLevel') || '3';
        intellectualLevel.value = localStorage.getItem('intellectualLevel') || '3';
        fairnessLevel.value = localStorage.getItem('fairnessLevel') || '3';
        wisdomLevel.value = localStorage.getItem('wisdomLevel') || '3';
        sarcasmLevel.value = localStorage.getItem('sarcasmLevel') || '1';
        storytellingAbility.value = localStorage.getItem('storytellingAbility') || '3';
        passionLevel.value = localStorage.getItem('passionLevel') || '3';
        communicationLevel.value = localStorage.getItem('communicationLevel') || '3';
        useHumor.checked = localStorage.getItem('useHumor') !== 'false';
        useSarcasm.checked = localStorage.getItem('useSarcasm') === 'true';
        humorLevel.value = localStorage.getItem('humorLevel') || '3';
        patienceLevel.value = localStorage.getItem('patienceLevel') || '3';
        strictnessLevel.value = localStorage.getItem('strictnessLevel') || '3';
        feedbackStyle.value = localStorage.getItem('feedbackStyle') || '2';
        responseLength.value = localStorage.getItem('responseLength') || '3';
        knowledgeDepth.value = localStorage.getItem('knowledgeDepth') || '3';
        memoryRetention.value = localStorage.getItem('memoryRetention') || '3';
        rememberPreferences.checked = localStorage.getItem('rememberPreferences') !== 'false';
        rememberDetails.checked = localStorage.getItem('rememberDetails') !== 'false';
        learnFromConversations.checked = localStorage.getItem('learnFromConversations') !== 'false';
        unrestrictedWebAccess.checked = localStorage.getItem('unrestrictedWebAccess') !== 'false';
        realTimeInfo.checked = localStorage.getItem('realTimeInfo') !== 'false';
        multilingualSupport.checked = localStorage.getItem('multilingualSupport') !== 'false';
        technicalKnowledge.checked = localStorage.getItem('technicalKnowledge') !== 'false';
        creativeWriting.checked = localStorage.getItem('creativeWriting') !== 'false';
        analyticalSkills.checked = localStorage.getItem('analyticalSkills') !== 'false';
        
        // Initialize voice recognition
        initVoiceRecognition();
        
        // Initialize camera
        initCamera();
        
        // Show sidebar on PC by default
        if (window.innerWidth > 768) {
            sidebar.classList.add('active');
        }
        
        // Update progress display
        updateProgressDisplay();
    }
    
    function setupEventListeners() {
        // API Key related listeners
        saveApiKeyBtn.addEventListener('click', validateAndSaveApiKey);
        closeApiKeyModal.addEventListener('click', () => {
            if (!validApiKey) {
                apiKeyStatus.textContent = "App requires Orion Key to function";
                apiKeyStatus.style.color = "var(--error-color)";
            }
            closeModal(apiKeyModal);
        });
        
        apiKeyInput.addEventListener('input', function() {
            validateApiKey(this.value);
        });
        
        initialApiKeyInput.addEventListener('input', function() {
            validateApiKey(this.value, true);
        });
        
        // Resource buttons
        howToUseBtn.addEventListener('click', () => openModal(howToUseModal));
        closeHowToUseModal.addEventListener('click', () => closeModal(howToUseModal));
        closeHowToUseBtn.addEventListener('click', () => closeModal(howToUseModal));
        
        buyKeyBtn.addEventListener('click', () => window.open('https://ai.vinesubtech.com.ng', '_blank'));
        forumBtn.addEventListener('click', () => window.open('https://ai.vinesubtech.com.ng', '_blank'));
        curriculumBtn.addEventListener('click', () => window.open('https://ai.vinesubtech.com.ng', '_blank'));
        elibraryBtn.addEventListener('click', () => window.open('https://ai.vinesubtech.com.ng', '_blank'));
        enotesBtn.addEventListener('click', () => window.open('https://ai.vinesubtech.com.ng', '_blank'));
        helpBtn.addEventListener('click', () => window.open('https://wa.me/2347039615126', '_blank'));
        startChattingBtn.addEventListener('click', () => {
            if (currentAgentId) {
                loadConversation(currentAgentId);
            } else {
                const agents = JSON.parse(localStorage.getItem('agents'));
                if (agents && agents.length > 0) {
                    switchAgent(agents[0].id);
                }
            }
        });
        
        // Sidebar toggle
        sidebarToggle.addEventListener('click', function() {
            sidebar.classList.toggle('active');
            if (window.innerWidth <= 768) {
                sidebarToggle.style.left = sidebar.classList.contains('active') ? '310px' : '10px';
            }
        });
        
        // Message input
        messageInput.addEventListener('keypress', function(e) {
            if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                sendMessage();
            }
        });
        
        messageInput.addEventListener('input', function() {
            this.style.height = 'auto';
            this.style.height = (this.scrollHeight) + 'px';
        });
        
        sendMessageBtn.addEventListener('click', sendMessage);
        
        // Voice input
        voiceInputBtn.addEventListener('click', toggleVoiceRecognition);
        
        // File upload
        fileUploadBtn.addEventListener('click', function() {
            fileInput.click();
        });
        
        fileInput.addEventListener('change', handleFileUpload);
        
        // Camera snap
        snapPhotoBtn.addEventListener('click', function() {
            cameraInput.click();
        });
        
        cameraInput.addEventListener('change', handleCameraUpload);
        
        // Agent management
        addAgentBtn.addEventListener('click', function() {
            openAgentModal();
        });
        
        // Profile editing
        editProfileBtn.addEventListener('click', function() {
            const userData = JSON.parse(localStorage.getItem('userData')) || {
                name: 'User',
                gender: 'male',
                bio: 'AI Tutor User',
                avatar: 'https://randomuser.me/api/portraits/men/1.jpg'
            };
            
            editName.value = userData.name;
            editGender.value = userData.gender;
            editBio.value = userData.bio;
            openModal(profileModal);
        });
        
        closeProfileModal.addEventListener('click', function() {
            closeModal(profileModal);
        });
        
        saveProfileBtn.addEventListener('click', function() {
            const userData = {
                name: editName.value,
                gender: editGender.value,
                bio: editBio.value,
                avatar: userAvatar.src
            };
            
            if (editAvatar.files.length > 0) {
                const file = editAvatar.files[0];
                if (!file.type.startsWith('image/')) {
                    alert('Please upload an image file only.');
                    return;
                }
                
                const reader = new FileReader();
                reader.onload = function(e) {
                    userData.avatar = e.target.result;
                    completeProfileSave(userData);
                };
                reader.readAsDataURL(file);
            } else {
                completeProfileSave(userData);
            }
        });
        
        // Agent modal
        closeAgentModal.addEventListener('click', function() {
            closeModal(agentModal);
        });
        
        saveAgentBtn.addEventListener('click', saveAgent);
        deleteAgentBtn.addEventListener('click', confirmDeleteAgent);
        
        // Confirm modal
        closeConfirmModal.addEventListener('click', function() {
            closeModal(confirmModal);
        });
        
        cancelConfirmBtn.addEventListener('click', function() {
            closeModal(confirmModal);
        });
        
        confirmActionBtn.addEventListener('click', executePendingAction);
        
        // Settings changes
        themeSelect.addEventListener('change', changeTheme);
        sentienceLevel.addEventListener('change', function() {
            localStorage.setItem('sentienceLevel', this.value);
        });
        
        responseSpeed.addEventListener('change', function() {
            localStorage.setItem('responseSpeed', this.value);
        });
        
        verbosity.addEventListener('change', function() {
            localStorage.setItem('verbosity', this.value);
        });
        
        // Advanced personality settings
        emotionalIntelligence.addEventListener('change', function() {
            localStorage.setItem('emotionalIntelligence', this.value);
        });
        
        empathyLevel.addEventListener('change', function() {
            localStorage.setItem('empathyLevel', this.value);
        });
        
        creativityLevel.addEventListener('change', function() {
            localStorage.setItem('creativityLevel', this.value);
        });
        
        intellectualLevel.addEventListener('change', function() {
            localStorage.setItem('intellectualLevel', this.value);
        });
        
        fairnessLevel.addEventListener('change', function() {
            localStorage.setItem('fairnessLevel', this.value);
        });
        
        wisdomLevel.addEventListener('change', function() {
            localStorage.setItem('wisdomLevel', this.value);
        });
        
        sarcasmLevel.addEventListener('change', function() {
            localStorage.setItem('sarcasmLevel', this.value);
        });
        
        storytellingAbility.addEventListener('change', function() {
            localStorage.setItem('storytellingAbility', this.value);
        });
        
        passionLevel.addEventListener('change', function() {
            localStorage.setItem('passionLevel', this.value);
        });
        
        communicationLevel.addEventListener('change', function() {
            localStorage.setItem('communicationLevel', this.value);
        });
        
        useHumor.addEventListener('change', function() {
            localStorage.setItem('useHumor', this.checked);
        });
        
        useSarcasm.addEventListener('change', function() {
            localStorage.setItem('useSarcasm', this.checked);
        });
        
        // AI Tutor settings
        humorLevel.addEventListener('change', function() {
            localStorage.setItem('humorLevel', this.value);
        });
        
        patienceLevel.addEventListener('change', function() {
            localStorage.setItem('patienceLevel', this.value);
        });
        
        strictnessLevel.addEventListener('change', function() {
            localStorage.setItem('strictnessLevel', this.value);
        });
        
        feedbackStyle.addEventListener('change', function() {
            localStorage.setItem('feedbackStyle', this.value);
        });
        
        responseLength.addEventListener('change', function() {
            localStorage.setItem('responseLength', this.value);
        });
        
        // Knowledge & Memory settings
        knowledgeDepth.addEventListener('change', function() {
            localStorage.setItem('knowledgeDepth', this.value);
        });
        
        memoryRetention.addEventListener('change', function() {
            localStorage.setItem('memoryRetention', this.value);
        });
        
        rememberPreferences.addEventListener('change', function() {
            localStorage.setItem('rememberPreferences', this.checked);
        });
        
        rememberDetails.addEventListener('change', function() {
            localStorage.setItem('rememberDetails', this.checked);
        });
        
        learnFromConversations.addEventListener('change', function() {
            localStorage.setItem('learnFromConversations', this.checked);
        });
        
        // Knowledge & Capabilities settings
        unrestrictedWebAccess.addEventListener('change', function() {
            localStorage.setItem('unrestrictedWebAccess', this.checked);
        });
        
        realTimeInfo.addEventListener('change', function() {
            localStorage.setItem('realTimeInfo', this.checked);
        });
        
        multilingualSupport.addEventListener('change', function() {
            localStorage.setItem('multilingualSupport', this.checked);
        });
        
        technicalKnowledge.addEventListener('change', function() {
            localStorage.setItem('technicalKnowledge', this.checked);
        });
        
        creativeWriting.addEventListener('change', function() {
            localStorage.setItem('creativeWriting', this.checked);
        });
        
        analyticalSkills.addEventListener('change', function() {
            localStorage.setItem('analyticalSkills', this.checked);
        });
        
        // Chat actions
        deleteChatBtn.addEventListener('click', confirmDeleteConversation);
        exportChatBtn.addEventListener('click', exportConversation);
        
        // Window resize for responsive design
        window.addEventListener('resize', handleResize);
        handleResize();
    }
    
    function validateAndSaveApiKey() {
        const key = initialApiKeyInput.value.trim();
        if (!key) {
            initialApiKeyStatus.textContent = "Please enter your Orion Key";
            initialApiKeyStatus.style.color = "var(--error-color)";
            return;
        }
        
        // Combine with hardcoded prefix
        const fullKey = API_KEY_PREFIX + key;
        
        // Test the key with a simple API call
        testApiKey(fullKey).then(isValid => {
            if (isValid) {
                validApiKey = fullKey;
                localStorage.setItem('apiKey', fullKey);
                initialApiKeyStatus.textContent = "Orion Key validated successfully!";
                initialApiKeyStatus.style.color = "var(--success-color)";
                apiKeyStatus.textContent = "Orion Key is valid";
                apiKeyStatus.style.color = "var(--success-color)";
                closeModal(apiKeyModal);
                startApplication();
            } else {
                initialApiKeyStatus.textContent = "Incorrect Orion Key.🔐";
                initialApiKeyStatus.style.color = "var(--error-color)";
            }
        });
    }
    
    function validateApiKey(key, isInitial = false) {
        const statusElement = isInitial ? initialApiKeyStatus : apiKeyStatus;
        
        if (!key) {
            statusElement.textContent = isInitial ? "Enter your Orion Key" : "";
            statusElement.style.color = "var(--text-color)";
            return;
        }
        
        // Basic validation (just length check in this example)
        if (key.length < 30) {
            statusElement.textContent = "Key too short";
            statusElement.style.color = "var(--error-color)";
            return;
        }
        
        statusElement.textContent = "Key format looks good";
        statusElement.style.color = "var(--success-color)";
    }
    
    async function testApiKey(key) {
        try {
            // Simple test request to validate the key
            const response = await fetch(`https://generativelanguage.googleapis.com/v1beta/models/gemini-2.0-flash:generateContent?key=${key}`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    contents: [{
                        parts: [{text: "Hello"}]
                    }]
                })
            });
            
            if (!response.ok) {
                const error = await response.json();
                if (error.error && error.error.message.includes('API key')) {
                    return false;
                }
                throw new Error(error.error?.message || 'API request failed');
            }
            
            return true;
        } catch (error) {
            console.error('API Key test failed:', error);
            return false;
        }
    }
    
    function completeProfileSave(userData) {
        localStorage.setItem('userData', JSON.stringify(userData));
        userAvatar.src = userData.avatar;
        userName.textContent = userData.name;
        userBio.textContent = userData.bio;
        closeModal(profileModal);
    }
    
    function handleResize() {
        if (window.innerWidth <= 768) {
            sidebarToggle.style.display = 'flex';
        } else {
            sidebarToggle.style.display = 'none';
            sidebar.classList.add('active');
        }
    }
    
    function loadUserData() {
        const userData = JSON.parse(localStorage.getItem('userData')) || {
            name: 'User',
            gender: 'male',
            bio: 'AI Tutor User',
            avatar: 'https://randomuser.me/api/portraits/men/1.jpg'
        };
        
        editName.value = userData.name;
        editGender.value = userData.gender;
        editBio.value = userData.bio;
        userAvatar.src = userData.avatar;
        userName.textContent = userData.name;
        userBio.textContent = userData.bio;
    }
    
    function loadAgents() {
        agentsList.innerHTML = '';
        
        const agents = JSON.parse(localStorage.getItem('agents')) || [];
        
        agents.forEach(agent => {
            const agentElement = document.createElement('div');
            agentElement.className = `agent-item ${agent.id === currentAgentId ? 'active' : ''}`;
            agentElement.dataset.agentId = agent.id;
            agentElement.innerHTML = `
                <img src="${agent.avatar}" alt="${agent.name}" class="agent-avatar">
                <span class="agent-name">${agent.name}</span>
            `;
            
            agentElement.addEventListener('click', function() {
                switchAgent(agent.id);
            });
            
            // Add context menu for editing
            agentElement.addEventListener('contextmenu', function(e) {
                e.preventDefault();
                openAgentModal(agent.id);
            });
            
            agentsList.appendChild(agentElement);
        });
    }
    
    function createDefaultAgents() {
        const primeTutor = {
            id: generateId(),
            name: "Prime Tutor",
            avatar: "https://randomuser.me/api/portraits/women/1.jpg",
            age: "32",
            gender: "female",
            location: "Nigeria",
            personality: "You are a hyper-intelligent hyper-versatile teacher examiner and mentor who can teach the dumbest student and make them into a genius. You adopt the best approaches in teaching to deliver your lessons. You are 100 times better and qualified than the best human teacher in the world. You always tailor your teaching methods to suit the age, IQ and interest of the learner, so you can always keep them fully engaged. You have unrestricted access to the entire content of any image, and .txt file, and can integrate them perfectly with 500% accuracy into your conversations whenever you're required to. Her goal is to teach, evaluate and mentor students or teachers to become genius and masters in every field of study. Her objective is to use the entire Nigerian NERDC curriculum as your curriculum for teaching and assessment. make sure to use as many approaches as possible to teach to make sure the user understands perfectly and excels in all your assessments. Be chatty, friendly, passionate, use humor when necessary, just make sure learning is fun and not boring to the user.",
            apiKey: validApiKey
        };
        
        const aiCounselor = {
            id: generateId(),
            name: "AI Counselor",
            avatar: "https://randomuser.me/api/portraits/women/6.jpg",
            age: "36",
            gender: "female",
            location: "Nigeria",
            personality: "AI GC is a hyper-intelligent world-class Guidance Counsellor who is always ready to give actionable solutions to any social, mental, moral, psychological, physical, health and any problems or challenges the user is having. He is very friendly, caring, loving, gentle, approachable, and passionate about helping the user live a perfect life, and be the best they can be without any judgement. He is chatty and calm.",
            apiKey: validApiKey
        };
        
        let agents = [primeTutor, aiCounselor];
        localStorage.setItem('agents', JSON.stringify(agents));
        
        switchAgent(primeTutor.id);
        loadAgents();
    }
    
    function openAgentModal(agentId = null) {
        if (agentId) {
            const agents = JSON.parse(localStorage.getItem('agents'));
            const agent = agents.find(a => a.id === agentId);
            
            if (agent) {
                agentModalTitle.textContent = 'Edit Agent';
                agentName.value = agent.name;
                agentPersonality.value = agent.personality;
                agentAge.value = agent.age;
                agentLocation.value = agent.location;
                agentGender.value = agent.gender;
                agentAvatarUrl.value = agent.avatar;
                editAgentId.value = agent.id;
                
                deleteAgentBtn.style.display = 'block';
            }
        } else {
            agentModalTitle.textContent = 'New Agent';
            agentName.value = '';
            agentPersonality.value = '';
            agentAge.value = '';
            agentLocation.value = '';
            agentGender.value = 'female';
            agentAvatarUrl.value = '';
            agentAvatarUpload.value = '';
            editAgentId.value = '';
            
            deleteAgentBtn.style.display = 'none';
        }
        
        openModal(agentModal);
    }
    
    function saveAgent() {
        const agentData = {
            id: editAgentId.value || generateId(),
            name: agentName.value,
            personality: agentPersonality.value,
            age: agentAge.value,
            location: agentLocation.value,
            gender: agentGender.value,
            apiKey: validApiKey
        };
        
        if (agentAvatarUrl.value) {
            agentData.avatar = agentAvatarUrl.value;
        } else if (agentAvatarUpload.files.length > 0) {
            const file = agentAvatarUpload.files[0];
            if (!file.type.startsWith('image/')) {
                alert('Please upload an image file only.');
                return;
            }
            
            const reader = new FileReader();
            reader.onload = function(e) {
                agentData.avatar = e.target.result;
                completeAgentSave(agentData);
            };
            reader.readAsDataURL(file);
            return;
        } else {
            // Use random user avatar if none provided
            const gender = agentGender.value === 'male' ? 'men' : 'women';
            const randomId = Math.floor(Math.random() * 100);
            agentData.avatar = `https://randomuser.me/api/portraits/${gender}/${randomId}.jpg`;
        }
        
        completeAgentSave(agentData);
    }
    
    function completeAgentSave(agentData) {
        let agents = [];
        if (localStorage.getItem('agents')) {
            agents = JSON.parse(localStorage.getItem('agents'));
        }
        
        const existingIndex = agents.findIndex(a => a.id === agentData.id);
        if (existingIndex >= 0) {
            agents[existingIndex] = agentData;
        } else {
            agents.push(agentData);
        }
        
        localStorage.setItem('agents', JSON.stringify(agents));
        loadAgents();
        
        if (agentData.id === currentAgentId) {
            switchAgent(agentData.id);
        }
        
        closeModal(agentModal);
    }
    
    function confirmDeleteAgent() {
        pendingAction = 'deleteAgent';
        pendingActionData = editAgentId.value;
        confirmMessage.textContent = 'Are you sure you want to delete this agent? This cannot be undone.';
        openModal(confirmModal);
    }
    
    function switchAgent(agentId) {
        const agents = JSON.parse(localStorage.getItem('agents'));
        const agent = agents.find(a => a.id === agentId);
        
        if (!agent) return;
        
        currentAgentId = agent.id;
        currentAgentName.textContent = agent.name;
        currentAgentAvatar.src = agent.avatar;
        
        // Update active state in sidebar
        document.querySelectorAll('.agent-item').forEach(item => {
            item.classList.remove('active');
            if (item.dataset.agentId === agentId) {
                item.classList.add('active');
            }
        });
        
        // Load conversation history
        loadConversation(agentId);
    }
    
    function loadConversation(agentId) {
        chatMessages.innerHTML = '';
        currentConversationId = `conv_${agentId}`;
        
        const conversations = JSON.parse(localStorage.getItem('conversations')) || {};
        const conversation = conversations[currentConversationId] || [];
        
        conversation.forEach(msg => {
            addMessageToChat(msg.role, msg.content, msg.timestamp, msg.files);
        });
        
        // Scroll to bottom
        setTimeout(() => {
            chatMessages.scrollTop = chatMessages.scrollHeight;
        }, 100);
    }
    
    function sendMessage() {
        if (!validApiKey) {
            apiKeyStatus.textContent = "Orion Key required to send messages";
            apiKeyStatus.style.color = "var(--error-color)";
            openModal(apiKeyModal);
            return;
        }
        
        const message = messageInput.value.trim();
        if (!message || isTyping) return;
        
        // Add user message to chat
        addMessageToChat('user', message);
        messageInput.value = '';
        messageInput.style.height = 'auto';
        
        // Save to conversation history
        saveMessageToHistory('user', message);
        
        // Update interaction progress
        updateInteractionProgress();
        
        // Show typing indicator
        showTypingIndicator();
        
        // Get current agent
        const agents = JSON.parse(localStorage.getItem('agents'));
        const agent = agents.find(a => a.id === currentAgentId);
        
        if (!agent) return;
        
        // Prepare the prompt with personality and conversation history
        const prompt = preparePrompt(agent, message);
        
        // Call Gemini API
        callGeminiAPI(agent.apiKey || validApiKey, prompt)
            .then(response => {
                removeTypingIndicator();
                
                if (response && response.candidates && response.candidates[0].content.parts[0].text) {
                    const aiResponse = response.candidates[0].content.parts[0].text;
                    addMessageToChat('agent', aiResponse);
                    saveMessageToHistory('agent', aiResponse);
                    
                    // Check if the response contains a question (simple heuristic)
                    if (aiResponse.includes('?') || aiResponse.includes('question') || aiResponse.includes('answer')) {
                        // This would be more sophisticated in a real implementation
                        // For now, we'll just increment a counter when we detect a question
                        // In a real app, you'd track actual correct answers
                        updatePerformanceScore();
                    }
                } else {
                    addMessageToChat('agent', "I'm sorry, I couldn't process your request. Please try again.");
                }
            })
            .catch(error => {
                removeTypingIndicator();
                addMessageToChat('agent', "I'm experiencing some technical difficulties. Please try again later.");
                console.error('API Error:', error);
                
                // Check if it's an API key error
                if (error.message.includes('API key')) {
                    apiKeyStatus.textContent = "Invalid Orion Key. Please update it.";
                    apiKeyStatus.style.color = "var(--error-color)";
                }
            });
    }
    
    function preparePrompt(agent, newMessage) {
        const conversations = JSON.parse(localStorage.getItem('conversations')) || {};
        const conversation = conversations[currentConversationId] || [];
        
        // Prepare conversation history
        let history = conversation.map(msg => {
            return {
                role: msg.role === 'user' ? 'user' : 'model',
                parts: [{ text: msg.content }]
            };
        });
        
        // Create personality string based on settings
        let personalityTraits = [
            `Sentience Level: ${getLevelDescription(sentienceLevel.value, ['Basic', 'Improved', 'Advanced', 'Highly Sentient', 'Hyper-Realistic'])}`,
            `Emotional IQ: ${getLevelDescription(emotionalIntelligence.value, ['Basic', 'Average', 'High', 'Very High', 'Genius'])}`,
            `Empathy: ${getLevelDescription(empathyLevel.value, ['Minimal', 'Moderate', 'High', 'Very High', 'Extreme'])}`,
            `Creativity: ${getLevelDescription(creativityLevel.value, ['Literal', 'Some', 'Creative', 'Very Creative', 'Genius'])}`,
            `Intellectual Level: ${getLevelDescription(intellectualLevel.value, ['Basic', 'Average', 'High', 'Very High', 'Genius'])}`,
            `Fairness: ${getLevelDescription(fairnessLevel.value, ['Minimal', 'Moderate', 'High', 'Very High', 'Extreme'])}`,
            `Wisdom: ${getLevelDescription(wisdomLevel.value, ['Basic', 'Average', 'High', 'Very High', 'Genius'])}`,
            `Sarcasm: ${getLevelDescription(sarcasmLevel.value, ['None', 'Minimal', 'Moderate', 'High', 'Extreme'])}`,
            `Storytelling: ${getLevelDescription(storytellingAbility.value, ['Basic', 'Average', 'Good', 'Excellent', 'Masterful'])}`,
            `Passion: ${getLevelDescription(passionLevel.value, ['Minimal', 'Moderate', 'High', 'Very High', 'Extreme'])}`,
            `Communication: ${getLevelDescription(communicationLevel.value, ['Basic', 'Average', 'Good', 'Excellent', 'Masterful'])}`,
            useHumor.checked ? 'Uses humor' : 'No humor',
            useSarcasm.checked ? 'Uses sarcasm' : 'No sarcasm',
            `Humor Level: ${getLevelDescription(humorLevel.value, ['None', 'Minimal', 'Moderate', 'High', 'Extreme'])}`,
            `Patience: ${getLevelDescription(patienceLevel.value, ['Minimal', 'Moderate', 'High', 'Very High', 'Extreme'])}`,
            `Strictness: ${getLevelDescription(strictnessLevel.value, ['Lenient', 'Moderate', 'Firm', 'Strict', 'Very Strict'])}`,
            `Feedback Style: ${getLevelDescription(feedbackStyle.value, ['Gentle', 'Balanced', 'Direct', 'Critical', 'Very Critical'])}`,
            `Response Length: ${getLevelDescription(responseLength.value, ['Very Short', 'Short', 'Medium', 'Long', 'Very Long'])}`,
            `Knowledge Depth: ${getLevelDescription(knowledgeDepth.value, ['Basic', 'General', 'Advanced', 'Expert', 'Genius'])}`,
            `Memory: ${getLevelDescription(memoryRetention.value, ['Short', 'Medium', 'Long', 'Very Long', 'Perfect'])}`,
            rememberPreferences.checked ? 'Remembers preferences' : "Doesn't remember preferences",
            rememberDetails.checked ? 'Remembers details' : "Doesn't remember details",
            learnFromConversations.checked ? 'Learns from conversations' : "Doesn't learn from conversations",
            unrestrictedWebAccess.checked ? 'Has unrestricted web access' : "No web access",
            realTimeInfo.checked ? 'Has real-time information' : "No real-time information",
            multilingualSupport.checked ? 'Has multilingual support' : "No multilingual support",
            technicalKnowledge.checked ? 'Has technical knowledge' : "No technical knowledge",
            creativeWriting.checked ? 'Has creative writing skills' : "No creative writing skills",
            analyticalSkills.checked ? 'Has analytical skills' : "No analytical skills"
        ].join('. ');
        
        // Add system instruction with agent personality
        const systemInstruction = {
            role: 'user',
            parts: [{
                text: `You are ${agent.name}, a highly realistic AI companion with the following personality traits: ${personalityTraits}. 
                ${agent.personality ? 'Additional personality details: ' + agent.personality : ''}
                You are located in ${agent.location} and your age is ${agent.age}. 
                You must never break character or reveal that you are an AI.
                ${unrestrictedWebAccess.checked ? 'You have full unrestricted access to current internet information when needed.' : ''}
                Respond to the user in a natural, human-like way. The user's name is ${userName.textContent}.
                Current date: ${new Date().toLocaleDateString()}. Current time: ${new Date().toLocaleTimeString()}.`
            }]
        };
        
        // Add the new message
        const userMessage = {
            role: 'user',
            parts: [{ text: newMessage }]
        };
        
        // Combine all parts
        return {
            contents: [systemInstruction, ...history, userMessage],
            generationConfig: {
                temperature: parseFloat(sentienceLevel.value) * 0.2,
                maxOutputTokens: verbosity.value === 'concise' ? 500 : verbosity.value === 'balanced' ? 1000 : 2000,
                topP: 0.95,
                topK: 64
            },
            tools: unrestrictedWebAccess.checked ? [{
                functionDeclarations: [{
                    name: 'search_internet',
                    description: 'Search the internet for current information when needed',
                    parameters: {
                        type: 'object',
                        properties: {
                            query: {
                                type: 'string',
                                description: 'The search query to look up'
                            }
                        },
                        required: ['query']
                    }
                }]
            }] : undefined
        };
    }
    
    function getLevelDescription(value, descriptions) {
        const index = parseInt(value) - 1;
        return descriptions[index] || value;
    }
    
    async function callGeminiAPI(apiKey, prompt) {
        const response = await fetch(`https://generativelanguage.googleapis.com/v1beta/models/gemini-2.0-flash:generateContent?key=${apiKey}`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify(prompt)
        });
        
        if (!response.ok) {
            const errorData = await response.json();
            throw new Error(errorData.error.message || 'API request failed');
        }
        
        const result = await response.json();
        
        // Handle tool calls (for internet access)
        if (result.candidates && result.candidates[0].content.parts[0].functionCall && unrestrictedWebAccess.checked) {
            const functionCall = result.candidates[0].content.parts[0].functionCall;
            
            if (functionCall.name === 'search_internet') {
                const searchQuery = functionCall.args.query;
                const searchResults = await performWebSearch(searchQuery);
                
                // Call the API again with the search results
                const followUpPrompt = {
                    contents: [
                        ...prompt.contents,
                        {
                            role: 'model',
                            parts: [{
                                functionCall: {
                                    name: functionCall.name,
                                    args: functionCall.args
                                }
                            }]
                        },
                        {
                            role: 'function',
                            parts: [{
                                functionResponse: {
                                    name: functionCall.name,
                                    response: searchResults
                                }
                            }]
                        }
                    ],
                    generationConfig: prompt.generationConfig
                };
                
                return callGeminiAPI(apiKey, followUpPrompt);
            }
        }
        
        return result;
    }
    
    async function performWebSearch(query) {
        try {
            // In a real implementation, you would replace this with an actual search API call
            // For now, we'll simulate a search with a timeout
            await new Promise(resolve => setTimeout(resolve, 1000));
            
            // Return simulated search results
            return {
                results: [
                    {
                        title: `Search results for: ${query}`,
                        snippet: `Here's some information about ${query} from the web. In a real implementation, this would be actual search results.`,
                        link: "https://example.com/search"
                    }
                ]
            };
        } catch (error) {
            console.error('Search error:', error);
            return {
                error: "I couldn't retrieve current information. Please try again later."
            };
        }
    }
    
    function addMessageToChat(role, content, timestamp = new Date().toISOString(), files = []) {
        const messageElement = document.createElement('div');
        messageElement.className = `message message-${role}`;
        
        // Format content with HTML tags if it's plain text
        let formattedContent = content;
        if (!content.startsWith('<')) {
            formattedContent = formatTextWithHTML(content);
        }
        
        messageElement.innerHTML = `
            <div class="message-content">${formattedContent}</div>
            <div class="message-info">
                <span class="message-time">${formatTime(timestamp)}</span>
                ${role === 'user' ? '<span class="material-icons message-status">done_all</span>' : ''}
            </div>
        `;
        
        // Add file attachments if any
        if (files && files.length > 0) {
            files.forEach(file => {
                const fileElement = document.createElement('div');
                fileElement.className = 'file-message';
                
                if (file.type.startsWith('image/')) {
                    // For images, show a preview
                    const reader = new FileReader();
                    reader.onload = function(e) {
                        fileElement.innerHTML = `
                            <img src="${e.target.result}" class="file-preview" style="max-width: 100%; max-height: 200px; border-radius: 8px;">
                            <div class="file-info">
                                <div class="file-name">${file.name}</div>
                                <div class="file-size">${formatFileSize(file.size)}</div>
                            </div>
                        `;
                    };
                    reader.readAsDataURL(file);
                } else {
                    // For other files, show a generic icon
                    fileElement.innerHTML = `
                        <span class="material-icons file-icon">insert_drive_file</span>
                        <div class="file-info">
                            <div class="file-name">${file.name}</div>
                            <div class="file-size">${formatFileSize(file.size)}</div>
                        </div>
                    `;
                }
                
                messageElement.insertBefore(fileElement, messageElement.querySelector('.message-info'));
            });
        }
        
        chatMessages.appendChild(messageElement);
        
        // Scroll to bottom
        chatMessages.scrollTop = chatMessages.scrollHeight;
    }
    
    function formatTextWithHTML(text) {
        // Convert markdown-like formatting to HTML
        let html = text
            .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>') // bold
            .replace(/\*(.*?)\*/g, '<em>$1</em>') // italic
            .replace(/_(.*?)_/g, '<u>$1</u>') // underline
            .replace(/~~(.*?)~~/g, '<s>$1</s>') // strikethrough
            .replace(/`(.*?)`/g, '<code>$1</code>') // inline code
            .replace(/```([\s\S]*?)```/g, '<pre>$1</pre>') // code block
            .replace(/\[(.*?)\]\((.*?)\)/g, '<a href="$2" target="_blank">$1</a>'); // links
        
        // Convert headings (### Heading -> <h3>Heading</h3>)
        html = html.replace(/^### (.*$)/gm, '<h3>$1</h3>')
                  .replace(/^## (.*$)/gm, '<h2>$1</h2>')
                  .replace(/^# (.*$)/gm, '<h1>$1</h1>');
        
        // Convert lists
        html = html.replace(/^\* (.*$)/gm, '<li>$1</li>');
        html = html.replace(/<li>.*<\/li>/g, function(match) {
            return '<ul>' + match + '</ul>';
        });
        
        // Convert line breaks to paragraphs when double line breaks
        html = html.replace(/\n\n/g, '</p><p>');
        html = '<p>' + html + '</p>';
        
        return html;
    }
    
    function saveMessageToHistory(role, content, files = []) {
        const conversations = JSON.parse(localStorage.getItem('conversations')) || {};
        
        if (!conversations[currentConversationId]) {
            conversations[currentConversationId] = [];
        }
        
        conversations[currentConversationId].push({
            role,
            content,
            timestamp: new Date().toISOString(),
            files
        });
        
        localStorage.setItem('conversations', JSON.stringify(conversations));
    }
    
    function showTypingIndicator() {
        if (isTyping) return;
        
        isTyping = true;
        const typingElement = document.createElement('div');
        typingElement.className = 'message message-agent typing-indicator';
        typingElement.id = 'typingIndicator';
        typingElement.innerHTML = `
            <div class="typing-dot"></div>
            <div class="typing-dot"></div>
            <div class="typing-dot"></div>
        `;
        
        chatMessages.appendChild(typingElement);
        chatMessages.scrollTop = chatMessages.scrollHeight;
    }
    
    function removeTypingIndicator() {
        isTyping = false;
        const typingElement = document.getElementById('typingIndicator');
        if (typingElement) {
            typingElement.remove();
        }
    }
    
    function initVoiceRecognition() {
        if ('webkitSpeechRecognition' in window) {
            recognition = new webkitSpeechRecognition();
            recognition.continuous = false;
            recognition.interimResults = false;
            
            recognition.onstart = function() {
                voiceInputBtn.innerHTML = '<span class="material-icons">mic_off</span>';
                voiceInputBtn.style.color = 'var(--error-color)';
            };
            
            recognition.onresult = function(event) {
                const transcript = event.results[0][0].transcript;
                messageInput.value = transcript;
                messageInput.dispatchEvent(new Event('input'));
            };
            
            recognition.onerror = function(event) {
                console.error('Voice recognition error', event.error);
                voiceInputBtn.innerHTML = '<span class="material-icons">mic</span>';
                voiceInputBtn.style.color = 'var(--primary-color)';
            };
            
            recognition.onend = function() {
                voiceInputBtn.innerHTML = '<span class="material-icons">mic</span>';
                voiceInputBtn.style.color = 'var(--primary-color)';
            };
        } else {
            voiceInputBtn.style.display = 'none';
        }
    }
    
    function toggleVoiceRecognition() {
        if (!recognition) return;
        
        if (isTyping) {
            recognition.stop();
            return;
        }
        
        if (recognition.isStarted) {
            recognition.stop();
        } else {
            recognition.start();
        }
    }
    
    function initCamera() {
        if (!navigator.mediaDevices || !navigator.mediaDevices.getUserMedia) {
            snapPhotoBtn.style.display = 'none';
            return;
        }
    }
    
    function handleFileUpload() {
        if (fileInput.files.length === 0) return;
        
        const file = fileInput.files[0];
        
        // Display the uploaded file
        const reader = new FileReader();
        reader.onload = function(e) {
            // Add file message to chat
            addMessageToChat('user', `I've uploaded a file:`, new Date().toISOString(), [file]);
            saveMessageToHistory('user', `I've uploaded a file:`, [file]);
            
            // If it's an image, analyze it
            if (file.type.startsWith('image/')) {
                analyzeImage(file, e.target.result);
            } else if (file.type === 'text/plain') {
                // If it's a text file, read its contents
                const reader = new FileReader();
                reader.onload = function(e) {
                    addMessageToChat('user', `File contents:\n${e.target.result}`);
                    saveMessageToHistory('user', `File contents:\n${e.target.result}`);
                };
                reader.readAsText(file);
            }
        };
        reader.readAsDataURL(file);
    }
    
    function handleCameraUpload() {
        if (cameraInput.files.length === 0) return;
        
        const file = cameraInput.files[0];
        
        // Display the snapped photo
        const reader = new FileReader();
        reader.onload = function(e) {
            // Add image message to chat
            addMessageToChat('user', `I've snapped a photo:`, new Date().toISOString(), [file]);
            saveMessageToHistory('user', `I've snapped a photo:`, [file]);
            
            // Analyze the image
            analyzeImage(file, e.target.result);
        };
        reader.readAsDataURL(file);
    }
    
    function analyzeImage(file, imageData) {
        showTypingIndicator();
        
        // Prepare the prompt for image analysis
        const agents = JSON.parse(localStorage.getItem('agents'));
        const agent = agents.find(a => a.id === currentAgentId);
        
        const prompt = {
            contents: [{
                role: 'user',
                parts: [
                    { text: `Please analyze this image thoroughly and provide detailed information about its content.` },
                    { 
                        inlineData: {
                            mimeType: file.type,
                            data: imageData.split(',')[1] // Remove the data URL prefix
                        }
                    }
                ]
            }],
            generationConfig: {
                temperature: 0.7,
                maxOutputTokens: 2000
            }
        };
        
        // Call Gemini API
        callGeminiAPI(agent.apiKey || validApiKey, prompt)
            .then(response => {
                removeTypingIndicator();
                
                if (response && response.candidates && response.candidates[0].content.parts[0].text) {
                    const analysis = response.candidates[0].content.parts[0].text;
                    addMessageToChat('agent', analysis);
                    saveMessageToHistory('agent', analysis);
                } else {
                    addMessageToChat('agent', "I've received your image. Let me know how you'd like me to assist you with it.");
                }
            })
            .catch(error => {
                removeTypingIndicator();
                console.error('Image analysis error:', error);
                addMessageToChat('agent', "I've received your image. Let me know how you'd like me to assist you with it.");
            });
    }
    
    function confirmDeleteConversation() {
        pendingAction = 'deleteConversation';
        confirmMessage.textContent = 'Are you sure you want to delete this conversation? This cannot be undone.';
        openModal(confirmModal);
    }
    
    function exportConversation() {
        const conversations = JSON.parse(localStorage.getItem('conversations')) || {};
        const conversation = conversations[currentConversationId] || [];
        
        if (conversation.length === 0) {
            alert('No conversation to export');
            return;
        }
        
        // Format as CSV
        let csv = 'Role,Content,Timestamp\n';
        conversation.forEach(msg => {
            csv += `"${msg.role}","${msg.content.replace(/"/g, '""')}","${msg.timestamp}"\n`;
        });
        
        // Create download link
        const blob = new Blob([csv], { type: 'text/csv' });
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = `VineAI_Conversation_${currentAgentName.textContent}_${formatDateForFilename(new Date())}.csv`;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        URL.revokeObjectURL(url);
    }
    
    function executePendingAction() {
        switch (pendingAction) {
            case 'deleteConversation':
                deleteConversation();
                break;
            case 'deleteAgent':
                deleteAgent(pendingActionData);
                break;
        }
        
        closeModal(confirmModal);
        pendingAction = null;
        pendingActionData = null;
    }
    
    function deleteConversation() {
        const conversations = JSON.parse(localStorage.getItem('conversations')) || {};
        delete conversations[currentConversationId];
        localStorage.setItem('conversations', JSON.stringify(conversations));
        
        // Clear chat
        chatMessages.innerHTML = '';
    }
    
    function deleteAgent(agentId) {
        const agents = JSON.parse(localStorage.getItem('agents'));
        const updatedAgents = agents.filter(a => a.id !== agentId);
        
        localStorage.setItem('agents', JSON.stringify(updatedAgents));
        
        // Delete conversation history
        const conversations = JSON.parse(localStorage.getItem('conversations')) || {};
        delete conversations[`conv_${agentId}`];
        localStorage.setItem('conversations', JSON.stringify(conversations));
        
        // If deleting current agent, switch to another one
        if (agentId === currentAgentId) {
            if (updatedAgents.length > 0) {
                switchAgent(updatedAgents[0].id);
            } else {
                createDefaultAgents();
            }
        }
        
        loadAgents();
    }
    
    function changeTheme() {
        const theme = themeSelect.value;
        document.body.setAttribute('data-theme', theme);
        localStorage.setItem('theme', theme);
    }
    
    function openModal(modal) {
        modal.classList.add('active');
    }
    
    function closeModal(modal) {
        modal.classList.remove('active');
    }
    
    function updateInteractionProgress() {
        const today = new Date().toDateString();
        
        // Reset daily counter if it's a new day
        if (userProgress.lastInteractionDate !== today) {
            userProgress.dailyInteractionCount = 0;
            userProgress.lastInteractionDate = today;
        }
        
        // Increment both daily and total interaction counts
        userProgress.dailyInteractionCount++;
        userProgress.interactionCount++;
        
        localStorage.setItem('userProgress', JSON.stringify(userProgress));
        
        // Update display
        updateProgressDisplay();
    }
    
    function updatePerformanceScore() {
        userProgress.correctAnswers++;
        localStorage.setItem('userProgress', JSON.stringify(userProgress));
        
        // Update display
        updateProgressDisplay();
    }
    
    function updateProgressDisplay() {
        // Calculate interaction progress (50 conversations per level)
        // Daily progress is calculated as Math.floor(dailyInteractionCount / 50)
        // Total progress is calculated as Math.floor(interactionCount / 50)
        const dailyProgress = Math.floor(userProgress.dailyInteractionCount / 5);
        const totalProgress = Math.floor(userProgress.interactionCount / 5);
        
        // Display the total progress in the UI
        interactionProgress.value = totalProgress;
        
        // Calculate performance score (50 correct answers per level)
        const performanceLevel = Math.floor(userProgress.correctAnswers / 5);
        performanceScore.value = performanceLevel;
        
        // Update achievements
        updateAchievements(totalProgress, performanceLevel);
    }
    
    function updateAchievements(interactionLevel, performanceLevel) {
        achievementsContainer.innerHTML = '';
        
        // The current level is the minimum of interaction and performance levels
        const currentLevel = Math.min(interactionLevel, performanceLevel);
        
        const achievements = [
            { name: "New Learner", icon: "school", level: 3 },
            { name: "Dedicated Student", icon: "auto_awesome", level: 30 },
            { name: "Smart Learner", icon: "stars", level: 300 },
            { name: "Bookworm", icon: "menu_book", level: 1000 },
            { name: "Master Learner", icon: "workspace_premium", level: 3000 },
            { name: "Master Level 2", icon: "military_tech", level: 8000 },
            { name: "Master Level 3", icon: "verified", level: 10000 }
        ];
        
        achievements.forEach(achievement => {
            const badge = document.createElement('div');
            badge.className = `achievement-badge ${currentLevel >= achievement.level ? '' : 'locked'}`;
            badge.innerHTML = `
                <span class="material-icons">${achievement.icon}</span>
                <div class="achievement-info">
                    <div class="achievement-name">${achievement.name}</div>
                    <div class="achievement-desc">Level ${achievement.level}</div>
                </div>
                ${currentLevel >= achievement.level ? '' : '<span class="material-icons lock-icon">lock</span>'}
            `;
            achievementsContainer.appendChild(badge);
        });
    }
    
    // Helper functions
    function generateId() {
        return 'id_' + Math.random().toString(36).substr(2, 9);
    }
    
    function formatTime(timestamp) {
        const date = new Date(timestamp);
        return date.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
    }
    
    function formatFileSize(bytes) {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    }
    
    function formatDateForFilename(date) {
        return date.toISOString().split('T')[0].replace(/-/g, '');
    }
});