<?php
require_once 'config.php';

// Redirect if not logged in or not admin
if (!is_logged_in() || !is_admin()) {
    redirect('login.php');
}

$settings = get_settings();
$users = json_decode(file_get_contents(USERS_JSON), true);
$licenses = get_all_licenses();
$transactions = get_all_transactions();
$plans = json_decode(file_get_contents(PLANS_JSON), true);

// Count statistics
$total_users = count($users);
$total_licenses = count($licenses);
$total_transactions = count($transactions);
$revenue = array_sum(array_column($transactions, 'amount'));

// Active licenses count
$active_licenses = array_filter($licenses, function($license) {
    return $license['status'] === 'active' && strtotime($license['expires_at']) > time();
});

// Expired licenses count
$expired_licenses = array_filter($licenses, function($license) {
    return $license['status'] === 'active' && strtotime($license['expires_at']) < time();
});

// Handle user actions
if (isset($_GET['action']) && isset($_GET['user_id'])) {
    $user_id = $_GET['user_id'];
    
    foreach ($users as $index => $user) {
        if ($user['id'] === $user_id) {
            if ($_GET['action'] === 'activate') {
                $users[$index]['status'] = 'active';
            } elseif ($_GET['action'] === 'suspend') {
                $users[$index]['status'] = 'suspended';
            } elseif ($_GET['action'] === 'delete') {
                // Cannot delete admin users
                if ($user['role'] !== ROLE_ADMIN) {
                    array_splice($users, $index, 1);
                }
            }
            break;
        }
    }
    
    file_put_contents(USERS_JSON, json_encode($users));
    redirect('admin.php');
}

// Handle license actions
if (isset($_GET['action']) && isset($_GET['license_key'])) {
    $license_key = $_GET['license_key'];
    
    foreach ($licenses as $index => $license) {
        if ($license['key'] === $license_key) {
            if ($_GET['action'] === 'activate') {
                $licenses[$index]['status'] = 'active';
            } elseif ($_GET['action'] === 'deactivate') {
                $licenses[$index]['status'] = 'inactive';
            } elseif ($_GET['action'] === 'extend') {
                $licenses[$index]['expires_at'] = date('Y-m-d H:i:s', strtotime('+1 year'));
            } elseif ($_GET['action'] === 'delete') {
                array_splice($licenses, $index, 1);
            }
            break;
        }
    }
    
    file_put_contents(LICENSES_JSON, json_encode($licenses));
    redirect('admin.php');
}

// Handle settings update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_settings'])) {
    $new_settings = [
        'site_name' => $_POST['site_name'],
        'currency' => $_POST['currency'],
        'company_name' => $_POST['company_name'],
        'company_address' => $_POST['company_address'],
        'support_email' => $_POST['support_email'],
        'low_license_threshold' => (int)$_POST['low_license_threshold'],
        'license_prefix' => $_POST['license_prefix'],
        'license_length' => (int)$_POST['license_length']
    ];
    
    file_put_contents(SETTINGS_JSON, json_encode($new_settings));
    redirect('admin.php?tab=settings');
}

// Handle plan update
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_plan'])) {
    $plan_id = $_POST['plan_id'];
    $plans[$plan_id] = [
        'id' => $plan_id,
        'name' => $_POST['name'],
        'price' => (float)$_POST['price'],
        'duration' => (int)$_POST['duration'],
        'features' => explode("\n", str_replace("\r", "", $_POST['features'])),
        'is_active' => isset($_POST['is_active'])
    ];
    
    file_put_contents(PLANS_JSON, json_encode($plans));
    redirect('admin.php?tab=plans');
}

// Get current tab
$tab = isset($_GET['tab']) ? $_GET['tab'] : 'dashboard';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Panel - <?php echo $settings['site_name']; ?></title>
    <link rel="stylesheet" href="styles.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body>
    <header class="header">
        <div class="container">
            <div class="logo">
                <h1><i class="fas fa-cog"></i> Admin Panel</h1>
            </div>
            <nav class="nav">
                <a href="index.php">Home</a>
                <a href="dashboard.php">Dashboard</a>
                <a href="logout.php">Logout</a>
            </nav>
            <div class="mobile-menu-btn">
                <i class="fas fa-bars"></i>
            </div>
        </div>
    </header>

    <main class="dashboard">
        <div class="container">
            <div class="admin-tabs">
                <a href="?tab=dashboard" class="<?php echo $tab === 'dashboard' ? 'active' : ''; ?>">Dashboard</a>
                <a href="?tab=users" class="<?php echo $tab === 'users' ? 'active' : ''; ?>">Users</a>
                <a href="?tab=licenses" class="<?php echo $tab === 'licenses' ? 'active' : ''; ?>">Licenses</a>
                <a href="?tab=transactions" class="<?php echo $tab === 'transactions' ? 'active' : ''; ?>">Transactions</a>
                <a href="?tab=plans" class="<?php echo $tab === 'plans' ? 'active' : ''; ?>">Plans</a>
                <a href="?tab=settings" class="<?php echo $tab === 'settings' ? 'active' : ''; ?>">Settings</a>
            </div>

            <?php if ($tab === 'dashboard'): ?>
            <div class="dashboard-header">
                <h2>Admin Dashboard</h2>
            </div>

            <div class="dashboard-cards">
                <div class="dashboard-card">
                    <h3>Total Users</h3>
                    <div class="number"><?php echo $total_users; ?></div>
                </div>
                <div class="dashboard-card">
                    <h3>Total Licenses</h3>
                    <div class="number"><?php echo $total_licenses; ?></div>
                </div>
                <div class="dashboard-card">
                    <h3>Active Licenses</h3>
                    <div class="number"><?php echo count($active_licenses); ?></div>
                </div>
                <div class="dashboard-card">
                    <h3>Expired Licenses</h3>
                    <div class="number"><?php echo count($expired_licenses); ?></div>
                </div>
                <div class="dashboard-card">
                    <h3>Total Transactions</h3>
                    <div class="number"><?php echo $total_transactions; ?></div>
                </div>
                <div class="dashboard-card">
                    <h3>Total Revenue</h3>
                    <div class="number">₦<?php echo number_format($revenue, 2); ?></div>
                </div>
            </div>

            <div class="dashboard-section">
                <h3>Recent Transactions</h3>
                <?php if (empty($transactions)): ?>
                <p>No transactions found.</p>
                <?php else: ?>
                <div class="table-responsive">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Transaction ID</th>
                                <th>User</th>
                                <th>Plan</th>
                                <th>Amount</th>
                                <th>Date</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach (array_slice($transactions, 0, 10) as $transaction): 
                                $user = get_user_by_id($transaction['user_id']);
                                $plan = get_plan_by_id($transaction['plan_id']);
                            ?>
                            <tr>
                                <td><?php echo $transaction['tx_ref']; ?></td>
                                <td><?php echo $user['username']; ?></td>
                                <td><?php echo $plan['name']; ?></td>
                                <td>₦<?php echo number_format($transaction['amount']); ?></td>
                                <td><?php echo date('M j, Y H:i', strtotime($transaction['created_at'])); ?></td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php endif; ?>
            </div>
            <?php endif; ?>

            <?php if ($tab === 'users'): ?>
            <div class="dashboard-section">
                <h3>User Management</h3>
                <?php if (empty($users)): ?>
                <p>No users found.</p>
                <?php else: ?>
                <div class="table-responsive">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Username</th>
                                <th>Email</th>
                                <th>Role</th>
                                <th>Status</th>
                                <th>Registered</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($users as $user): ?>
                            <tr>
                                <td><?php echo $user['username']; ?></td>
                                <td><?php echo $user['email']; ?></td>
                                <td><?php echo ucfirst($user['role']); ?></td>
                                <td>
                                    <span class="<?php echo $user['status'] === 'active' ? 'status-active' : 'status-inactive'; ?>">
                                        <?php echo ucfirst($user['status']); ?>
                                    </span>
                                </td>
                                <td><?php echo date('M j, Y', strtotime($user['created_at'])); ?></td>
                                <td>
                                    <?php if ($user['role'] !== ROLE_ADMIN): ?>
                                        <?php if ($user['status'] === 'active'): ?>
                                        <a href="?tab=users&action=suspend&user_id=<?php echo $user['id']; ?>" class="btn btn-warning">Suspend</a>
                                        <?php else: ?>
                                        <a href="?tab=users&action=activate&user_id=<?php echo $user['id']; ?>" class="btn btn-success">Activate</a>
                                        <?php endif; ?>
                                        <a href="?tab=users&action=delete&user_id=<?php echo $user['id']; ?>" class="btn btn-danger" onclick="return confirm('Are you sure you want to delete this user?')">Delete</a>
                                    <?php else: ?>
                                        <span class="status-active">Protected</span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php endif; ?>
            </div>
            <?php endif; ?>

            <?php if ($tab === 'licenses'): ?>
            <div class="dashboard-section">
                <h3>License Management</h3>
                <?php if (empty($licenses)): ?>
                <p>No licenses found.</p>
                <?php else: ?>
                <div class="table-responsive">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>License Key</th>
                                <th>User</th>
                                <th>Plan</th>
                                <th>Status</th>
                                <th>Expires</th>
                                <th>Domains</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($licenses as $license): 
                                $user = get_user_by_id($license['user_id']);
                                $plan = get_plan_by_id($license['plan_id']);
                                $is_expired = strtotime($license['expires_at']) < time();
                            ?>
                            <tr>
                                <td><code><?php echo $license['key']; ?></code></td>
                                <td><?php echo $user['username']; ?></td>
                                <td><?php echo $plan['name']; ?></td>
                                <td>
                                    <span class="<?php echo ($license['status'] === 'active' && !$is_expired) ? 'status-active' : 'status-inactive'; ?>">
                                        <?php echo ($license['status'] === 'active' && !$is_expired) ? 'Active' : 'Inactive'; ?>
                                    </span>
                                </td>
                                <td><?php echo date('M j, Y', strtotime($license['expires_at'])); ?></td>
                                <td>
                                    <?php if (isset($license['domains']) && !empty($license['domains'])): ?>
                                        <?php echo count($license['domains']); ?> domain(s)
                                    <?php else: ?>
                                        0 domains
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($license['status'] === 'active'): ?>
                                    <a href="?tab=licenses&action=deactivate&license_key=<?php echo $license['key']; ?>" class="btn btn-warning">Deactivate</a>
                                    <?php else: ?>
                                    <a href="?tab=licenses&action=activate&license_key=<?php echo $license['key']; ?>" class="btn btn-success">Activate</a>
                                    <?php endif; ?>
                                    <a href="?tab=licenses&action=extend&license_key=<?php echo $license['key']; ?>" class="btn btn-secondary">Extend</a>
                                    <a href="?tab=licenses&action=delete&license_key=<?php echo $license['key']; ?>" class="btn btn-danger" onclick="return confirm('Are you sure you want to delete this license?')">Delete</a>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php endif; ?>
            </div>
            <?php endif; ?>

            <?php if ($tab === 'transactions'): ?>
            <div class="dashboard-section">
                <h3>Transaction History</h3>
                <?php if (empty($transactions)): ?>
                <p>No transactions found.</p>
                <?php else: ?>
                <div class="table-responsive">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Transaction ID</th>
                                <th>User</th>
                                <th>Plan</th>
                                <th>Amount</th>
                                <th>Date</th>
                                <th>Status</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($transactions as $transaction): 
                                $user = get_user_by_id($transaction['user_id']);
                                $plan = get_plan_by_id($transaction['plan_id']);
                            ?>
                            <tr>
                                <td><?php echo $transaction['tx_ref']; ?></td>
                                <td><?php echo $user['username']; ?></td>
                                <td><?php echo $plan['name']; ?></td>
                                <td>₦<?php echo number_format($transaction['amount']); ?></td>
                                <td><?php echo date('M j, Y H:i', strtotime($transaction['created_at'])); ?></td>
                                <td>
                                    <span class="status-active">Completed</span>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
                <?php endif; ?>
            </div>
            <?php endif; ?>

            <?php if ($tab === 'plans'): ?>
            <div class="dashboard-section">
                <h3>Subscription Plans</h3>
                
                <?php foreach ($plans as $plan_id => $plan): ?>
                <div class="form-container" style="margin-bottom: 30px;">
                    <h4>Edit Plan: <?php echo $plan['name']; ?></h4>
                    <form method="POST" action="">
                        <input type="hidden" name="plan_id" value="<?php echo $plan_id; ?>">
                        <input type="hidden" name="update_plan" value="1">
                        
                        <div class="form-group">
                            <label for="name_<?php echo $plan_id; ?>">Plan Name</label>
                            <input type="text" id="name_<?php echo $plan_id; ?>" name="name" value="<?php echo $plan['name']; ?>" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="price_<?php echo $plan_id; ?>">Price (₦)</label>
                            <input type="number" id="price_<?php echo $plan_id; ?>" name="price" value="<?php echo $plan['price']; ?>" required min="0" step="0.01">
                        </div>
                        
                        <div class="form-group">
                            <label for="duration_<?php echo $plan_id; ?>">Duration (days)</label>
                            <input type="number" id="duration_<?php echo $plan_id; ?>" name="duration" value="<?php echo $plan['duration']; ?>" required min="1">
                        </div>
                        
                        <div class="form-group">
                            <label for="features_<?php echo $plan_id; ?>">Features (one per line)</label>
                            <textarea id="features_<?php echo $plan_id; ?>" name="features" rows="5" required><?php echo implode("\n", $plan['features']); ?></textarea>
                        </div>
                        
                        <div class="form-group">
                            <label>
                                <input type="checkbox" name="is_active" <?php echo $plan['is_active'] ? 'checked' : ''; ?>>
                                Active (available for purchase)
                            </label>
                        </div>
                        
                        <button type="submit" class="btn btn-primary">Update Plan</button>
                    </form>
                </div>
                <?php endforeach; ?>
            </div>
            <?php endif; ?>

            <?php if ($tab === 'settings'): ?>
            <div class="dashboard-section">
                <h3>System Settings</h3>
                <div class="form-container">
                    <form method="POST" action="">
                        <input type="hidden" name="update_settings" value="1">
                        
                        <div class="form-group">
                            <label for="site_name">Site Name</label>
                            <input type="text" id="site_name" name="site_name" value="<?php echo $settings['site_name']; ?>" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="currency">Currency</label>
                            <input type="text" id="currency" name="currency" value="<?php echo $settings['currency']; ?>" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="company_name">Company Name</label>
                            <input type="text" id="company_name" name="company_name" value="<?php echo $settings['company_name']; ?>" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="company_address">Company Address</label>
                            <input type="text" id="company_address" name="company_address" value="<?php echo $settings['company_address']; ?>" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="support_email">Support Email</label>
                            <input type="email" id="support_email" name="support_email" value="<?php echo $settings['support_email']; ?>" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="low_license_threshold">Low License Threshold</label>
                            <input type="number" id="low_license_threshold" name="low_license_threshold" value="<?php echo $settings['low_license_threshold']; ?>" required min="1">
                            <p class="description">Show warning when available licenses fall below this number</p>
                        </div>
                        
                        <div class="form-group">
                            <label for="license_prefix">License Prefix</label>
                            <input type="text" id="license_prefix" name="license_prefix" value="<?php echo $settings['license_prefix']; ?>" required>
                        </div>
                        
                        <div class="form-group">
                            <label for="license_length">License Key Length</label>
                            <input type="number" id="license_length" name="license_length" value="<?php echo $settings['license_length']; ?>" required min="8" max="32">
                            <p class="description">Length of the random part of the license key (excluding prefix)</p>
                        </div>
                        
                        <button type="submit" class="btn btn-primary">Save Settings</button>
                    </form>
                </div>
            </div>
            <?php endif; ?>
        </div>
    </main>

    <footer class="footer">
        <div class="container">
            <div class="footer-content">
                <div class="footer-section">
                    <h3><?php echo $settings['site_name']; ?></h3>
                    <p>Advanced AI writing tool for WordPress that generates human-like content without restrictions.</p>
                </div>
                <div class="footer-section">
                    <h4>Quick Links</h4>
                    <ul>
                        <li><a href="index.php">Home</a></li>
                        <li><a href="pricing.php">Pricing</a></li>
                        <li><a href="dashboard.php">Dashboard</a></li>
                    </ul>
                </div>
                <div class="footer-section">
                    <h4>Support</h4>
                    <ul>
                        <li><a href="#">Documentation</a></li>
                        <li><a href="#">Contact Us</a></li>
                        <li><a href="#">Privacy Policy</a></li>
                        <li><a href="#">Terms of Service</a></li>
                    </ul>
                </div>
                <div class="footer-section">
                    <h4>Contact Info</h4>
                    <p><i class="fas fa-envelope"></i> <?php echo $settings['support_email']; ?></p>
                    <p><i class="fas fa-map-marker-alt"></i> <?php echo $settings['company_address']; ?></p>
                </div>
            </div>
            <div class="footer-bottom">
                <p>&copy; <?php echo date('Y'); ?> <?php echo $settings['company_name']; ?>. All rights reserved.</p>
            </div>
        </div>
    </footer>

    <script src="script.js"></script>
</body>
</html>