<?php
// config.php - Application Configuration
session_start();

// Define base configuration
define('BASE_URL', 'https://p.vinesubtech.com.ng/writing-ai/');
define('LICENSE_DURATION', 365); // License validity in days
define('FLUTTERWAVE_TEST_MODE', true);

// User roles
define('ROLE_ADMIN', 'admin');
define('ROLE_USER', 'user');

// Database files
define('USERS_JSON', 'data/users.json');
define('PLANS_JSON', 'data/plans.json');
define('LICENSES_JSON', 'data/licenses.json');
define('TRANSACTIONS_JSON', 'data/transactions.json');
define('SETTINGS_JSON', 'data/settings.json');

// Flutterwave API configuration
define('FLW_PUBLIC_KEY', 'FLWPUBK_TEST-xxxxxxxxxxxx-X');
define('FLW_SECRET_KEY', 'FLWSECK_TEST-xxxxxxxxxxxx-X');
define('FLW_ENCRYPTION_KEY', 'FLWSECK_TESTxxxxxxxxxxxx');
define('FLW_BASE_URL', 'https://api.flutterwave.com/v3/');

// Initialize data files
init_data_files();

// Initialize admin user
init_admin_user();

// Helper functions
function init_data_files() {
    $data_files = [USERS_JSON, PLANS_JSON, LICENSES_JSON, TRANSACTIONS_JSON, SETTINGS_JSON];
    $data_dir = dirname(__FILE__) . '/data';
    
    if (!file_exists($data_dir)) {
        mkdir($data_dir, 0755, true);
    }
    
    foreach ($data_files as $file) {
        if (!file_exists($file)) {
            file_put_contents($file, json_encode([]));
        }
    }
    
    // Initialize default settings if empty
    $settings = json_decode(file_get_contents(SETTINGS_JSON), true);
    if (empty($settings)) {
        $default_settings = [
            'site_name' => 'Vine AI License Manager',
            'currency' => 'NGN',
            'company_name' => 'VineSub Technologies',
            'company_address' => 'Lagos, Nigeria',
            'support_email' => 'support@vinesubtech.com.ng',
            'low_license_threshold' => 10,
            'license_prefix' => 'VINEAI',
            'license_length' => 16
        ];
        file_put_contents(SETTINGS_JSON, json_encode($default_settings));
    }
    
    // Initialize default plans if empty
    $plans = json_decode(file_get_contents(PLANS_JSON), true);
    if (empty($plans)) {
        $default_plans = [
            'basic' => [
                'id' => 'basic',
                'name' => 'Basic Plan',
                'price' => 5000,
                'duration' => 365,
                'features' => [
                    '1 Website License',
                    'Basic Support',
                    'Regular Updates',
                    '1 Year Validity'
                ],
                'is_active' => true
            ],
            'professional' => [
                'id' => 'professional',
                'name' => 'Professional Plan',
                'price' => 10000,
                'duration' => 365,
                'features' => [
                    '3 Website Licenses',
                    'Priority Support',
                    'All Updates',
                    '1 Year Validity'
                ],
                'is_active' => true
            ],
            'enterprise' => [
                'id' => 'enterprise',
                'name' => 'Enterprise Plan',
                'price' => 20000,
                'duration' => 365,
                'features' => [
                    'Unlimited Licenses',
                    '24/7 Support',
                    'All Updates + Beta Features',
                    '1 Year Validity'
                ],
                'is_active' => true
            ]
        ];
        file_put_contents(PLANS_JSON, json_encode($default_plans));
    }
}

function init_admin_user() {
    $users = json_decode(file_get_contents(USERS_JSON), true);
    
    // Check if admin user already exists
    $admin_exists = false;
    foreach ($users as $user) {
        if ($user['username'] === 'vineadmin') {
            $admin_exists = true;
            break;
        }
    }
    
    if (!$admin_exists) {
        $admin_user = [
            'id' => uniqid(),
            'username' => 'vineadmin',
            'password' => password_hash('VineAdmin123!', PASSWORD_DEFAULT),
            'email' => 'prudenceeni@gmail.com',
            'role' => ROLE_ADMIN,
            'created_at' => date('Y-m-d H:i:s'),
            'status' => 'active'
        ];
        
        $users[] = $admin_user;
        file_put_contents(USERS_JSON, json_encode($users));
    }
}

function is_logged_in() {
    return isset($_SESSION['user_id']);
}

function is_admin() {
    return isset($_SESSION['user_role']) && $_SESSION['user_role'] === ROLE_ADMIN;
}

function redirect($url) {
    header("Location: $url");
    exit;
}

function get_settings() {
    return json_decode(file_get_contents(SETTINGS_JSON), true);
}

function generate_license_key() {
    $settings = get_settings();
    $prefix = $settings['license_prefix'] ?? 'VINEAI';
    $length = $settings['license_length'] ?? 16;
    
    $characters = '0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $characters_length = strlen($characters);
    $random_string = '';
    
    for ($i = 0; $i < $length; $i++) {
        $random_string .= $characters[rand(0, $characters_length - 1)];
    }
    
    return $prefix . '-' . $random_string;
}

function verify_license($license_key, $domain = null) {
    $licenses = json_decode(file_get_contents(LICENSES_JSON), true);
    
    foreach ($licenses as $license) {
        if ($license['key'] === $license_key && $license['status'] === 'active') {
            // Check if license is not expired
            if (strtotime($license['expires_at']) > time()) {
                // If domain is provided, check if it matches
                if ($domain && (!isset($license['domains']) || !in_array($domain, $license['domains']))) {
                    // Domain not registered with this license, add it
                    if (!isset($license['domains'])) {
                        $license['domains'] = [];
                    }
                    
                    // Check if we haven't exceeded the domain limit
                    $plan = get_plan_by_id($license['plan_id']);
                    $max_domains = $plan['id'] === 'enterprise' ? 999 : ($plan['id'] === 'professional' ? 3 : 1);
                    
                    if (count($license['domains']) < $max_domains) {
                        $license['domains'][] = $domain;
                        
                        // Update license in the database
                        update_license($license);
                        return ['status' => 'active', 'license' => $license];
                    } else {
                        return ['status' => 'domain_limit_exceeded'];
                    }
                }
                
                return ['status' => 'active', 'license' => $license];
            } else {
                return ['status' => 'expired'];
            }
        }
    }
    
    return ['status' => 'invalid'];
}

function get_plan_by_id($plan_id) {
    $plans = json_decode(file_get_contents(PLANS_JSON), true);
    return isset($plans[$plan_id]) ? $plans[$plan_id] : null;
}

function update_license($license_data) {
    $licenses = json_decode(file_get_contents(LICENSES_JSON), true);
    $updated = false;
    
    foreach ($licenses as $index => $license) {
        if ($license['key'] === $license_data['key']) {
            $licenses[$index] = $license_data;
            $updated = true;
            break;
        }
    }
    
    if ($updated) {
        file_put_contents(LICENSES_JSON, json_encode($licenses));
        return true;
    }
    
    return false;
}

function get_user_licenses($user_id) {
    $licenses = json_decode(file_get_contents(LICENSES_JSON), true);
    $user_licenses = [];
    
    foreach ($licenses as $license) {
        if ($license['user_id'] === $user_id) {
            $user_licenses[] = $license;
        }
    }
    
    return $user_licenses;
}

function get_all_licenses() {
    return json_decode(file_get_contents(LICENSES_JSON), true);
}

function get_user_transactions($user_id) {
    $transactions = json_decode(file_get_contents(TRANSACTIONS_JSON), true);
    $user_transactions = [];
    
    foreach ($transactions as $transaction) {
        if ($transaction['user_id'] === $user_id) {
            $user_transactions[] = $transaction;
        }
    }
    
    return $user_transactions;
}

function get_all_transactions() {
    return json_decode(file_get_contents(TRANSACTIONS_JSON), true);
}

function get_user_by_id($user_id) {
    $users = json_decode(file_get_contents(USERS_JSON), true);
    
    foreach ($users as $user) {
        if ($user['id'] === $user_id) {
            return $user;
        }
    }
    
    return null;
}

function add_transaction($transaction_data) {
    $transactions = json_decode(file_get_contents(TRANSACTIONS_JSON), true);
    $transaction_data['id'] = uniqid();
    $transaction_data['created_at'] = date('Y-m-d H:i:s');
    $transactions[] = $transaction_data;
    file_put_contents(TRANSACTIONS_JSON, json_encode($transactions));
    return $transaction_data['id'];
}

function add_license($license_data) {
    $licenses = json_decode(file_get_contents(LICENSES_JSON), true);
    $licenses[] = $license_data;
    file_put_contents(LICENSES_JSON, json_encode($licenses));
    return true;
}

// API endpoint for license verification (used by WordPress plugin)
if (isset($_GET['action']) && $_GET['action'] === 'verify_license') {
    header('Content-Type: application/json');
    
    if (isset($_POST['license_key']) && isset($_POST['domain'])) {
        $result = verify_license($_POST['license_key'], $_POST['domain']);
        echo json_encode($result);
    } else {
        echo json_encode(['status' => 'error', 'message' => 'Missing parameters']);
    }
    exit;
}
?>