<?php
require_once 'config.php';

// Redirect if already logged in
if (is_logged_in()) {
    redirect('dashboard.php');
}

$settings = get_settings();
$plans = json_decode(file_get_contents(PLANS_JSON), true);

// Get selected plan
$selected_plan_id = isset($_GET['plan']) ? $_GET['plan'] : 'basic';
$selected_plan = isset($plans[$selected_plan_id]) ? $plans[$selected_plan_id] : $plans['basic'];

$error = '';
$success = '';

// Process registration
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = trim($_POST['username']);
    $email = trim($_POST['email']);
    $password = $_POST['password'];
    $confirm_password = $_POST['confirm_password'];
    $plan_id = $_POST['plan_id'];
    
    // Validate inputs
    if (empty($username) || empty($email) || empty($password)) {
        $error = 'All fields are required.';
    } elseif ($password !== $confirm_password) {
        $error = 'Passwords do not match.';
    } elseif (strlen($password) < 6) {
        $error = 'Password must be at least 6 characters long.';
    } else {
        $users = json_decode(file_get_contents(USERS_JSON), true);
        
        // Check if username or email already exists
        foreach ($users as $user) {
            if ($user['username'] === $username) {
                $error = 'Username already exists.';
                break;
            }
            if ($user['email'] === $email) {
                $error = 'Email already exists.';
                break;
            }
        }
        
        if (empty($error)) {
            // Create new user
            $new_user = [
                'id' => uniqid(),
                'username' => $username,
                'email' => $email,
                'password' => password_hash($password, PASSWORD_DEFAULT),
                'role' => ROLE_USER,
                'created_at' => date('Y-m-d H:i:s'),
                'status' => 'active'
            ];
            
            $users[] = $new_user;
            file_put_contents(USERS_JSON, json_encode($users));
            
            // Log user in
            $_SESSION['user_id'] = $new_user['id'];
            $_SESSION['username'] = $new_user['username'];
            $_SESSION['user_role'] = $new_user['role'];
            $_SESSION['user_email'] = $new_user['email'];
            
            // Redirect to payment page
            redirect("payment.php?plan_id=$plan_id");
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Register - <?php echo $settings['site_name']; ?></title>
    <link rel="stylesheet" href="styles.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body>
    <header class="header">
        <div class="container">
            <div class="logo">
                <h1><i class="fas fa-key"></i> <?php echo $settings['site_name']; ?></h1>
            </div>
            <nav class="nav">
                <a href="index.php">Home</a>
                <a href="pricing.php">Pricing</a>
                <a href="register.php" class="active">Register</a>
                <a href="login.php">Login</a>
            </nav>
            <div class="mobile-menu-btn">
                <i class="fas fa-bars"></i>
            </div>
        </div>
    </header>

    <main class="main">
        <div class="container">
            <div class="form-container">
                <h2>Create Your Account</h2>
                <p class="subtitle">Register to purchase your Vine AI plugin license</p>
                
                <?php if ($error): ?>
                <div class="alert alert-error">
                    <?php echo $error; ?>
                </div>
                <?php endif; ?>
                
                <?php if ($success): ?>
                <div class="alert alert-success">
                    <?php echo $success; ?>
                </div>
                <?php endif; ?>
                
                <form method="POST" action="">
                    <div class="form-group">
                        <label for="username">Username</label>
                        <input type="text" id="username" name="username" value="<?php echo isset($_POST['username']) ? htmlspecialchars($_POST['username']) : ''; ?>" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="email">Email Address</label>
                        <input type="email" id="email" name="email" value="<?php echo isset($_POST['email']) ? htmlspecialchars($_POST['email']) : ''; ?>" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="password">Password</label>
                        <input type="password" id="password" name="password" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="confirm_password">Confirm Password</label>
                        <input type="password" id="confirm_password" name="confirm_password" required>
                    </div>
                    
                    <div class="form-group">
                        <label for="plan_id">Select Plan</label>
                        <select id="plan_id" name="plan_id" required>
                            <?php foreach ($plans as $plan_id => $plan): ?>
                                <?php if ($plan['is_active']): ?>
                                <option value="<?php echo $plan_id; ?>" <?php echo $plan_id === $selected_plan_id ? 'selected' : ''; ?>>
                                    <?php echo $plan['name']; ?> - ₦<?php echo number_format($plan['price']); ?>
                                </option>
                                <?php endif; ?>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <button type="submit" class="btn btn-primary" style="width: 100%;">Register & Continue to Payment</button>
                    
                    <div class="form-footer">
                        <p>Already have an account? <a href="login.php">Login here</a></p>
                    </div>
                </form>
            </div>

            <div class="plan-summary">
                <h3>Selected Plan: <?php echo $selected_plan['name']; ?></h3>
                <div class="price">₦<?php echo number_format($selected_plan['price']); ?></div>
                <ul>
                    <?php foreach ($selected_plan['features'] as $feature): ?>
                    <li><i class="fas fa-check"></i> <?php echo $feature; ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        </div>
    </main>

    <footer class="footer">
        <div class="container">
            <div class="footer-content">
                <div class="footer-section">
                    <h3><?php echo $settings['site_name']; ?></h3>
                    <p>Advanced AI writing tool for WordPress that generates human-like content without restrictions.</p>
                </div>
                <div class="footer-section">
                    <h4>Quick Links</h4>
                    <ul>
                        <li><a href="index.php">Home</a></li>
                        <li><a href="pricing.php">Pricing</a></li>
                        <li><a href="register.php">Register</a></li>
                    </ul>
                </div>
                <div class="footer-section">
                    <h4>Support</h4>
                    <ul>
                        <li><a href="#">Documentation</a></li>
                        <li><a href="#">Contact Us</a></li>
                        <li><a href="#">Privacy Policy</a></li>
                        <li><a href="#">Terms of Service</a></li>
                    </ul>
                </div>
                <div class="footer-section">
                    <h4>Contact Info</h4>
                    <p><i class="fas fa-envelope"></i> <?php echo $settings['support_email']; ?></p>
                    <p><i class="fas fa-map-marker-alt"></i> <?php echo $settings['company_address']; ?></p>
                </div>
            </div>
            <div class="footer-bottom">
                <p>&copy; <?php echo date('Y'); ?> <?php echo $settings['company_name']; ?>. All rights reserved.</p>
            </div>
        </div>
    </footer>

    <script src="script.js"></script>
</body>
</html>