// DOM Ready
document.addEventListener('DOMContentLoaded', function() {
    // Initialize all functionality
    initMobileMenu();
    initFAQToggle();
    initFormValidation();
    initCopyButtons();
    initPasswordToggle();
    initPlanSelectors();
    initPaymentHandlers();
    initSettingsTabs();
    initLicenseActions();
});

// Mobile Menu Toggle
function initMobileMenu() {
    const mobileMenuBtn = document.querySelector('.mobile-menu-btn');
    const nav = document.querySelector('.nav');
    
    if (mobileMenuBtn && nav) {
        mobileMenuBtn.addEventListener('click', function() {
            nav.classList.toggle('active');
            this.querySelector('i').classList.toggle('fa-bars');
            this.querySelector('i').classList.toggle('fa-times');
        });
    }
    
    // Close mobile menu when clicking outside
    document.addEventListener('click', function(e) {
        if (nav && nav.classList.contains('active') && 
            !nav.contains(e.target) && 
            !mobileMenuBtn.contains(e.target)) {
            nav.classList.remove('active');
            if (mobileMenuBtn) {
                mobileMenuBtn.querySelector('i').classList.add('fa-bars');
                mobileMenuBtn.querySelector('i').classList.remove('fa-times');
            }
        }
    });
}

// FAQ Toggle Functionality
function initFAQToggle() {
    const faqItems = document.querySelectorAll('.faq-item');
    
    faqItems.forEach(item => {
        const question = item.querySelector('h3');
        
        question.addEventListener('click', function() {
            // Toggle active class on clicked item
            item.classList.toggle('active');
            
            // Close other FAQ items
            faqItems.forEach(otherItem => {
                if (otherItem !== item) {
                    otherItem.classList.remove('active');
                }
            });
        });
    });
}

// Form Validation
function initFormValidation() {
    const forms = document.querySelectorAll('form');
    
    forms.forEach(form => {
        form.addEventListener('submit', function(e) {
            let isValid = true;
            const inputs = form.querySelectorAll('input[required], select[required], textarea[required]');
            
            inputs.forEach(input => {
                if (!input.value.trim()) {
                    isValid = false;
                    highlightError(input, 'This field is required');
                } else {
                    removeErrorHighlight(input);
                    
                    // Email validation
                    if (input.type === 'email' && !isValidEmail(input.value)) {
                        isValid = false;
                        highlightError(input, 'Please enter a valid email address');
                    }
                    
                    // Password validation
                    if (input.type === 'password' && input.value.length < 6) {
                        isValid = false;
                        highlightError(input, 'Password must be at least 6 characters');
                    }
                    
                    // Confirm password validation
                    if (input.id === 'confirm_password' || input.name === 'confirm_password') {
                        const password = form.querySelector('input[name="password"], input[name="new_password"]');
                        if (password && input.value !== password.value) {
                            isValid = false;
                            highlightError(input, 'Passwords do not match');
                        }
                    }
                }
            });
            
            if (!isValid) {
                e.preventDefault();
                // Scroll to first error
                const firstError = form.querySelector('.error-highlight');
                if (firstError) {
                    firstError.scrollIntoView({ behavior: 'smooth', block: 'center' });
                }
            }
        });
    });
    
    // Real-time validation
    document.querySelectorAll('input, select, textarea').forEach(input => {
        input.addEventListener('blur', function() {
            if (this.hasAttribute('required') && !this.value.trim()) {
                highlightError(this, 'This field is required');
            } else {
                removeErrorHighlight(this);
                
                // Email validation
                if (this.type === 'email' && this.value && !isValidEmail(this.value)) {
                    highlightError(this, 'Please enter a valid email address');
                }
                
                // Password validation
                if (this.type === 'password' && this.value && this.value.length < 6) {
                    highlightError(this, 'Password must be at least 6 characters');
                }
            }
        });
        
        // Clear error on input
        input.addEventListener('input', function() {
            removeErrorHighlight(this);
        });
    });
}

// Helper function for email validation
function isValidEmail(email) {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex.test(email);
}

// Helper function to highlight form errors
function highlightError(input, message) {
    removeErrorHighlight(input);
    
    input.classList.add('error-highlight');
    
    // Create error message element
    let errorElement = input.nextElementSibling;
    if (!errorElement || !errorElement.classList.contains('error-message')) {
        errorElement = document.createElement('div');
        errorElement.className = 'error-message';
        input.parentNode.insertBefore(errorElement, input.nextSibling);
    }
    
    errorElement.textContent = message;
    errorElement.style.display = 'block';
}

// Helper function to remove error highlighting
function removeErrorHighlight(input) {
    input.classList.remove('error-highlight');
    
    const errorElement = input.nextElementSibling;
    if (errorElement && errorElement.classList.contains('error-message')) {
        errorElement.style.display = 'none';
    }
}

// Copy to Clipboard functionality
function initCopyButtons() {
    document.addEventListener('click', function(e) {
        if (e.target.classList.contains('copy-btn') || e.target.closest('.copy-btn')) {
            e.preventDefault();
            
            const button = e.target.classList.contains('copy-btn') ? e.target : e.target.closest('.copy-btn');
            const licenseKey = button.getAttribute('data-license') || 
                              button.closest('.license-key')?.textContent.trim() ||
                              button.previousElementSibling?.textContent.trim();
            
            if (licenseKey) {
                copyToClipboard(licenseKey).then(() => {
                    showNotification('License key copied to clipboard!', 'success');
                    
                    // Visual feedback
                    const originalText = button.innerHTML;
                    button.innerHTML = '<i class="fas fa-check"></i> Copied!';
                    button.classList.add('copied');
                    
                    setTimeout(() => {
                        button.innerHTML = originalText;
                        button.classList.remove('copied');
                    }, 2000);
                }).catch(err => {
                    showNotification('Failed to copy license key', 'error');
                    console.error('Copy failed:', err);
                });
            }
        }
    });
    
    // Copy license key from success page
    const copyLicenseButtons = document.querySelectorAll('.copy-license');
    copyLicenseButtons.forEach(button => {
        button.addEventListener('click', function() {
            const licenseKey = this.getAttribute('data-license');
            copyToClipboard(licenseKey).then(() => {
                const originalText = this.textContent;
                this.textContent = 'Copied!';
                setTimeout(() => {
                    this.textContent = originalText;
                }, 2000);
            });
        });
    });
}

// Clipboard API function
function copyToClipboard(text) {
    if (navigator.clipboard && window.isSecureContext) {
        return navigator.clipboard.writeText(text);
    } else {
        // Fallback for older browsers
        const textArea = document.createElement('textarea');
        textArea.value = text;
        textArea.style.position = 'fixed';
        textArea.style.opacity = '0';
        document.body.appendChild(textArea);
        textArea.select();
        
        try {
            const successful = document.execCommand('copy');
            document.body.removeChild(textArea);
            return successful ? Promise.resolve() : Promise.reject('Copy command failed');
        } catch (err) {
            document.body.removeChild(textArea);
            return Promise.reject(err);
        }
    }
}

// Password visibility toggle
function initPasswordToggle() {
    const passwordToggles = document.querySelectorAll('.password-toggle');
    
    passwordToggles.forEach(toggle => {
        toggle.addEventListener('click', function() {
            const input = this.previousElementSibling;
            if (input.type === 'password') {
                input.type = 'text';
                this.innerHTML = '<i class="fas fa-eye-slash"></i>';
            } else {
                input.type = 'password';
                this.innerHTML = '<i class="fas fa-eye"></i>';
            }
        });
    });
    
    // Add toggle buttons to password fields if they don't exist
    const passwordFields = document.querySelectorAll('input[type="password"]');
    passwordFields.forEach(field => {
        if (!field.nextElementSibling?.classList.contains('password-toggle')) {
            const toggle = document.createElement('span');
            toggle.className = 'password-toggle';
            toggle.innerHTML = '<i class="fas fa-eye"></i>';
            toggle.style.cursor = 'pointer';
            toggle.style.marginLeft = '10px';
            field.parentNode.insertBefore(toggle, field.nextSibling);
            
            toggle.addEventListener('click', function() {
                if (field.type === 'password') {
                    field.type = 'text';
                    this.innerHTML = '<i class="fas fa-eye-slash"></i>';
                } else {
                    field.type = 'password';
                    this.innerHTML = '<i class="fas fa-eye"></i>';
                }
            });
        }
    });
}

// Plan selection functionality
function initPlanSelectors() {
    const planSelects = document.querySelectorAll('select[name="plan_id"]');
    
    planSelects.forEach(select => {
        select.addEventListener('change', function() {
            updatePlanSummary(this.value);
        });
        
        // Initialize summary on page load
        if (select.value) {
            updatePlanSummary(select.value);
        }
    });
    
    // Plan card selection
    const planCards = document.querySelectorAll('.pricing-card');
    planCards.forEach(card => {
        card.addEventListener('click', function(e) {
            if (!e.target.closest('a') && !e.target.closest('button')) {
                const planId = this.getAttribute('data-plan-id');
                const select = document.querySelector('select[name="plan_id"]');
                
                if (select && planId) {
                    select.value = planId;
                    updatePlanSummary(planId);
                    
                    // Scroll to form
                    const form = document.querySelector('form');
                    if (form) {
                        form.scrollIntoView({ behavior: 'smooth' });
                    }
                }
            }
        });
    });
}

// Update plan summary based on selected plan
function updatePlanSummary(planId) {
    // This would typically make an AJAX call to get plan details
    // For now, we'll simulate with available data
    const planSummary = document.querySelector('.plan-summary');
    if (!planSummary) return;
    
    // Simulate loading
    planSummary.innerHTML = '<p>Loading plan details...</p>';
    
    // Simulate API call with timeout
    setTimeout(() => {
        const plans = {
            'basic': {
                name: 'Basic Plan',
                price: '5,000',
                features: ['1 Website License', 'Basic Support', 'Regular Updates', '1 Year Validity']
            },
            'professional': {
                name: 'Professional Plan',
                price: '10,000',
                features: ['3 Website Licenses', 'Priority Support', 'All Updates', '1 Year Validity']
            },
            'enterprise': {
                name: 'Enterprise Plan',
                price: '20,000',
                features: ['Unlimited Licenses', '24/7 Support', 'All Updates + Beta Features', '1 Year Validity']
            }
        };
        
        const plan = plans[planId] || plans['basic'];
        
        planSummary.innerHTML = `
            <h3>Selected Plan: ${plan.name}</h3>
            <div class="price">₦${plan.price}</div>
            <ul>
                ${plan.features.map(feature => `<li><i class="fas fa-check"></i> ${feature}</li>`).join('')}
            </ul>
        `;
    }, 300);
}

// Payment handling with Flutterwave
function initPaymentHandlers() {
    const paymentForms = document.querySelectorAll('form[data-payment-form]');
    
    paymentForms.forEach(form => {
        form.addEventListener('submit', function(e) {
            e.preventDefault();
            
            // Validate form first
            if (!validateForm(this)) return;
            
            // Get form data
            const formData = new FormData(this);
            const planId = formData.get('plan_id');
            const amount = formData.get('amount');
            const email = formData.get('email');
            const name = formData.get('name');
            
            // Generate transaction reference
            const txRef = 'VINE-' + Date.now() + '-' + Math.random().toString(36).substr(2, 9);
            
            // Initialize Flutterwave payment
            makePayment(planId, amount, email, name, txRef);
        });
    });
}

// Flutterwave payment function
function makePayment(planId, amount, email, name, txRef) {
    // Show loading state
    showLoading('Processing payment...');
    
    // Flutterwave configuration
    const FLW_PUBLIC_KEY = 'FLWPUBK_TEST-xxxxxxxxxxxx-X'; // Replace with your actual public key
    
    // Create payment data
    const paymentData = {
        public_key: FLW_PUBLIC_KEY,
        tx_ref: txRef,
        amount: amount,
        currency: 'NGN',
        payment_options: 'card, banktransfer, ussd',
        redirect_url: `${window.location.origin}/success.php?tx_ref=${txRef}`,
        customer: {
            email: email,
            name: name
        },
        customizations: {
            title: 'Vine AI License Purchase',
            description: `Payment for ${planId} plan`,
            logo: `${window.location.origin}/assets/logo.png`
        }
    };
    
    // Make payment request
    flutterwavePayment(paymentData);
}

// Flutterwave payment handler
function flutterwavePayment(paymentData) {
    // This would integrate with Flutterwave's JavaScript SDK
    // For now, we'll simulate the payment flow
    
    console.log('Initiating payment with data:', paymentData);
    
    // Simulate payment processing
    setTimeout(() => {
        hideLoading();
        
        // For demo purposes, we'll redirect to success page
        // In a real implementation, this would be handled by Flutterwave's redirect
        window.location.href = `success.php?tx_ref=${paymentData.tx_ref}&status=successful`;
    }, 2000);
}

// Settings tabs functionality
function initSettingsTabs() {
    const tabLinks = document.querySelectorAll('.admin-tabs a');
    const tabContents = document.querySelectorAll('.dashboard-section');
    
    tabLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            
            // Remove active class from all tabs
            tabLinks.forEach(tab => tab.classList.remove('active'));
            
            // Add active class to clicked tab
            this.classList.add('active');
            
            // Get target tab
            const targetTab = this.getAttribute('href').substring(1);
            
            // Hide all tab contents
            tabContents.forEach(content => {
                content.style.display = 'none';
            });
            
            // Show target tab content
            const targetContent = document.getElementById(targetTab);
            if (targetContent) {
                targetContent.style.display = 'block';
            }
        });
    });
}

// License actions (add/remove domains)
function initLicenseActions() {
    // Domain form submission
    const domainForms = document.querySelectorAll('.add-domain-form form');
    
    domainForms.forEach(form => {
        form.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const domainInput = this.querySelector('input[name="domain"]');
            const domain = domainInput.value.trim();
            
            if (!isValidDomain(domain)) {
                highlightError(domainInput, 'Please enter a valid domain name');
                return;
            }
            
            // Submit form
            this.submit();
        });
    });
}

// Domain validation
function isValidDomain(domain) {
    // Simple domain validation
    const domainRegex = /^[a-zA-Z0-9][a-zA-Z0-9-]{1,61}[a-zA-Z0-9](?:\.[a-zA-Z]{2,})+$/;
    return domainRegex.test(domain);
}

// Notification system
function showNotification(message, type = 'info') {
    // Remove existing notifications
    const existingNotifications = document.querySelectorAll('.notification');
    existingNotifications.forEach(notification => {
        notification.remove();
    });
    
    // Create notification element
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <div class="notification-content">
            <i class="fas fa-${getNotificationIcon(type)}"></i>
            <span>${message}</span>
        </div>
        <button class="notification-close"><i class="fas fa-times"></i></button>
    `;
    
    // Add to page
    document.body.appendChild(notification);
    
    // Show notification
    setTimeout(() => {
        notification.classList.add('show');
    }, 10);
    
    // Auto-remove after 5 seconds
    const autoRemove = setTimeout(() => {
        removeNotification(notification);
    }, 5000);
    
    // Close button handler
    const closeBtn = notification.querySelector('.notification-close');
    closeBtn.addEventListener('click', () => {
        clearTimeout(autoRemove);
        removeNotification(notification);
    });
}

function getNotificationIcon(type) {
    const icons = {
        'success': 'check-circle',
        'error': 'exclamation-circle',
        'warning': 'exclamation-triangle',
        'info': 'info-circle'
    };
    
    return icons[type] || 'info-circle';
}

function removeNotification(notification) {
    notification.classList.remove('show');
    setTimeout(() => {
        if (notification.parentNode) {
            notification.parentNode.removeChild(notification);
        }
    }, 300);
}

// Loading overlay
function showLoading(message = 'Loading...') {
    // Remove existing loading overlay
    hideLoading();
    
    // Create loading overlay
    const loadingOverlay = document.createElement('div');
    loadingOverlay.className = 'loading-overlay';
    loadingOverlay.innerHTML = `
        <div class="loading-spinner">
            <div class="spinner"></div>
            <p>${message}</p>
        </div>
    `;
    
    // Add to page
    document.body.appendChild(loadingOverlay);
    
    // Prevent scrolling
    document.body.style.overflow = 'hidden';
}

function hideLoading() {
    const loadingOverlay = document.querySelector('.loading-overlay');
    if (loadingOverlay) {
        loadingOverlay.remove();
    }
    
    // Restore scrolling
    document.body.style.overflow = '';
}

// Form validation helper
function validateForm(form) {
    let isValid = true;
    const inputs = form.querySelectorAll('input[required], select[required], textarea[required]');
    
    inputs.forEach(input => {
        if (!input.value.trim()) {
            isValid = false;
            highlightError(input, 'This field is required');
        } else {
            removeErrorHighlight(input);
            
            // Email validation
            if (input.type === 'email' && !isValidEmail(input.value)) {
                isValid = false;
                highlightError(input, 'Please enter a valid email address');
            }
        }
    });
    
    return isValid;
}

// AJAX helper functions
function makeRequest(url, method = 'GET', data = null) {
    return new Promise((resolve, reject) => {
        const xhr = new XMLHttpRequest();
        xhr.open(method, url);
        xhr.setRequestHeader('Content-Type', 'application/json');
        xhr.setRequestHeader('X-Requested-With', 'XMLHttpRequest');
        
        xhr.onload = function() {
            if (xhr.status >= 200 && xhr.status < 300) {
                try {
                    const response = JSON.parse(xhr.responseText);
                    resolve(response);
                } catch (e) {
                    resolve(xhr.responseText);
                }
            } else {
                reject(new Error(`Request failed with status ${xhr.status}`));
            }
        };
        
        xhr.onerror = function() {
            reject(new Error('Network error'));
        };
        
        xhr.send(data ? JSON.stringify(data) : null);
    });
}

// Export functions for global access (if needed)
window.copyLicenseKey = function(licenseKey) {
    copyToClipboard(licenseKey).then(() => {
        showNotification('License key copied to clipboard!', 'success');
    });
};

// Add CSS for additional elements
const additionalStyles = `
    /* Error highlighting */
    .error-highlight {
        border-color: #ef4444 !important;
        box-shadow: 0 0 0 2px rgba(239, 68, 68, 0.1) !important;
    }
    
    .error-message {
        color: #ef4444;
        font-size: 0.875rem;
        margin-top: 0.25rem;
        display: none;
    }
    
    /* Notifications */
    .notification {
        position: fixed;
        top: 20px;
        right: 20px;
        background: white;
        border-left: 4px solid #4f46e5;
        border-radius: 6px;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        padding: 16px;
        display: flex;
        align-items: center;
        justify-content: space-between;
        min-width: 300px;
        max-width: 500px;
        transform: translateX(100%);
        transition: transform 0.3s ease;
        z-index: 10000;
    }
    
    .notification.show {
        transform: translateX(0);
    }
    
    .notification-success {
        border-left-color: #10b981;
    }
    
    .notification-error {
        border-left-color: #ef4444;
    }
    
    .notification-warning {
        border-left-color: #f59e0b;
    }
    
    .notification-content {
        display: flex;
        align-items: center;
        gap: 12px;
    }
    
    .notification-close {
        background: none;
        border: none;
        cursor: pointer;
        padding: 4px;
        border-radius: 4px;
        color: #6b7280;
    }
    
    .notification-close:hover {
        color: #374151;
        background: #f3f4f6;
    }
    
    /* Loading overlay */
    .loading-overlay {
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(0, 0, 0, 0.5);
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: 9999;
    }
    
    .loading-spinner {
        background: white;
        padding: 24px;
        border-radius: 8px;
        text-align: center;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
    }
    
    .spinner {
        width: 40px;
        height: 40px;
        border: 3px solid #f3f4f6;
        border-top: 3px solid #4f46e5;
        border-radius: 50%;
        animation: spin 1s linear infinite;
        margin: 0 auto 16px;
    }
    
    @keyframes spin {
        0% { transform: rotate(0deg); }
        100% { transform: rotate(360deg); }
    }
    
    /* Copy button states */
    .copy-btn.copied {
        background-color: #10b981 !important;
    }
    
    /* Form inline group */
    .form-group-inline {
        display: flex;
        gap: 12px;
    }
    
    .form-group-inline input {
        flex: 1;
    }
    
    /* Responsive improvements */
    @media (max-width: 768px) {
        .form-group-inline {
            flex-direction: column;
        }
        
        .notification {
            left: 20px;
            right: 20px;
            min-width: auto;
        }
    }
`;

// Inject additional styles
const styleSheet = document.createElement('style');
styleSheet.textContent = additionalStyles;
document.head.appendChild(styleSheet);